package plugins.adufour.activemeshes.energy;

import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import javax.vecmath.Point3d;

import plugins.adufour.activemeshes.mesh.Mesh;
import plugins.adufour.activemeshes.mesh.Vertex;

public class CouplingTerm extends EnergyTerm
{
    public CouplingTerm(ExecutorService service)
    {
        super(service, null); // no weight for coupling
    }
    
    private final ArrayList<Mesh> meshes = new ArrayList<Mesh>();
    
    public synchronized void registerMesh(Mesh mesh)
    {
        meshes.add(mesh);
        
        mesh.addModel(new Model(mesh)
        {
            ArrayList<Future<Integer>> couplingTasks = new ArrayList<Future<Integer>>();
            
            @Override
            public void computeForces()
            {
                Point3d center = mesh.getMassCenter();
                double radius = mesh.getMaxDistanceToCenter();
                
                couplingTasks.clear();
                couplingTasks.ensureCapacity(meshes.size());
                
                for (final Mesh target : meshes)
                {
                    if (mesh != target && center.distance(target.getMassCenter()) < radius + target.getMaxDistanceToCenter())
                    {
                        couplingTasks.add(multiThreadService.submit(new Callable<Integer>()
                        {
                            public Integer call()
                            {
                                return computeFeedback(target);
                            }
                        }));
                        // single thread mode (comment everything else)
                        // computeFeedback(target);
                    }
                }
                
                try
                {
                    for (Future<Integer> task : couplingTasks)
                        task.get();
                }
                catch (InterruptedException e)
                {
                    e.printStackTrace();
                }
                catch (ExecutionException e)
                {
                    e.printStackTrace();
                }
            }
            
            /**
             * Computes the feedback forces yielded by the penetration of the current contour into
             * the target contour
             * 
             * @param target
             *            the contour that is being penetrated
             * @return the number of vertices tested
             */
            private int computeFeedback(Mesh target)
            {
                double distanceSq, penetration = 0;
                
                int tests = 0;
                
                double targetRadiusSq = target.getMaxDistanceToCenter();
                targetRadiusSq *= targetRadiusSq;
                
                Point3d targetCenter = target.getMassCenter();
                
                // Vector3d couplingForce = new Vector3d();
                
                for (Vertex v : mesh.vertices)
                {
                    if (v == null) continue;
                    
                    distanceSq = v.position.distanceSquared(targetCenter);
                    
                    if (distanceSq < targetRadiusSq)
                    {
                        tests++;
                        
                        if ((penetration = target.isInside(v)) > 0)
                        {
                            // couplingForce.scale(penetration * -1.0, v.normal);
                            // v.forces.add(couplingForce);
                            
                            // feedback is not strong enough
                            // => replace all forces by the feedback
                            
                            v.forces.scale(penetration * -2.0, v.normal);
                        }
                    }
                }
                
                return tests;
            }
            
            @Override
            public void removeMesh(Mesh mesh)
            {
                unregisterMesh(mesh);
            }
        });
    }
    
    @Override
    public void unregisterMesh(Mesh mesh)
    {
        meshes.remove(mesh);
    }
    
    @Override
    public void unregisterMeshes()
    {
        meshes.clear();
    }
}