package plugins.tprovoost.Microscopy.MicroManager.gui;

import icy.common.MenuCallback;
import icy.file.FileUtil;
import icy.gui.component.ExtTabbedPanel;
import icy.gui.component.ExternalizablePanel;
import icy.gui.component.NumberTextField;
import icy.gui.component.NumberTextField.ValueChangeListener;
import icy.gui.dialog.ConfirmDialog;
import icy.gui.dialog.MessageDialog;
import icy.gui.dialog.SaveDialog;
import icy.gui.frame.IcyFrame;
import icy.gui.frame.IcyFrameAdapter;
import icy.gui.frame.IcyFrameEvent;
import icy.gui.frame.progress.FailedAnnounceFrame;
import icy.gui.util.ComponentUtil;
import icy.main.Icy;
import icy.network.NetworkUtil;
import icy.plugin.PluginDescriptor;
import icy.plugin.PluginLauncher;
import icy.plugin.PluginLoader;
import icy.preferences.PluginPreferences;
import icy.preferences.XMLPreferences;
import icy.resource.ResourceUtil;
import icy.resource.icon.IcyIcon;
import icy.system.IcyExceptionHandler;
import icy.system.SystemUtil;
import icy.system.thread.ThreadUtil;
import icy.util.StringUtil;

import java.awt.AlphaComposite;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.AbstractCellEditor;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.WindowConstants;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import mmcorej.CMMCore;
import mmcorej.DeviceType;
import mmcorej.MMCoreJ;
import mmcorej.MMEventCallback;
import mmcorej.StrVector;

import org.micromanager.CalibrationListDlg;
import org.micromanager.ConfigGroupPad;
import org.micromanager.GroupEditor;
import org.micromanager.MMStudioMainFrame;
import org.micromanager.MMVersion;
import org.micromanager.PresetEditor;
import org.micromanager.PropertyEditor;
import org.micromanager.api.PositionList;
import org.micromanager.conf2.ConfiguratorDlg2;
import org.micromanager.conf2.MMConfigFileException;
import org.micromanager.conf2.MicroscopeModel;
import org.micromanager.utils.ColorEditor;
import org.micromanager.utils.ColorRenderer;
import org.micromanager.utils.MMScriptException;
import org.micromanager.utils.PropertyItem;
import org.micromanager.utils.ReportingUtils;

import plugins.tprovoost.Microscopy.MicroManager.MicroManager;
import plugins.tprovoost.Microscopy.MicroManager.MicroscopeOverlayPreferences;
import plugins.tprovoost.Microscopy.MicroManager.tools.FrameUtils;
import plugins.tprovoost.Microscopy.MicroManager.tools.StageMover;
import plugins.tprovoost.Microscopy.MicroManagerForIcy.MicromanagerPlugin;
import plugins.tprovoost.Microscopy.MicroManagerForIcy.MicroscopePlugin;

/**
 * Main frame for Micro Manager Plugin (Singleton pattern)
 * 
 * @author Irsath Nguyen
 */
public class MMMainFrame extends IcyFrame
{
    /** Is the MMMainFrame instanced ? */
    boolean instanced = false;
    /** Is the MMMainFrame currently instancing ? */
    boolean instancing = false;

    public MMStudioMainFrame MMFRAME;

    // CAMERA SETTINGS PART
    /** Camera Name */
    String _camera_label;
    /** Text Field containing exposure. */
    NumberTextField _txtExposure;
    /** ComboBox containing current value of binning. */
    JComboBox _combo_binning;
    /** ComboBox containing current shutter. */
    JComboBox _combo_shutters;

    // CONFIG PART
    /** Configuration panel. */
    ConfigGroupPad _groupPad;

    // ACQUISITION PART
    /** Container for the right part of interface */
    ExternalizablePanel _panelAcquisitions;

    // COLOR SETTINGS
    MicroscopeOverlayPreferences painterPreferences;
    MMEventCallback main_callback;

    // PLUGINS
    JToolBar pluginToolbar;
    /** List of all plugins added to the main plugin via addPlugin method. */
    List<MicroscopePlugin> _list_plugin = new ArrayList<MicroscopePlugin>();

    // PROGRESS BARS
    JTable painterTable;
    /** List of all progresses bars added to the main plugin. */
    CopyOnWriteArrayList<RunningProgress> _list_progress = new CopyOnWriteArrayList<RunningProgress>();
    /** Used to show a bar when a continuous acquisition is currently running. */
    RunningProgress continuousProgress;

    // EXCLUSIVE ACCESS
    final ReentrantLock rlock;
    boolean modifyingExposure;

    // ------------
    // PREFERENCES
    // --------------
    public XMLPreferences pluginPreferences;

    /**
     * Singleton pattern : private constructor Use instead.
     * 
     * @throws Exception
     */
    public MMMainFrame() throws Exception
    {
        super("Micro-Manager For Icy", true, true, false, true);

        instancing = true;
        instanced = false;
        try
        {
            modifyingExposure = false;
            pluginPreferences = PluginPreferences.getPreferences().node(MicromanagerPlugin.class.getName());
            rlock = new ReentrantLock(true);

            ThreadUtil.invokeNow(new Callable<Object>()
            {
                @Override
                public Object call() throws Exception
                {
                    continuousProgress = new RunningProgress();

                    try
                    {
                        MMFRAME = new MMStudioMainFrame(false);
                        // force some initialization stuff on micro manager
                        MMFRAME.dispatchEvent(new WindowEvent(MMFRAME, WindowEvent.WINDOW_OPENED));

                        // ok
                        return null;
                    }
                    catch (Exception e)
                    {
                        // shutdown all activities
                        MMFRAME.stopAllActivity();
                        MMFRAME.getCore().unloadAllDevices();
                        MMFRAME.dispose();

                        // forward exception
                        throw new Exception("Could not retrieve Micro Manager core !", e);
                    }
                }
            });

            if (MMFRAME.getCore() == null)
                throw new Exception("Could not retrieve Micro Manager core !");

            MMFRAME.getCore().enableDebugLog(false);
            MMFRAME.getCore().enableStderrLog(false);

            intializeCoreCallback();

            _camera_label = MMCoreJ.getG_Keyword_CameraName();
            if (_camera_label == null)
                _camera_label = "";
            try
            {
                MMFRAME.setPositionList(new PositionList());
            }
            catch (MMScriptException e1)
            {
                MMFRAME.logError(e1);
            }

            instanced = true;
        }
        finally
        {
            instancing = false;
        }
    }

    /**
     * Should be used internally only.
     */
    public void init()
    {
        ThreadUtil.invokeNow(new Runnable()
        {
            @Override
            public void run()
            {
                initializeSystemMenu();
                initializeGUI();

                setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
                addFrameListener(new IcyFrameAdapter()
                {
                    @Override
                    public void icyFrameClosing(IcyFrameEvent e)
                    {
                        customClose();
                    }
                });

                // To have a minimum size of Icy's default frame size
                setMinimumSizeExternal(getSize());
                setMinimumSizeInternal(getSize());
                setPreferredSizeExternal(getSize());
                setPreferredSizeInternal(getSize());
                pack();

                addToDesktopPane();
            }
        });
    }

    /**
     * Create all needed graphics components and listeners and show the main frame.
     */
    void initializeGUI()
    {
        // SETUP
        _groupPad = new ConfigGroupPad();
        _groupPad.setFont(new Font("", 0, 10));
        _groupPad.setCore(MMFRAME.getCore());
        _groupPad.setParentGUI(MMFRAME);

        final JPanel configButtonPanel = new JPanel(new GridLayout());
        configButtonPanel.add(new JLabel("Group: "));
        configButtonPanel.add(FrameUtils.createUIButton("", "sq_plus.png", new Runnable()
        {
            @Override
            public void run()
            {
                new GroupEditor("", "", MMFRAME, MMFRAME.getCore(), true);
            }
        }));
        configButtonPanel.add(FrameUtils.createUIButton("", "sq_minus.png", new Runnable()
        {
            @Override
            public void run()
            {
                String groupName = _groupPad.getGroup();
                if (groupName.length() > 0)
                {
                    if (ConfirmDialog.confirm("Remove the " + groupName + " group?",
                            "Are you sure you want to remove group " + groupName + " and all associated presets?"))
                        try
                        {
                            MMFRAME.getCore().deleteConfigGroup(groupName);
                        }
                        catch (Exception e)
                        {
                            ReportingUtils.logError(e);
                        }
                }
                else
                {
                    MessageDialog
                            .showDialog("If you want to remove a group, select it on the Configurations panel first.");
                }
            }
        }));
        configButtonPanel.add(FrameUtils.createUIButton("", "doc_lines.png", new Runnable()
        {
            @Override
            public void run()
            {
                String groupName = _groupPad.getGroup();
                if (groupName.length() == 0)
                    MessageDialog.showDialog("To edit a group, please select it first, then press the edit button.");
                else
                    new GroupEditor(groupName, _groupPad.getPreset(), MMFRAME, MMFRAME.getCore(), false);
            }
        }));

        configButtonPanel.add(new JLabel("   Preset: "));
        configButtonPanel.add(FrameUtils.createUIButton("", "sq_plus.png", new Runnable()
        {
            @Override
            public void run()
            {
                String groupName = _groupPad.getGroup();
                if (groupName.length() == 0)
                    MessageDialog
                            .showDialog("To add a preset to a group, please select the group first, then press the edit button.");
                else
                    new PresetEditor(groupName, "", MMFRAME, MMFRAME.getCore(), true);
            }
        }));
        configButtonPanel.add(FrameUtils.createUIButton("", "sq_minus.png", new Runnable()
        {
            @Override
            public void run()
            {
                String groupName = _groupPad.getGroup();
                String presetName = _groupPad.getPreset();
                if (groupName.length() > 0)
                    if (MMFRAME.getCore().getAvailableConfigs(groupName).size() == 1L)
                    {
                        int result = JOptionPane.showConfirmDialog(
                                configButtonPanel,
                                (new StringBuilder()).append("\"").append(presetName)
                                        .append("\" is the last preset for the \"").append(groupName)
                                        .append("\" group.\nDelete both preset and group?").toString(),
                                "Remove last preset in group", 0, 1);
                        if (result == 0)
                            try
                            {
                                MMFRAME.getCore().deleteConfig(groupName, presetName);
                                MMFRAME.getCore().deleteConfigGroup(groupName);
                            }
                            catch (Exception e)
                            {
                                ReportingUtils.logError(e);
                            }
                    }
                    else
                    {
                        int result = JOptionPane.showConfirmDialog(
                                configButtonPanel,
                                (new StringBuilder()).append("Are you sure you want to remove preset ")
                                        .append(presetName).append(" from the ").append(groupName).append(" group?")
                                        .toString(), "Remove preset", 0, 1);
                        if (result == 0)
                            try
                            {
                                MMFRAME.getCore().deleteConfig(groupName, presetName);
                            }
                            catch (Exception e)
                            {
                                ReportingUtils.logError(e);
                            }
                    }
            }
        }));
        configButtonPanel.add(FrameUtils.createUIButton("", "doc_lines.png", new Runnable()
        {
            @Override
            public void run()
            {
                String presetName = _groupPad.getPreset();
                String groupName = _groupPad.getGroup();
                if (groupName.length() == 0 || presetName.length() == 0)
                    MessageDialog
                            .showDialog("To edit a preset, please select the preset first, then press the edit button.");
                else
                    new PresetEditor(groupName, presetName, MMFRAME, MMFRAME.getCore(), false);
            }
        }));

        configButtonPanel.add(new JLabel("  Refresh: "));
        configButtonPanel.add(FrameUtils.createUIButton("", "rot_unclock.png", new Runnable()
        {
            @Override
            public void run()
            {
                updateGUI();
            }
        }));

        // LEFT PART OF INTERFACE
        JPanel _panelConfig = new JPanel();
        _panelConfig.setLayout(new BoxLayout(_panelConfig, BoxLayout.Y_AXIS));
        _panelConfig.add(_groupPad, BorderLayout.CENTER);
        _panelConfig.add(configButtonPanel, BorderLayout.SOUTH);

        // MIDDLE PART OF INTERFACE
        JPanel _panel_cameraSettings = new JPanel();
        _panel_cameraSettings.setLayout(new GridLayout(5, 2));

        _txtExposure = new NumberTextField();
        _txtExposure.addValueListener(new ValueChangeListener()
        {
            @Override
            public void valueChanged(double newValue, boolean validate)
            {
                if (modifyingExposure)
                    return;

                if (validate)
                {
                    final double exposure = _txtExposure.getNumericValue();

                    if (exposure < 1)
                        _txtExposure.setNumericValue(1);
                    else
                    {
                        try
                        {
                            MicroManager.setExposure(exposure);
                        }
                        catch (Exception e)
                        {
                            IcyExceptionHandler.showErrorMessage(e, true);
                        }
                    }
                }
            }
        });

        _panel_cameraSettings.add(new JLabel("Exposure [ms]: "));
        _panel_cameraSettings.add(_txtExposure);

        _combo_binning = new JComboBox();
        _combo_binning.setMaximumRowCount(4);
        _combo_binning.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                ThreadUtil.bgRun(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        try
                        {
                            if (_camera_label.length() > 0)
                            {
                                Object item = _combo_binning.getSelectedItem();
                                if (item == null)
                                    return;
                                CMMCore core = MMFRAME.getCore();
                                synchronized (core)
                                {
                                    if (core.getProperty(_camera_label, MMCoreJ.getG_Keyword_Binning()).equals(
                                            item.toString()))
                                        return;
                                    boolean bWasRunning = core.isSequenceRunning();
                                    if (bWasRunning)
                                        core.stopSequenceAcquisition();
                                    core.waitForDevice(_camera_label);
                                    core.setProperty(_camera_label, MMCoreJ.getG_Keyword_Binning(), item.toString());
                                    if (bWasRunning && !core.isSequenceRunning())
                                        core.startContinuousSequenceAcquisition(0.0D);
                                }
                            }

                        }
                        catch (Exception e)
                        {
                            MMFRAME.logError(e);
                        }
                    }
                });
            }
        });

        _panel_cameraSettings.add(new JLabel("Binning: "));
        _panel_cameraSettings.add(_combo_binning);

        _combo_shutters = new JComboBox();
        _combo_shutters.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                try
                {
                    if (_combo_shutters.getSelectedItem() != null
                            && !MMFRAME.getCore().getShutterDevice()
                                    .equals(_combo_shutters.getSelectedItem().toString()))
                        MMFRAME.getCore().setShutterDevice((String) _combo_shutters.getSelectedItem());
                }
                catch (Exception e)
                {
                    MMFRAME.logError(e);
                }
            }
        });
        _panel_cameraSettings.add(new JLabel("Shutter : "));
        _panel_cameraSettings.add(_combo_shutters);

        // Acquisition
        _panelAcquisitions = new ExternalizablePanel("Running Acquisitions");
        _panelAcquisitions.setLayout(new BoxLayout(_panelAcquisitions, BoxLayout.Y_AXIS));
        _panelAcquisitions.add(new JLabel("No acquisition running."));
        // Color settings
        painterPreferences = MicroscopeOverlayPreferences.getInstance();
        painterPreferences.setPreferences(pluginPreferences.node("paintersPreferences"));
        painterPreferences.loadColors();

        final HashMap<String, Color> allColors = painterPreferences.getColors();
        String[] allKeys = allColors.keySet().toArray(new String[0]);
        final String[] columnNames = {"Painter", "Color", "Transparency"};
        final Object[][] data = new Object[allKeys.length][3];

        for (int i = 0; i < allKeys.length; ++i)
        {
            final int actualRow = i;
            final String actualKey = allKeys[i];
            data[i][0] = actualKey;
            data[i][1] = allColors.get(actualKey);
            final JSlider slider = new JSlider(0, 255, allColors.get(actualKey).getAlpha())
            {
                private static final long serialVersionUID = -7571997343734398732L;

                @Override
                public void paint(Graphics g)
                {
                    Graphics2D g2 = g == null ? null : (Graphics2D) g.create();
                    // Computation to have an alpha between 30% and 100%
                    float alpha = (((float) getValue()) / getMaximum() * .7f) + .3f;
                    g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha));
                    super.paint(g2);
                }
            };
            slider.setFocusable(true);
            slider.setBackground((Color) data[i][1]);
            slider.setOpaque(false);
            slider.addChangeListener(new ChangeListener()
            {
                @Override
                public void stateChanged(ChangeEvent changeevent)
                {
                    painterTable.setValueAt(slider, actualRow, 2);
                    slider.setBackground(allColors.get(actualKey));
                }
            });
            data[i][2] = slider;
        }
        final AbstractTableModel tableModel = new AbstractTableModel()
        {
            private static final long serialVersionUID = -4702254916669824054L;

            @Override
            public int getColumnCount()
            {
                return columnNames.length;
            }

            @Override
            public int getRowCount()
            {
                return data.length;
            }

            @Override
            public String getColumnName(int col)
            {
                return columnNames[col];
            }

            @Override
            public Class<?> getColumnClass(int col)
            {
                return getValueAt(0, col).getClass();
            }

            @Override
            public boolean isCellEditable(int row, int col)
            {
                return col >= 1;
            }

            @Override
            public Object getValueAt(int row, int col)
            {
                return data[row][col];
            }

            @Override
            public void setValueAt(Object value, int row, int col)
            {
                data[row][col] = value;
                fireTableCellUpdated(row, col);
            }
        };

        painterTable = new JTable(tableModel);
        painterTable.getTableHeader().setReorderingAllowed(false);
        painterTable.getModel().addTableModelListener(new TableModelListener()
        {
            @Override
            public void tableChanged(TableModelEvent tablemodelevent)
            {
                if (tablemodelevent.getType() == TableModelEvent.UPDATE)
                {
                    int row = tablemodelevent.getFirstRow();
                    int col = tablemodelevent.getColumn();
                    String columnName = tableModel.getColumnName(col);
                    String painterName = (String) tableModel.getValueAt(row, 0);
                    if (columnName.contains("Color"))
                    {
                        // New color value
                        int alpha = painterPreferences.getColor(painterName).getAlpha();
                        Color color = (Color) tableModel.getValueAt(row, 1);
                        Color newColor = new Color(color.getRed(), color.getGreen(), color.getBlue(), alpha);
                        painterPreferences.setColor(painterName, newColor);
                        ((JSlider) tableModel.getValueAt(row, 2)).setBackground(newColor);

                    }
                    else if (columnName.contains("Transparency"))
                    {
                        // New alpha value
                        Color c = painterPreferences.getColor(painterName);
                        int alphaValue = ((JSlider) tableModel.getValueAt(row, 2)).getValue();
                        painterPreferences.setColor(painterName, new Color(c.getRed(), c.getGreen(), c.getBlue(),
                                alphaValue));
                    }
                }
            }
        });
        painterTable.setFillsViewportHeight(true);
        painterTable.setDefaultRenderer(Color.class, new ColorRenderer(true));
        painterTable.setDefaultEditor(Color.class, new ColorEditor(tableModel, 0));

        class SliderEditor extends AbstractCellEditor implements TableCellEditor
        {
            private static final long serialVersionUID = 4574031691985928116L;
            private JSlider slider;

            @Override
            public JSlider getCellEditorValue()
            {
                return slider;
            }

            @Override
            public Component getTableCellEditorComponent(JTable jtable, Object obj, boolean flag, int i, int j)
            {
                slider = (JSlider) obj;
                return slider;
            }
        }
        painterTable.setDefaultRenderer(JSlider.class, new TableCellRenderer()
        {
            @Override
            public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                    boolean hasFocus, int row, int column)
            {
                return (JSlider) value;
            }
        });
        painterTable.setDefaultEditor(JSlider.class, new SliderEditor());

        ExternalizablePanel _panelCameraSettingsContainer = new ExternalizablePanel("Camera Settings");
        _panelCameraSettingsContainer.setLayout(new GridLayout(0, 1));
        _panelCameraSettingsContainer.add(_panel_cameraSettings);

        ExternalizablePanel panelPainterSettingsContainer = new ExternalizablePanel("Overlay Settings");
        panelPainterSettingsContainer.setLayout(new GridLayout(0, 1));
        panelPainterSettingsContainer.add(new JScrollPane(painterTable));

        final ExtTabbedPanel _tabbedPanel = new ExtTabbedPanel();
        _tabbedPanel.setTabLayoutPolicy(JTabbedPane.SCROLL_TAB_LAYOUT);
        _tabbedPanel.add("Configuration", _panelConfig);
        _tabbedPanel.add(_panelCameraSettingsContainer.getFrame().getTitle(), _panelCameraSettingsContainer);
        _tabbedPanel.add(_panelAcquisitions.getFrame().getTitle(), _panelAcquisitions);
        _tabbedPanel.add(panelPainterSettingsContainer.getFrame().getTitle(), panelPainterSettingsContainer);

        final List<PluginDescriptor> microscopePlugins = PluginLoader.getPlugins(MicroscopePlugin.class);
        pluginToolbar = new JToolBar(SwingConstants.HORIZONTAL);
        pluginToolbar.setRollover(true);
        pluginToolbar.setFloatable(false);
        pluginToolbar.setToolTipText("Micro Manager plugins for Icy");

        for (final PluginDescriptor plugin : microscopePlugins)
        {
            pluginToolbar.add(FrameUtils.createPluginButton(plugin, new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent e)
                {
                    // create the microscope plugin
                    final MicroscopePlugin microscopePlugin = (MicroscopePlugin) PluginLauncher.start(plugin);

                    if (microscopePlugin != null)
                    {
                        // register and start
                        addPlugin(microscopePlugin);
                        microscopePlugin.start();
                    }
                }
            }));
        }

        if (microscopePlugins.size() == 0)
        {
            MessageDialog.showDialog("Information",
                    "You don't any Micro manager plugins installed, use the search bar to install some",
                    MessageDialog.INFORMATION_MESSAGE);
            // search for micro manager plugin
            Icy.getMainInterface().getSearchEngine().search("micro-manager");
        }

        JPanel pluginPanel = new JPanel();
        pluginPanel.setLayout(new BorderLayout());
        pluginPanel.setBorder(new TitledBorder("Plugins"));
        pluginPanel.add(pluginToolbar, BorderLayout.CENTER);

        // Main Frame
        setLayout(new BorderLayout());
        add(_tabbedPanel, BorderLayout.CENTER);
        add(pluginPanel, BorderLayout.SOUTH);
        setVisible(true);
        updateGUI();
    }

    void intializeCoreCallback()
    {
        main_callback = new MMEventCallback()
        {
            @Override
            public void onPropertiesChanged()
            {
                updateGUI();
            }

            @Override
            public void onConfigGroupChanged(String groupName, String newConfig)
            {
                _groupPad.refreshGroup(groupName, newConfig);
                updateGUI();
            }

            @Override
            public void onExposureChanged(String deviceName, double exposure)
            {
                modifyingExposure = true;
                try
                {
                    _txtExposure.setText(StringUtil.toString(exposure));
                }
                finally
                {
                    modifyingExposure = false;
                }

                for (MicroscopePlugin e : _list_plugin)
                    e.onExposureChanged(exposure);
            }

            @Override
            public void onPropertyChanged(String deviceName, String propName, String propValue)
            {
                if (propName.equals(MMCoreJ.getG_Keyword_Binning()))
                    _combo_binning.setSelectedItem(Integer.valueOf(propValue));

                for (MicroscopePlugin e : _list_plugin)
                    e.onCorePropertyChanged(deviceName, propName, propValue);
            }

            @Override
            public void onStagePositionChanged(String deviceName, double pos)
            {
                StageMover.onStagePositionChanged(deviceName, pos);
            }

            @Override
            public void onXYStagePositionChanged(String deviceName, double xPos, double yPos)
            {
                StageMover.onXYStagePositionChanged(deviceName, xPos, yPos);
            };

            @Override
            public void onSystemConfigurationLoaded()
            {
                for (MicroscopePlugin e : _list_plugin)
                    e.onSystemConfigurationLoaded();
            }
        };
        MMFRAME.getCore().registerCallback(main_callback);
    }

    void initializeSystemMenu()
    {
        final int SHORTCUTKEY_MASK = SystemUtil.getMenuCtrlMask();
        setSystemMenuCallback(new MenuCallback()
        {
            @Override
            public JMenu getMenu()
            {
                JMenu toReturn = getDefaultSystemMenu();
                JMenuItem hconfig = new JMenuItem("Configuration Wizard");
                hconfig.setIcon(new IcyIcon(ResourceUtil.ICON_COG));
                hconfig.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent e)
                    {
                        if (!_list_plugin.isEmpty()
                                && !ConfirmDialog.confirm("Are you sure ?",
                                        "<html>Loading the Configuration Wizard will unload all the devices"
                                                + " and pause all running acquisitions.</br>"
                                                + " Are you sure you want to continue ?</html>"))
                            return;
                        try
                        {
                            MMFRAME.getCore().unloadAllDevices();
                        }
                        catch (Exception e1)
                        {
                            MMFRAME.logError(e1);
                        }
                        ConfiguratorDlg2 configurator = new ConfiguratorDlg2(MMFRAME.getCore(), getConfigFile());
                        configurator.setVisible(true);
                        setConfigFile(configurator.getFileName());
                        loadConfig(getConfigFile());
                        MMFRAME.refreshGUI();
                        _groupPad.refreshStructure(false);
                        repaint();
                    }
                });

                JMenuItem menuPxSizeConfigItem = new JMenuItem("Pixel Size Config");
                menuPxSizeConfigItem.setIcon(new IcyIcon(ResourceUtil.ICON_PROPERTIES));
                menuPxSizeConfigItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G, InputEvent.SHIFT_DOWN_MASK
                        | SHORTCUTKEY_MASK));
                menuPxSizeConfigItem.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent e)
                    {
                        CalibrationListDlg dlg = new CalibrationListDlg(MMFRAME.getCore());
                        dlg.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
                        dlg.setParentGUI(MMFRAME);
                        final IcyFrame pixelSizeConfig = FrameUtils.addMMWindowToDesktopPane(dlg);
                        pixelSizeConfig.setResizable(true);
                    }
                });

                JMenuItem loadConfigItem = new JMenuItem("Load Configuration");
                loadConfigItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, InputEvent.SHIFT_DOWN_MASK
                        | SHORTCUTKEY_MASK));
                loadConfigItem.setIcon(new IcyIcon(ResourceUtil.ICON_OPEN));
                loadConfigItem.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent e)
                    {
                        // TODO see in micro-manager what they do before they load another config
                        // because our loadNewConfig() can crash the jvm sometime
                        // TODO show _progressFrame while loading
                        loadNewConfig();
                    }
                });

                JMenuItem saveConfigItem = new JMenuItem("Save Configuration");
                saveConfigItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.SHIFT_DOWN_MASK
                        | SHORTCUTKEY_MASK));
                saveConfigItem.setIcon(new IcyIcon(ResourceUtil.ICON_SAVE));
                saveConfigItem.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent e)
                    {
                        saveConfig();
                    }
                });

                JMenuItem aboutItem = new JMenuItem("About");
                aboutItem.setIcon(new IcyIcon(ResourceUtil.ICON_INFO));
                aboutItem.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent e)
                    {
                        final JDialog dialog = new JDialog(Icy.getMainInterface().getMainFrame(), "About");
                        JPanel panel_container = new JPanel();
                        panel_container.setBorder(BorderFactory.createEmptyBorder(10, 20, 10, 20));
                        JPanel center = new JPanel(new BorderLayout());
                        final JLabel value = new JLabel(
                                "<html><body>"
                                        + "<h2>About</h2><p>Micro-Manager for Icy is being developed by Stephane Dallongeville, Irsath Nguyen and Thomas Provoost."
                                        + "<br/>Copyright 2015, Institut Pasteur</p><br/>"
                                        + "<p>This plugin is based on Micro-Manager "
                                        + MMVersion.VERSION_STRING
                                        + " which is developed under the following license:<br/>"
                                        + "<i>This software is distributed free of charge in the hope that it will be<br/>"
                                        + "useful, but WITHOUT ANY WARRANTY; without even the implied<br/>"
                                        + "warranty of merchantability or fitness for a particular purpose. In no<br/>"
                                        + "event shall the copyright owner or contributors be liable for any direct,<br/>"
                                        + "indirect, incidental spacial, examplary, or consequential damages.<br/>"
                                        + "Copyright University of California San Francisco, 2007, 2008, 2009,<br/>"
                                        + "2010. All rights reserved.</i>" + "</p>" + "</body></html>");
                        JLabel link = new JLabel(
                                "<html><a href=\"\">For more information, please follow this link.</a></html>");
                        link.addMouseListener(new MouseAdapter()
                        {
                            @Override
                            public void mousePressed(MouseEvent mouseevent)
                            {
                                NetworkUtil.openBrowser("http://www.micro-manager.org/");
                            }
                        });
                        value.setSize(new Dimension(50, 18));
                        value.setAlignmentX(SwingConstants.HORIZONTAL);
                        value.setBorder(BorderFactory.createEmptyBorder(0, 0, 20, 0));

                        center.add(value, BorderLayout.CENTER);
                        center.add(link, BorderLayout.SOUTH);

                        JPanel panel_south = new JPanel();
                        panel_south.setLayout(new BoxLayout(panel_south, BoxLayout.X_AXIS));
                        JButton btn = new JButton("OK");
                        btn.addActionListener(new ActionListener()
                        {
                            @Override
                            public void actionPerformed(ActionEvent actionevent)
                            {
                                dialog.dispose();
                            }
                        });
                        panel_south.add(Box.createHorizontalGlue());
                        panel_south.add(btn);
                        panel_south.add(Box.createHorizontalGlue());

                        dialog.setLayout(new BorderLayout());
                        panel_container.setLayout(new BorderLayout());
                        panel_container.add(center, BorderLayout.CENTER);
                        panel_container.add(panel_south, BorderLayout.SOUTH);
                        dialog.add(panel_container, BorderLayout.CENTER);
                        dialog.setResizable(false);
                        dialog.setVisible(true);
                        dialog.pack();
                        ComponentUtil.center(dialog);
                    }
                });

                JMenuItem propertyBrowserItem = new JMenuItem("Property Browser");
                propertyBrowserItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_COMMA, SHORTCUTKEY_MASK));
                propertyBrowserItem.setIcon(new IcyIcon(ResourceUtil.ICON_DATABASE));
                propertyBrowserItem.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent e)
                    {
                        PropertyEditor editor = new PropertyEditor();
                        editor.setGui(MMFRAME);
                        editor.setCore(MMFRAME.getCore());
                        editor.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
                        final IcyFrame propertyBrowser = FrameUtils.addMMWindowToDesktopPane(editor);
                        propertyBrowser.setResizable(true);
                    }
                });

                JMenuItem loadPresetConfigItem = new JMenuItem("Load Core properties from XML");
                loadPresetConfigItem.setIcon(new IcyIcon(ResourceUtil.ICON_DOC_IMPORT));
                loadPresetConfigItem.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent e)
                    {
                        loadXMLFile(pluginPreferences.node("CoreProperties"));
                    }
                });

                JMenuItem savePresetConfigItem = new JMenuItem("Save Core properties to XML");
                savePresetConfigItem.setIcon(new IcyIcon(ResourceUtil.ICON_DOC_EXPORT));
                savePresetConfigItem.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent e)
                    {
                        saveToXML(pluginPreferences.node("CoreProperties"));
                    }
                });

                int idx = 0;
                toReturn.insert(hconfig, idx++);
                toReturn.insert(loadConfigItem, idx++);
                toReturn.insert(saveConfigItem, idx++);
                toReturn.insertSeparator(idx++);
                toReturn.insert(loadPresetConfigItem, idx++);
                toReturn.insert(savePresetConfigItem, idx++);
                toReturn.insertSeparator(idx++);
                toReturn.insert(propertyBrowserItem, idx++);
                toReturn.insert(menuPxSizeConfigItem, idx++);
                toReturn.insertSeparator(idx++);
                toReturn.insert(aboutItem, idx++);

                return toReturn;
            }
        });
    }

    /**
     * @return Returns if this class is being instanced.
     */
    public boolean isInstancing()
    {
        return instancing;
    }

    public boolean isInstanced()
    {
        return instanced;
    }

    public void lock()
    {
        rlock.lock();
    }

    public boolean lock(long wait) throws InterruptedException
    {
        return rlock.tryLock(wait, TimeUnit.MILLISECONDS);
    }

    public void unlock()
    {
        rlock.unlock();
    }

    void loadNewConfig()
    {
        ThreadUtil.invokeNow(new Runnable()
        {
            @Override
            public void run()
            {
                LoadFrame f = new LoadFrame();

                if (1 != f.showDialog())
                {
                    setConfigFile(f.getConfigFilePath());
                    loadConfig(getConfigFile());
                    MMFRAME.refreshGUI();
                    _groupPad.refreshStructure(false);
                    repaint();
                }
            }
        });
    }

    /**
     * Blocking method
     * 
     * @param filePath
     */
    void loadConfig(final String filePath)
    {
        // show loading message
        final LoadingFrame loadingFrame = new LoadingFrame(
                "Please wait while loading Micro-Manager configuration, it may take a while...");

        loadingFrame.show();
        try
        {
            MMFRAME.getCore().waitForSystem();
            MMFRAME.getCore().loadSystemConfiguration(filePath);
        }
        catch (Exception e)
        {
            MessageDialog
                    .showDialog(
                            "Error while initializing the microscope: please check if all devices are correctly turned on "
                                    + "and recognized by the computer and quit any program using those devices. Pleas check also that your configuration file is correct.",
                            MessageDialog.ERROR_MESSAGE);
        }
        finally
        {
            loadingFrame.close();
        }
    }

    /**
     * Save the configuration presets. From Micro-Manager.
     */
    protected void saveConfig()
    {
        try
        {
            MicroscopeModel model = new MicroscopeModel();
            model.loadFromFile(getConfigFile());
            model.createSetupConfigsFromHardware(MMFRAME.getCore());
            model.createResolutionsFromHardware(MMFRAME.getCore());
            String path = SaveDialog.chooseFile("Save the configuration file", FileUtil.getApplicationDirectory(),
                    "myConfig", ".cfg");
            if (path != null)
                model.saveToFile(path);
        }
        catch (MMConfigFileException e)
        {
            new FailedAnnounceFrame("Unable to save configuration file");
        }
    }

    public void customClose()
    {
        if (getInternalFrame().getDefaultCloseOperation() == WindowConstants.DO_NOTHING_ON_CLOSE)
        {
            if (!Icy.isExiting() && !_list_plugin.isEmpty())
            {
                if (!ConfirmDialog
                        .confirm("Some Micro-Manager plugins are still running. Closing this frame may close other plugins using Micro-Manager. Continue ?"))
                    return;
            }

            setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
            MicroManager.shutdown();
        }
    }

    @Override
    public void onClosed()
    {
        if (painterPreferences != null)
            painterPreferences.saveColors();

        if (_list_plugin != null)
        {
            // shutdown all plugins
            for (int i = 0; i < _list_plugin.size(); i++)
                _list_plugin.get(i).shutdown();

            // shutdown all activities
            MMFRAME.stopAllActivity();
            try
            {
                MMFRAME.getCore().unloadAllDevices();
            }
            catch (Exception e)
            {
                ReportingUtils.logError("Unable to unload all devices from core");
            }

            MMFRAME.dispose();

            super.onClosed();
        }
    }

    void updateGUI()
    {
        ThreadUtil.invokeNow(new Runnable()
        {
            @Override
            public void run()
            {
                // Retrieving info from core
                _camera_label = MMFRAME.getCore().getCameraDevice();
                if (_camera_label.length() > 0)
                {
                    if (_combo_binning.getItemCount() > 0)
                        _combo_binning.removeAllItems();

                    try
                    {
                        StrVector availableBinnings = MMFRAME.getCore().getAllowedPropertyValues(_camera_label,
                                MMCoreJ.getG_Keyword_Binning());
                        for (String v : availableBinnings)
                            _combo_binning.addItem(v);
                        _combo_binning.setMaximumRowCount((int) availableBinnings.size());
                        _combo_binning.setEditable(availableBinnings.isEmpty());
                        _combo_binning.setSelectedIndex(StringUtil.parseInt(
                                MMFRAME.getCore().getProperty(_camera_label, MMCoreJ.getG_Keyword_Binning()), 1));
                    }
                    catch (Exception e1)
                    {
                    }
                }
                try
                {
                    StrVector _shutters = MMFRAME.getCore().getLoadedDevicesOfType(DeviceType.ShutterDevice);
                    _combo_shutters.removeAllItems();
                    for (String v : _shutters)
                        _combo_shutters.addItem(v);
                    _combo_shutters.setSelectedItem(MMFRAME.getCore().getShutterDevice());
                }
                catch (Exception e)
                {
                    MMFRAME.logError(e);
                }

                try
                {
                    _txtExposure.setText(String.valueOf(MMFRAME.getCore().getExposure()));
                }
                catch (Exception e2)
                {
                }
            }
        });
    }

    void updateProgressFrame()
    {
        ThreadUtil.invokeNow(new Runnable()
        {
            @Override
            public void run()
            {
                if (_list_progress.size() == 0)
                    _groupPad.refreshStructure(false);
                _panelAcquisitions.removeAll();
                if (_list_progress.size() == 0)
                {
                    _panelAcquisitions.add(new JLabel("No acquisition running."));
                }
                else
                {
                    for (RunningProgress prog : _list_progress)
                    {
                        _panelAcquisitions.add(Box.createRigidArea(new Dimension(1, 10)));
                        if (prog.isValueDisplayed())
                            prog.setString(prog.getRenderedName() + " : " + prog.getProgress() + "%");
                        else
                            prog.setString(prog.getRenderedName());
                        prog.setStringPainted(true);
                        prog.setMinimum(0);
                        prog.setBounds(50, 50, 100, 30);

                        if (prog.getPlugin() != null)
                        {
                            prog.setMaximum(100);
                            prog.setValue(prog.getProgress());
                        }
                        else
                        {
                            prog.setIndeterminate(true);
                            prog.setMaximum(1000);
                        }
                        _panelAcquisitions.add(prog);
                    }
                }
                _panelAcquisitions.setSize(300, 300);
                _panelAcquisitions.validate();
            }
        });
    }

    /**
     * Adds the plugin to the plugin list of MMMainFrame.
     * 
     * @param plugin
     *        : plugin to be added.
     * @see #removePlugin(MicroscopePlugin)
     */
    public void addPlugin(MicroscopePlugin plugin)
    {
        if (!instanced)
            return;

        _list_plugin.add(plugin);
    }

    /**
     * Removes the plugin from the plugin list of MMMainFrame. If no more plugin
     * using the acquisition is running, acquisition is stopped.
     * 
     * @param plugin
     *        : plugin to be removed.
     * @see #addPlugin(MicroscopePlugin)
     */
    public void removePlugin(MicroscopePlugin plugin)
    {
        if (!instanced)
            return;
        if (plugin == null)
            return;

        _list_plugin.remove(plugin);
    }

    /**
     * Notify the GUI that he need to draw the indeterminate progress bar representing a live
     * acquisition running.
     */
    public void notifyLiveRunning()
    {
        if (!instanced)
            return;

        if (!_list_progress.contains(continuousProgress))
        {
            _list_progress.add(continuousProgress);
            updateProgressFrame();
        }
    }

    /**
     * Notify the GUI that he no more need to draw the indeterminate progress bar representing a
     * live acquisition running.
     */
    public void notifyLiveStopped()
    {
        if (!instanced)
            return;

        if (_list_progress.contains(continuousProgress))
        {
            _list_progress.remove(continuousProgress);
            updateProgressFrame();
        }
    }

    public void showProgressBar(final MicroscopePlugin MicroManagerPlugin, final boolean valueDisplayed)
    {
        int i = 0;
        String name = MicroManagerPlugin.getDescriptor().getName();
        // we want to add an index to the name if another plugin
        // with this name is already running an acquisition.
        for (RunningProgress runp : _list_progress)
            if (runp.getRenderedName().contains(name))
                ++i;

        if (i != 0)
            name = name + " (" + (i + 1) + ")";
        final String namefinal = name;
        ThreadUtil.invokeNow(new Runnable()
        {
            @Override
            public void run()
            {
                _list_progress.add(new RunningProgress(MicroManagerPlugin, namefinal, valueDisplayed));
                updateProgressFrame();
            }
        });
    }

    public void removeProgressBar(MicroscopePlugin plugin)
    {
        if (plugin != null)
        {
            int idx = indexOfPlugin(plugin);
            if (idx >= 0)
            {
                _list_progress.remove(idx);
                updateProgressFrame();
            }
        }
    }

    public void notifyProgress(MicroscopePlugin plugin, int progress)
    {
        int idx = indexOfPlugin(plugin);
        if (idx >= 0)
        {
            _list_progress.get(indexOfPlugin(plugin)).setProgress(progress);
            updateProgressFrame();
        }
    }

    /**
     * Returns the index of the plugin.
     * 
     * @param p
     *        : plugin to find
     * @return Returns the index if existing, -1 if not existing or null.
     */
    private int indexOfPlugin(MicroscopePlugin plugin)
    {
        if (plugin != null)
            for (int i = 0; i < _list_progress.size(); ++i)
                if (_list_progress.get(i).getPlugin() == plugin)
                    return i;
        return -1;
    }

    /**
     * Save all the properties into an XML file.
     * 
     * @param root
     *        : file and node where data is saved.
     */
    void saveToXML(XMLPreferences root)
    {
        StrVector devices = MMFRAME.getCore().getLoadedDevices();
        for (int i = 0; i < devices.size(); i++)
        {
            XMLPreferences prefs = root.node(devices.get(i));
            StrVector properties;
            try
            {
                properties = MMFRAME.getCore().getDevicePropertyNames(devices.get(i));
            }
            catch (Exception e)
            {
                continue;
            }
            for (int j = 0; j < properties.size(); j++)
            {
                PropertyItem item = new PropertyItem();
                item.readFromCore(MMFRAME.getCore(), devices.get(i), properties.get(j), false);
                prefs.put(properties.get(j), item.value);
            }
        }
    }

    /**
     * Load all the properties into a file.
     * 
     * @param root
     *        : file and node where data is saved.
     */
    void loadXMLFile(XMLPreferences root)
    {
        for (XMLPreferences device : root.getChildren())
        {
            for (String propName : device.keys())
            {
                String value = device.get(propName, "");

                if (!StringUtil.isEmpty(value))
                {
                    try
                    {
                        MMFRAME.getCore().setProperty(device.name(), propName, value);
                        MMFRAME.getCore().waitForSystem();
                    }
                    catch (Exception e)
                    {
                        continue;
                    }
                }
            }
        }
    }

    public String getConfigFile()
    {
        return System.getProperty("org.micromanager.default.config.file", "MMConfig_demo.cfg");
    }

    void setConfigFile(String fileName)
    {
        if (fileName != null)
            System.setProperty("org.micromanager.default.config.file", fileName);
    }
}