package plugins.tprovoost.Microscopy.MicroManager.tools;

import icy.file.FileUtil;
import icy.gui.dialog.MessageDialog;
import icy.gui.frame.progress.FailedAnnounceFrame;
import icy.image.IcyBufferedImage;
import icy.main.Icy;
import icy.network.NetworkUtil;
import icy.plugin.PluginLoader;
import icy.plugin.PluginLoader.PluginClassLoader;
import icy.preferences.PluginsPreferences;
import icy.preferences.XMLPreferences;
import icy.system.SystemUtil;
import icy.system.thread.ThreadUtil;
import icy.type.DataType;
import icy.type.collection.CollectionUtil;
import icy.type.collection.array.ArrayUtil;

import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import mmcorej.TaggedImage;

import org.json.JSONException;
import org.micromanager.acquisition.TaggedImageQueue;
import org.micromanager.utils.MDUtils;

import plugins.tprovoost.Microscopy.MicroManager.gui.LoadingFrame;
import plugins.tprovoost.Microscopy.MicroManagerForIcy.MicromanagerPlugin;

/**
 * Utility class for loading MicromanagerPlugin native libraries such as MMCoreJ_wrap and device
 * libraries.
 * 
 * @author Irsath Nguyen
 */
public class MMUtils
{
    private final static String MM_PATH_ID = "libray_path";

    static XMLPreferences prefs = PluginsPreferences.root(MicromanagerPlugin.class);
    static String uManagerRep = null;
    public static File demoConfigFile = null;
    private static boolean loaded = false;

    public static boolean isSystemLibrairiesLoaded()
    {
        return loaded;
    }

    public static boolean fixSystemLibrairies()
    {
        if (loaded)
            return loaded;

        uManagerRep = prefs.get(MM_PATH_ID, "");

        final File uManagerLibraryRep = new File(uManagerRep);

        // empty or not existing ?
        if (uManagerRep.isEmpty() || !uManagerLibraryRep.exists() || !uManagerLibraryRep.isDirectory())
        {
            ThreadUtil.invokeNow(new Runnable()
            {
                @Override
                public void run()
                {
                    final int option = JOptionPane.showOptionDialog(Icy.getMainInterface().getMainFrame(),
                            "Have you already installed Micro-Manager ?", "Micro-Manager For Icy",
                            JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE, null, new String[] {
                                    "Select Micro-Manger directory", "Download Micro-Manager"},
                            "Download Micro-Manager");

                    if (option != JOptionPane.YES_OPTION)
                    {
                        NetworkUtil
                                .openBrowser("http://www.micro-manager.org/wiki/Download%20Micro-Manager_Latest%20Release");
                        MessageDialog.showDialog("Restart this plugin after Micro-Manager installation complete.",
                                MessageDialog.INFORMATION_MESSAGE);
                        uManagerRep = null;
                    }
                    else
                    {
                        final JFileChooser fc = new JFileChooser();
                        fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
                        if (fc.showOpenDialog(Icy.getMainInterface().getMainFrame()) == JFileChooser.APPROVE_OPTION)
                            uManagerRep = fc.getSelectedFile().getAbsolutePath();
                        else
                            uManagerRep = null;
                    }
                }
            });
        }

        // operation canceled or directory set
        if (uManagerRep == null)
            return false;

        if (!uManagerRep.endsWith(FileUtil.separator))
            uManagerRep += FileUtil.separator;

        final LoadingFrame loadingFrame = new LoadingFrame(
                "Please wait while loading Micro-Manager, Icy interface may not respond...");

        // show loading message
        loadingFrame.show();
        try
        {
            loaded = loadJarFrom(new File(uManagerRep + FileUtil.separator + "plugins" + FileUtil.separator
                    + "Micro-Manager" + FileUtil.separator));
        }
        finally
        {
            loadingFrame.close();
        }

        if (!loaded)
        {
            new FailedAnnounceFrame(
                    "Error while loading libraries, have you choosen the correct directory ? Please try again.");
        }
        else
        {
            loadDllFrom(new File(uManagerRep));
            File[] cfg = new File(uManagerRep).listFiles(new FilenameFilter()
            {
                @Override
                public boolean accept(File file, String s)
                {
                    return s.equalsIgnoreCase("MMConfig_demo.cfg");
                }
            });
            if (cfg != null && cfg.length > 0)
                demoConfigFile = cfg[0];

            prefs.put(MM_PATH_ID, uManagerRep);
            System.setProperty("mmcorej.library.path", uManagerRep);
        }

        return loaded;
    }

    public static void resetLibrayPath()
    {
        prefs.put(MM_PATH_ID, "");
        System.setProperty("mmcorej.library.path", "");
    }

    /**
     * Convert a list of {@link TaggedImage} (Micro Manager) to {@link IcyBufferedImage} (Icy) where
     * each {@link TaggedImage} represents one channel of the output image.
     * 
     * @throws JSONException
     */
    public static IcyBufferedImage convertToIcyImage(List<TaggedImage> images) throws JSONException
    {
        final List<TaggedImage> goodImages = new ArrayList<TaggedImage>(images.size());
        int w, h, bpp;

        w = -1;
        h = -1;
        bpp = -1;
        for (TaggedImage image : images)
        {
            if ((image != null) && !TaggedImageQueue.isPoison(image))
            {
                if (w == -1)
                    w = MDUtils.getWidth(image.tags);
                else if (w != MDUtils.getWidth(image.tags))
                    continue;
                if (h == -1)
                    h = MDUtils.getHeight(image.tags);
                else if (h != MDUtils.getHeight(image.tags))
                    continue;
                if (bpp == -1)
                    bpp = MDUtils.getBitDepth(image.tags);
                else if (bpp != MDUtils.getBitDepth(image.tags))
                    continue;

                goodImages.add(image);
            }
        }

        if ((goodImages.size() > 0) && (w > 0) && (h > 0) && (bpp > 0))
        {
            // get the datatype
            final DataType type = ArrayUtil.getDataType(goodImages.get(0).pix, false);
            // create image
            final IcyBufferedImage result = new IcyBufferedImage(w, h, goodImages.size(), type);

            result.beginUpdate();
            try
            {
                // copy data
                for (int c = 0; c < result.getSizeC(); c++)
                    result.setDataXY(c, goodImages.get(c).pix);
            }
            finally
            {
                result.endUpdate();
            }

            return result;
        }

        return null;
    }

    /**
     * Convert a {@link TaggedImage} (Micro Manager) to {@link IcyBufferedImage} (Icy).
     * 
     * @throws JSONException
     */
    public static IcyBufferedImage convertToIcyImage(TaggedImage img) throws JSONException
    {
        return convertToIcyImage(CollectionUtil.createArrayList(img));
    }

    private static boolean loadDllFrom(File microManagerDirectory)
    {
        ArrayList<File> dll = new ArrayList<File>(Arrays.asList(FileUtil.getFiles(microManagerDirectory,
                new FileFilter()
                {
                    @Override
                    public boolean accept(File pathname)
                    {
                        String extension = FileUtil.getFileExtension(pathname.getAbsolutePath(), false);
                        return (extension.equalsIgnoreCase("dll") || extension.equalsIgnoreCase("jnilib"))
                                && !pathname.getName().contains("mmgr_dal_")
                                && !pathname.getName().contains("MMCoreJ_wrap");
                    }
                }, true, false, true)));

        if (dll.isEmpty())
            return false;

        int numberOfTry = 0;
        while (!dll.isEmpty())
        {
            numberOfTry++;
            for (File f : dll)
                try
                {
                    SystemUtil.loadLibrary(f.getAbsolutePath());
                }
                catch (UnsatisfiedLinkError e)
                {
                }
            if (numberOfTry > 9)
                break;
        }
        return true;
    }

    private static boolean loadJarFrom(File microManagerDirectoryPath)
    {
        File[] files = FileUtil.getFiles(microManagerDirectoryPath, new FileFilter()
        {
            @Override
            public boolean accept(File pathname)
            {
                return FileUtil.getFileExtension(pathname.getAbsolutePath(), false).equalsIgnoreCase("jar");
            }
        }, true, false, true);

        if (files == null || files.length == 0)
            return false;
        for (File f : files)
            loadJar(f);
        return true;
    }

    private static void loadJar(File jarPath)
    {
        ClassLoader cl = PluginLoader.getLoader();
        if (cl instanceof PluginClassLoader)
        {
            ((PluginClassLoader) cl).add(jarPath.getAbsolutePath());
        }
    }
}