package plugins.adufour.roi;

import icy.canvas.Canvas3D;
import icy.main.Icy;
import icy.roi.ROI2D;
import icy.roi.ROI2DArea;
import icy.sequence.Sequence;
import icy.type.point.Point3D;
import icy.util.ShapeUtil.ShapeOperation;
import icy.vtk.VtkUtil;

import java.awt.Point;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Arrays;

import vtk.vtkActor;
import vtk.vtkPolyDataMapper;
import vtk.vtkProp;
import vtk.vtkStructuredGrid;
import vtk.vtkStructuredGridOutlineFilter;

public class ROI3DArea extends ROI3DStack<ROI2DArea>
{
    public ROI3DArea()
    {
        super(ROI2DArea.class.getName());
        
        setName("3D area");
    }
    
    public ROI3DArea(int x, int y, int z)
    {
        this();
        addPoint(x, y, z);
    }
    
    public ROI3DArea(Point2D pt)
    {
        this((int) pt.getX(), (int) pt.getY(), 0);
    }
    
    /**
     * Adds the specified point to this ROI
     * 
     * @param x
     * @param y
     * @param z
     */
    public void addPoint(int x, int y, int z)
    {
        getSlice(z, true).addPoint(x, y);
    }
    
    @Override
    public void addROI2D(int z, ROI2D roi, boolean merge)
    {
        if (roi == null) throw new IllegalArgumentException("Cannot add an empty slice in a 3D ROI");
        
        ROI2DArea currentSlice = getSlice(z, false);
        
        ROI2DArea newSlice = null;
        
        if (merge && currentSlice != null)
        {
            // since currentSlice is a ROI2DArea, the merge operation will also return a ROI2DArea
            newSlice = (ROI2DArea) ROI2D.merge(new ROI2D[] { currentSlice, roi }, ShapeOperation.OR);
        }
        else
        {
            newSlice = new ROI2DArea(roi.getBooleanMask(true));
        }
        
        setSlice(z, newSlice);
    }
    
    @Override
    protected ROIPainter createPainter()
    {
        return new ROI3DAreaPainter();
    }
    
    public Point3D[] getEdgePoints()
    {
        ArrayList<Point3D> points = new ArrayList<Point3D>();
        
        for (ROI2DArea slice : this)
        {
            for (Point pt : slice.getBooleanMask(true).getEdgePoints())
                points.add(new Point3D.Integer(pt.x, pt.y, slice.getZ()));
        }
        
        return points.toArray(new Point3D[points.size()]);
    }
    
    public Point3D[] getPoints()
    {
        ArrayList<Point3D> points = new ArrayList<Point3D>();
        
        for (ROI2DArea slice : this)
        {
            for (Point pt : slice.getBooleanMask().getPoints())
                points.add(new Point3D.Integer(pt.x, pt.y, slice.getZ()));
        }
        
        return points.toArray(new Point3D[points.size()]);
    }
    
    public class ROI3DAreaPainter extends ROI3DStackPainter
    {
        // VTK 3D objects, we use Object to prevent UnsatisfiedLinkError
        final Object grid;
        final Object gridOutline;
        // final Object polyData;
        final Object polyMapper;
        final Object actor;
        // 3D internal
        boolean      needRebuild;
        double       scaling[];
        
        public ROI3DAreaPainter()
        {
            // avoid to use vtk objects when vtk library is missing.
            if (Icy.isVtkLibraryLoaded())
            {
                // init 3D painters stuff
                // polyData = new vtkPolyData();
                grid = new vtkStructuredGrid();
                gridOutline = new vtkStructuredGridOutlineFilter();
                
                ((vtkStructuredGridOutlineFilter) gridOutline).SetInput((vtkStructuredGrid) grid);
                
                polyMapper = new vtkPolyDataMapper();
                // ((vtkPolyDataMapper) polyMapper).SetInput((vtkPolyData) polyData);
                ((vtkPolyDataMapper) polyMapper).SetInput(((vtkStructuredGridOutlineFilter) gridOutline).GetOutput());
                
                actor = new vtkActor();
                ((vtkActor) actor).SetMapper((vtkPolyDataMapper) polyMapper);
            }
            else
            {
                // polyData = null;
                grid = null;
                gridOutline = null;
                polyMapper = null;
                actor = null;
            }
            
            scaling = new double[3];
            Arrays.fill(scaling, 1d);
            
            needRebuild = true;
        }
        
        /**
         * update 3D painter for 3D canvas (called only when vtk is loaded).
         */
        protected void rebuild3DPainter(Canvas3D canvas)
        {
            final Sequence seq = canvas.getSequence();
            
            // nothing to update
            if (seq == null) return;
            
            int width = seq.getSizeX();
            int height = seq.getSizeY();
            
            ArrayList<double[]> verticesArray = new ArrayList<double[]>(0);
            
            for (ROI2DArea area : ROI3DArea.this)
            {
                int k = area.getZ();
                
                verticesArray.ensureCapacity(verticesArray.size() + width * height);
                
                boolean[] mask = area.getBooleanMask(0, 0, width, height, true);
                int offset = 0;
                for (int j = 0; j < height; j++)
                    for (int i = 0; i < width; i++, offset++)
                    {
                        if (mask[offset]) verticesArray.add(new double[] { i, j, k });
                    }
            }
            
            double[][] points = new double[verticesArray.size()][3];
            verticesArray.toArray(points);
            
            ((vtkStructuredGrid) grid).SetDimensions(width, height, getSizeZ());
            ((vtkStructuredGrid) grid).SetPoints(VtkUtil.getPoints(points));
            
            ((vtkStructuredGridOutlineFilter) gridOutline).Update();
            ((vtkPolyDataMapper) polyMapper).Update();
        }
        
        @Override
        public vtkProp[] getProps()
        {
            return new vtkActor[] { (vtkActor) actor };
        }
    }
}
