package plugins.tprovoost.Microscopy.MicroManagerForIcy;

import icy.plugin.abstract_.Plugin;
import plugins.tprovoost.Microscopy.MicroManager.MicroManager;

/**
 * This is the class to inherit in order to create a Microscope Plugin.<br/>
 * Any {@link MicroscopePlugin} has to implement the start() method. That
 * way, you will have access to the main interface and the core, and your plugin
 * will automatically wait for the Micro-Manager For Icy to be running.<br/>
 * <b>At the end of life of your plugin you should manually call shutdown()</b>
 * <p>
 * <b>Example: </b></br> start() { <br/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(core.getAPIVersionInfo());<br/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;shutdown();<br/>
 * }
 * </p>
 * 
 * @see #start()
 * @author Irsath Nguyen
 */
public abstract class MicroscopePlugin extends Plugin
{
    public MicroscopePlugin()
    {
        super();

        MicromanagerPlugin.init();

        if (!MicroManager.isInitialized())
            throw new RuntimeException("Cannot initialize Micro-Manager !");
    }

    /**
     * Method to define in sub-classes in replacement of usual run() method of
     * Icy plugin's
     * It is called only <b>AFTER</b> Micro-Manager For Icy is launched.
     */
    public abstract void start();

    /**
     * Called when main gui is closed, also mean that the core is destroyed. <br/>
     * Override this method to be able to compute some data saving before exit in this method, like
     * frame position saving. <br/>
     * <b>You should dispose all your thread or resources using MicroManager in this method,
     * and not
     * use them anymore.<br/>
     * You should not call MicroManager's method in this method and after. Also you don't need
     * to
     * remove your listener's (Stage, Live and Acquisition)</b><br/>
     * <b>Don't delete the call to super.shutdown();</b>
     */
    public void shutdown()
    {
        // need to test as sometime client frame is closed after micro manager frame
        if (MicroManager.getInstance() != null)
            MicroManager.getInstance().removePlugin(this);
    }

    /**
     * Notify GUI to show a progress bar for the plugin in the running acquisition tab of
     * micro-manager for icy GUI. <br/>
     * You can use it to show a progression of the plugin work, such as the progression of your
     * acquisition, or the progression of the extraction of channels. <br/>
     * <b>This method is not required to be called , it's only used for plugins that will show their
     * progress in the GUI as progress bars.</b>
     * 
     * @param valueDisplayed
     *        : displays or not the actual progress text. It is highly
     *        advised to not forget to use notifyProgress if the boolean is
     *        true. Otherwise you progress bar will be at 0%
     * @see #removeProgressBar()
     * @see #notifyProgress(int)
     */
    public void showProgressBar(boolean valueDisplayed)
    {
        MicroManager.getInstance().showProgressBar(this, valueDisplayed);
    }

    /**
     * Removes the plugin progress bar, call it when the plugin have done is work.
     * This will remove (if exist) is progress bar created by {@link #showProgressBar(boolean)} in
     * the GUI.
     * 
     * @see #showProgressBar
     * @see #notifyProgress
     */
    public void removeProgressBar()
    {
        MicroManager.getInstance().removeProgressBar(this);
    }

    /**
     * Updates the progression of the progress bar of this plugin (if exist) , <br />
     * created by {@link #showProgressBar(boolean)} in the GUI.
     * 
     * @param progress
     *        : percentage of progress of the plugin work, should be <b>( 0 <= progress <= 100 )</b>
     * @see #showProgressBar(boolean)
     * @see #removeProgressBar()
     */
    public void notifyProgress(int progress)
    {
        MicroManager.getInstance().notifyProgress(this, progress);
    }

    /**
     * Override this method in order to listen to exposure changes.
     * 
     * @param newExposure
     */
    public void onExposureChanged(double newExposure)
    {

    }

    /**
     * Override this method in order to listen to core properties changes.
     * 
     * @param deviceName
     * @param propName
     *        Defined as string's in MMCoreJ.getG_Keyword_[....]
     * @param propValue
     */
    public void onCorePropertyChanged(String deviceName, String propName, String propValue)
    {

    }

    /**
     * Ovveride this method in order to be notified when an configuration (cfg file) is loaded is
     * Micro-Manager
     */
    public void onSystemConfigurationLoaded()
    {

    }
}