package plugins.adufour.rotation3d;

import icy.canvas.Canvas3D;
import icy.canvas.IcyCanvas;
import icy.gui.viewer.Viewer;
import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.system.thread.ThreadUtil;
import icy.type.DataType;

import java.util.concurrent.Callable;

import plugins.adufour.ezplug.EzException;
import plugins.adufour.ezplug.EzLabel;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVarInteger;
import vtk.vtkCamera;
import vtk.vtkRenderWindow;
import vtk.vtkUnsignedCharArray;

public class Rotation3D extends EzPlug implements EzStoppable
{
    private EzVarInteger stepAngle;
    
    private boolean      stopFlag;
    
    @Override
    protected void initialize()
    {
        addEzComponent(new EzLabel("Rotation is achieved about the vertical axis"));
        addEzComponent(stepAngle = new EzVarInteger("Angle step", 5, 1, 90, 1));
        stepAngle.setToolTipText("Lower step = slower, smoother rotation");
    }
    
    @Override
    public void execute()
    {
        stopFlag = false;
        
        Viewer activeViewer = getActiveViewer();
        
        if (activeViewer == null) throw new EzException("No sequence opened.", true);
        
        IcyCanvas canvas = getActiveViewer().getCanvas();
        
        if (!(canvas instanceof Canvas3D)) throw new EzException("Rotation only works on a 3D (VTK) view.", true);
        
        final Sequence rotatedView = new Sequence("3D rotated view of " + getActiveSequence().getName());
        
        final Canvas3D c3d = (Canvas3D) canvas;
        final vtkCamera camera = c3d.getRenderer().GetActiveCamera();
        
        final int step = stepAngle.getValue();
        final int n = 360 / step;
        
        rotatedView.beginUpdate();
        
        Capture capture = new Capture(c3d);
        
        for (int i = 0; i < n; i++)
        {
            getUI().setProgressBarValue(i / (double) n);
            getUI().setProgressBarMessage("capturing angle " + i + "/" + n);
            
            // 1) set the angle
            camera.Azimuth(step);
            
            try
            {
                rotatedView.setImage(i, 0, ThreadUtil.invokeLater(capture, false).get());
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
            
            if (stopFlag) break;
        }
        rotatedView.endUpdate();
        
        addSequence(rotatedView);
    }
    
    private class Capture implements Callable<IcyBufferedImage>
    {
        final Canvas3D canvas;
        
        public Capture(Canvas3D canvas3D)
        {
            this.canvas = canvas3D;
        }
        
        public IcyBufferedImage call()
        {
            canvas.getPanel3D().Render();
            
            final vtkRenderWindow renderWindow = canvas.getRenderer().GetRenderWindow();
            final int[] size = renderWindow.GetSize();
            final int w = size[0];
            final int h = size[1];
            final vtkUnsignedCharArray array = new vtkUnsignedCharArray();
            
            // NOTE: in vtk the [0,0] pixel is bottom left, so a vertical flip is required
            renderWindow.GetRGBACharPixelData(0, 0, w - 1, h - 1, 1, array);
            
            // convert the vtk array into a IcyBufferedImage
            // convert the vtk array into a BufferedImage
            final byte[] inData = array.GetJavaArray();
            final IcyBufferedImage image = new IcyBufferedImage(w, h, 3, DataType.UBYTE);
            final byte[] r = image.getDataXYAsByte(0);
            final byte[] g = image.getDataXYAsByte(1);
            final byte[] b = image.getDataXYAsByte(2);
            
            int inOffset = 0;
            for (int y = h - 1; y >= 0; y--)
            {
                int outOffset = y * w;
                
                for (int x = 0; x < w; x++, outOffset++)
                {
                    r[outOffset] = inData[inOffset++];
                    g[outOffset] = inData[inOffset++];
                    b[outOffset] = inData[inOffset++];
                    inOffset++;// no alpha
                }
            }
            
            return image;
        }
    }
    
    @Override
    public void clean()
    {
        
    }
    
    @Override
    public void stopExecution()
    {
        stopFlag = true;
    }
}
