package plugins.adufour.activemeshes.energy;

import javax.vecmath.Point3d;

import plugins.adufour.activemeshes.mesh.Mesh;
import plugins.adufour.activemeshes.mesh.Vertex;

public abstract class Model
{
	protected final Mesh	mesh;
	
	public Model(Mesh mesh)
	{
		this.mesh = mesh;
	}
	
	/**
	 * Sends all stored deformations to the specified mesh depending on the given time step, and
	 * clears the list of deformations for the next iteration
	 * 
	 * @param mesh
	 *            the mesh to apply deformations from
	 * @param timeStep
	 *            the displacement time step
	 * @param min
	 *            the minimum bounds of the space volume, or null if no bounding is needed
	 * @param max
	 *            the maximum bounds of the space volume, or null if no bounding is needed
	 * @return the total squared displacement for all mesh vertices
	 */
	public static final void flushDeformations_OLD(Mesh mesh, double timeStep, Point3d min, Point3d max)
	{
		double maxDisp = mesh.getResolution() * timeStep;
		double maxDispSq = maxDisp * maxDisp;
		
		for (Vertex v : mesh.vertices)
		{
			if (v == null) continue;
			
			// cap vertex displacement to ensure stability
			double dispSq = v.forces.lengthSquared();
			if (dispSq > maxDispSq)
			{
				v.forces.scale(maxDisp / Math.sqrt(dispSq));
			}
			
			// apply the final computed force
			v.position.add(v.forces);
			
			// bound the force
			if (min != null)
			{
				double delta;
				if ((delta = (v.position.x - min.x)) < 0) v.position.x -= delta;
				if ((delta = (v.position.y - min.y)) < 0) v.position.y -= delta;
				if ((delta = (v.position.z - min.z)) < 0) v.position.z -= delta;
			}
			
			// bound the force
			if (max != null)
			{
				double delta;
				if ((delta = (v.position.x - max.x)) > 0) v.position.x -= delta;
				if ((delta = (v.position.y - max.y)) > 0) v.position.y -= delta;
				if ((delta = (v.position.z - max.z)) > 0) v.position.z -= delta;
			}
			
			// finally, reset forces for the next iteration
			v.forces.set(0, 0, 0);
		}
		
		mesh.updateMassCenter();
	}
	
	/**
	 * Sends all stored deformations to the specified mesh depending on the given time step, and
	 * clears the list of deformations for the next iteration
	 * 
	 * @param mesh
	 *            the mesh to apply deformations from
	 * @param timeStep
	 *            the displacement time step
	 * @param min
	 *            the minimum bounds of the space volume, or null if no bounding is needed
	 * @param max
	 *            the maximum bounds of the space volume, or null if no bounding is needed
	 * @return the total squared displacement for all mesh vertices
	 */
	public static final void flushDeformations(Mesh mesh, double timeStep, Point3d min, Point3d max)
	{
		double absMaxDisp = mesh.getResolution() * timeStep;
		double absMaxDispSquared = absMaxDisp * absMaxDisp;
		
		// compute the largest deformation first
		double maxDispSquared = 0;
		
		for (Vertex v : mesh.vertices)
			if (v != null)
			{
				double dispSquared = v.forces.lengthSquared();
				if (dispSquared > maxDispSquared) maxDispSquared = dispSquared;
			}
		
		// normalize all displacement according to this maximum
		double scaleFactor = (maxDispSquared > absMaxDispSquared) ? absMaxDisp / Math.sqrt(maxDispSquared) : 1.0;
		
		for (Vertex v : mesh.vertices)
		{
			if (v == null) continue;
			
			// normalize displacement to ensure stability
			if (scaleFactor != 1.0) v.forces.scale(scaleFactor);
			
			// apply the final computed force
			v.position.add(v.forces);
			
			// bound the force
			if (min != null)
			{
				double delta;
				if ((delta = (v.position.x - min.x)) < 0) v.position.x -= delta;
				if ((delta = (v.position.y - min.y)) < 0) v.position.y -= delta;
				if ((delta = (v.position.z - min.z)) < 0) v.position.z -= delta;
			}
			
			// bound the force
			if (max != null)
			{
				double delta;
				if ((delta = (v.position.x - max.x)) > 0) v.position.x -= delta;
				if ((delta = (v.position.y - max.y)) > 0) v.position.y -= delta;
				if ((delta = (v.position.z - max.z)) > 0) v.position.z -= delta;
			}
			
			// finally, reset forces for the next iteration
			v.forces.set(0, 0, 0);
		}
		
		mesh.updateMassCenter();
	}
	
	/**
	 * Computes forces linked to the current deformer implementation. The computed forces are not
	 * applied directly to the mesh, but stored locally in each mesh vertex. To apply these
	 * deformations, the {@link #flushDeformations(Mesh, double)} method must be called.
	 */
	public abstract void computeForces();
	
	/**
	 * unregister the mesh from this deformer
	 * 
	 * @param mesh
	 */
	public abstract void removeMesh(Mesh mesh); // FIXME design flaw: method shouldn't exist here
}
