package plugins.adufour.distancetransforms;

import icy.image.IcyBufferedImage;
import icy.roi.ROI;
import icy.sequence.Sequence;
import icy.sequence.SequenceDataIterator;
import icy.type.DataType;
import icy.type.collection.array.Array1DUtil;
import icy.util.OMEUtil;

import java.util.Arrays;
import java.util.List;

public abstract class ChamferDistanceTransform extends DistanceTransform
{
    
    @Override
    public Sequence createDistanceMap(Sequence input, int channel, double threshold, boolean invertMap, boolean useRealUnits)
    {
        int width = input.getSizeX();
        int height = input.getSizeY();
        int depth = input.getSizeZ();
        int time = input.getSizeT();
        DataType type = input.getDataType_();
        
        int maxDistance = width * height * depth;
        
        Sequence output = new Sequence("Distance map of " + input.getName());
        
        float pixelSizeXY = (float) (useRealUnits ? input.getPixelSizeX() : 1);
        float pixelSizeZ = (float) (useRealUnits ? input.getPixelSizeZ() : 1);
        
        for (int t = 0; t < time; t++)
        {
            for (int z = 0; z < depth; z++)
                output.setImage(t, z, new IcyBufferedImage(width, height, 1, DataType.FLOAT));
            
            float[][] map = output.getDataXYZAsFloat(t, 0);
            
            // Initialize the distance map according to the threshold
            for (int z = 0; z < depth; z++)
            {
                float[] mapSlice = map[z];
                Object in = input.getDataXY(t, z, channel);
                
                if (invertMap)
                {
                    for (int i = 0; i < mapSlice.length; i++)
                        if (Array1DUtil.getValue(in, i, type) > threshold) mapSlice[i] = maxDistance;
                }
                else
                {
                    for (int i = 0; i < mapSlice.length; i++)
                        if (Array1DUtil.getValue(in, i, type) <= threshold) mapSlice[i] = maxDistance;
                }
            }
            
            if (depth == 1)
            {
                updateUnsignedChamferDistance2D(map[0], width, pixelSizeXY);
            }
            else
            {
                updateUnsignedChamferDistance3D(map, width, pixelSizeXY, pixelSizeZ);
            }
            
        }
        
        output.dataChanged();
        return output;
    }
    
    /**
     * Creates a distance map from the ROI contained in the specified sequence. The resulting
     * distance map is a new sequence with one channel of type float and of same 4D dimensions as
     * the original sequence
     * 
     * @param sequence
     *            the sequence where ROI should be found and mapped
     * @param mapSelectedROI
     *            <code>true</code> will only take into account selected ROI when building the map
     * @param invertMap
     *            <code>true</code> will map in the inside of the ROI instead of the outside
     * @param useRealUnits
     *            <code>true</code> will return map values in real (metric) units, i.e. taking into
     *            account the pixel size
     * @return
     */
    public Sequence createDistanceMap(Sequence sequence, boolean mapOnlySelectedROI, boolean invertMap, boolean useRealUnits)
    {
        List<ROI> roi = mapOnlySelectedROI ? sequence.getSelectedROIs() : sequence.getROIs();
        
        String title = "Distance map ";
        title += "(in " + (useRealUnits ? "microns)" : "pixels)");
        title += " to ROI from " + sequence.getName();
        
        Sequence output = new Sequence(OMEUtil.createOMEMetadata(sequence.getMetadata()), title);
        
        int width = sequence.getSizeX();
        int height = sequence.getSizeY();
        int depth = sequence.getSizeZ();
        int length = sequence.getSizeT();
        
        for (int t = 0; t < length; t++)
        {
            for (int z = 0; z < depth; z++)
                output.setImage(t, z, new IcyBufferedImage(width, height, 1, DataType.FLOAT));
            
            float[][] map = output.getDataXYZAsFloat(t, 0);
            
            // Initialize the distance map
            
            int maxDistance = width * height * depth;
            
            if (!invertMap)
            {
                for (float[] slice : map)
                    Arrays.fill(slice, maxDistance);
                maxDistance = 0;
            }
            
            for (ROI r : roi)
            {
                SequenceDataIterator iterator = new SequenceDataIterator(output, r, true);
                while (!iterator.done())
                {
                    map[iterator.getPositionZ()][iterator.getPositionX() + iterator.getPositionY() * width] = maxDistance;
                    iterator.next();
                }
            }
            
            // pixel size
            float sizeXY = (float) (useRealUnits ? sequence.getPixelSizeX() : 1);
            float sizeZ = (float) (useRealUnits ? sequence.getPixelSizeZ() : 1);
            
            if (map.length == 1)
            {
                updateUnsignedChamferDistance2D(map[0], width, sizeXY);
            }
            else
            {
                updateUnsignedChamferDistance3D(map, width, sizeXY, sizeZ);
            }
        }
        
        return output;
        
    }
    
    /**
     * Updates an already-initialized 2D unsigned distance map (background value should be
     * infinity). The final map is given in the same units as given by the <code>pixelWidth</code>
     * parameter.
     * 
     * @param map
     *            the initialized map
     * @param lineSizeInPixels
     *            the width of the map
     * @param pixelWidth
     *            the width of the pixel (indicate <code>1</code> to obtain a pixel-based distance
     *            map, or the pixel size to obtain a distance map in real _metric_ units)
     */
    public abstract void updateUnsignedChamferDistance2D(final float[] map, int lineSizeInPixels, float pixelWidth);
    
    /**
     * Updates an already-initialized 3D unsigned distance map (background value should be infinity)
     * 
     * @param map
     *            the initialized map
     * @param lineSizeInPixels
     *            the width of the map
     * @param planarPixelSize
     *            the planar width of each voxel (indicate <code>1</code> to obtain a voxel-based
     *            distance map, or the voxel width along the XY axis to obtain a distance map in
     *            real _metric_ units)
     * @param planarPixelSize
     *            the axial width of each voxel (indicate <code>1</code> to obtain a voxel-based
     *            distance map, or the voxel width along the Z axis to obtain a distance map in real
     *            _metric_ units)
     */
    public abstract void updateUnsignedChamferDistance3D(final float[][] map, int lineSizeInPixels, float planarPixelSize, float axialPixelSize);
    
}
