package plugins.nchenouard.particletracking.legacytracker;

/**
 * 
 * Part of the Spot Tracking plugin for ICY: http://icy.bioimageanalysis.org/plugin/Spot_Tracking
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @version 3.1
 * @date 2013-11-13
 * @license gpl v3.0
*/

import icy.gui.frame.IcyFrame;
import icy.gui.frame.progress.AnnounceFrame;
import icy.gui.frame.progress.ProgressFrame;
import icy.main.Icy;
import icy.plugin.abstract_.PluginActionable;
import icy.sequence.Sequence;
import icy.swimmingPool.SwimmingObject;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.SwingUtilities;

import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackManager;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.particletracking.legacytracker.associationMethod.InstantaneousTracker;
import plugins.nchenouard.particletracking.legacytracker.associationMethod.Track;
import plugins.nchenouard.particletracking.legacytracker.gui.PanelTracking;
import plugins.nchenouard.spot.Detection;
import plugins.nchenouard.spot.DetectionResult;
import plugins.nchenouard.spot.Spot;

public class SpotTrackingPlugin extends PluginActionable implements
ActionListener {

	private IcyFrame mainFrame;
	private PanelTracking panelTracking;
	private StoppableTracker runningTracker = null;
	private boolean stopTracking = false;
	private boolean isRunning = false;

	private Lock stopLock = new ReentrantLock();
	private Condition stopCondition = stopLock.newCondition();
	private TrackManager trackManager = null;
	
	private Thread runningThread = null;
	
	@Override
	public void run() {
		panelTracking = new PanelTracking();
		panelTracking.trackingStartButton.addActionListener(this);
		enableTracking(true);
		
		mainFrame = new IcyFrame("Spot tracking", false, true, true, true);
		mainFrame.setContentPane(panelTracking);
		mainFrame.pack();
		mainFrame.addToMainDesktopPane();
		mainFrame.center();
		mainFrame.setVisible(true);
		mainFrame.requestFocus();

	}

	private void enableTracking(final boolean isEnableTracking)
	{
		SwingUtilities.invokeLater(new Runnable(){
			@Override
			public void run() {
				panelTracking.changeTrackingState(!isEnableTracking);				
			}});
	}

	public void actionPerformed(ActionEvent e)
	{
		if (e.getSource() == panelTracking.trackingStartButton)
		{
			if (isRunning)
			{
				stopLock.lock();
				if (runningTracker != null)
					runningTracker.stopComputing();
				stopTracking = true;
				try {
					while (isRunning)
						stopCondition.await();
				} catch (InterruptedException ie) {
					ie.printStackTrace();
				}
				finally
				{
					stopLock.unlock();
				}
				stopTracking = false;
			}
			else
			{
				DetectionResult detections = getDetectionsFromSwimmingpool();
				if (detections != null) {
					enableTracking(false);
					trackingRun(detections, detections.getSequence());
				} else
					new AnnounceFrame("Please first select a detection set.");
			}
		}
	}

	protected DetectionResult getDetectionsFromSwimmingpool() {
		return panelTracking.getDetectionResults();
	}

	private void sendTracksToPool(final TrackGroup trackGroup,
			final Sequence sequence) {
		SwingUtilities.invokeLater(new Runnable()
		{
			public void run() {
				SwimmingObject result = new SwimmingObject(trackGroup);// should
				Icy.getMainInterface().getSwimmingPool().add(result);
				if (trackManager != null)
					if (trackManager.isTrackManagerDestroyed())
						trackManager = null;
				if (trackManager == null)
					trackManager = new TrackManager();
				trackManager.setDisplaySequence(sequence);
			}
		});
	}

	private synchronized void trackingRun(final DetectionResult detections,
			final Sequence sequence) {
		if (detections != null) {
			isRunning = true;
			Thread trackThread = new Thread() {
				public void run() {
					int dim = 2;
					if (sequence.getSizeZ() > 1)
						dim = 3;
					Tracker tracker = panelTracking.buildTracker(dim);
					if (tracker instanceof StoppableTracker)
						runningTracker = (StoppableTracker)tracker;
					int firstT = detections.getFirstFrameTime();
					int lastT = detections.getLastFrameTime();
					String message = "Tracking at frame ";
					ProgressFrame announceFrame = new ProgressFrame("");
					announceFrame.setLength(lastT - firstT + 1);
					boolean stopped = false;
					for (int t = firstT; t <= lastT; t++) {
						if (!stopTracking)
						{
							announceFrame.setPosition(t);
							announceFrame.setMessage(message+" "+t);
							tracker.track(t, detections.getDetectionsAtT(t));
						}
						else
						{
							stopped = true;
							break;
						}
					}
					announceFrame.close();
					if (tracker instanceof InstantaneousTracker) {
						TrackGroup trackGroup = new TrackGroup(sequence);
						trackGroup.setDescription(panelTracking.getTrackGroupName());
						for (Track track : ((InstantaneousTracker) tracker)
								.getTracks()) {
							TrackSegment ts = convertTrack2TrackSegment(track,
									null);
							if (ts != null && ts.getDetectionList().size() > 1)//TODO: remove the size constraint
								trackGroup.addTrackSegment(ts);
						}
						sendTracksToPool(trackGroup, sequence);
						new AnnounceFrame("Tracks exported as "+trackGroup.getDescription()+" in TrackEditor");
					}
					isRunning = false;
					enableTracking(true);
					if (stopped)
					{
						stopLock.lock();
						isRunning = false;
						stopCondition.signalAll();
						stopLock.unlock();
					}
					runningThread = null;
					runningTracker = null;
				}
			};
			runningThread = trackThread;
			trackThread.start();
		}
	}

	public static TrackSegment convertTrack2TrackSegment(Track track,
			Object detectionEditor) {
		int firstIndex = track.getFirstIndex();
		int lastIndex = track.getLastIndex();
		for (int i = lastIndex; i > firstIndex; i--)
		{
			if (track.isPredictionAtFrame(i))
				lastIndex = i-1;
			else
				break;
		}
		if (firstIndex <= lastIndex)
		{
			ArrayList<Detection> detections = new ArrayList<Detection>(lastIndex - firstIndex + 1);
			for (int i = firstIndex; i <= lastIndex; i++)
			{
				Spot s = track.getSpotAtFrame(i);
				Detection detect = new Detection(s.mass_center.x, s.mass_center.y, s.mass_center.z, i);
				if (track.isPredictionAtFrame(i))
					detect.setDetectionType(Detection.DETECTIONTYPE_VIRTUAL_DETECTION);
				detections.add(detect);
			}
			return new TrackSegment(detections);
		}
		else
			return null;
	}
	
	public Tracker buildTrackerFromGUI(int dim)
	{
		return panelTracking.buildTracker(dim);
	}
}