package plugins.adufour.tracking;

import icy.canvas.IcyCanvas;
import icy.main.Icy;
import icy.painter.Overlay;
import icy.painter.Overlay.OverlayPriority;
import icy.sequence.Sequence;
import icy.swimmingPool.SwimmingObject;
import icy.system.thread.ThreadUtil;
import icy.type.point.Point5D.Double;

import java.awt.event.MouseEvent;
import java.util.HashMap;

import plugins.adufour.ezplug.EzLabel;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVarText;
import plugins.adufour.trackprocessors.TrackPainter;
import plugins.adufour.vars.util.VarException;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackManager;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.spot.Detection;

public class ManualTracking extends EzPlug implements EzStoppable
{
    final EzVarText                           trackGroupName = new EzVarText("Name of the group");
    
    private static TrackManager               trackManager;
    
    private final HashMap<String, TrackGroup> trackGroups    = new HashMap<String, TrackGroup>();
    
    @Override
    protected void initialize()
    {
        String instructions = "Instructions:\n";
        instructions += "1. Open (or focus on) the desired sequence\n";
        instructions += "2. Choose a group name for the tracks\n";
        instructions += "3. Start the plug-in (the Track Manager will appear)\n";
        instructions += "4. Click repeatedly on the center of an object to track it\n";
        instructions += "5. Stop the plug-in to stop the track\n";
        instructions += "6. Return to step 3 to track a new object in the same group\n";
        instructions += "7. Return to step 2 to create a new group\n";
        
        addEzComponent(new EzLabel(instructions));
        addEzComponent(trackGroupName);
        trackGroupName.setValue("My objects");
    }
    
    @Override
    public void clean()
    {
        trackGroups.clear();
    }
    
    @Override
    protected void execute()
    {
        if (trackGroupName.getValue().isEmpty()) throw new VarException(trackGroupName.getVariable(), "The group name cannot be empty");
        
        final Sequence sequence = getActiveSequence();
        
        if (sequence == null) return;
        
        ThreadUtil.invokeLater(new Runnable()
        {
            @Override
            public void run()
            {
                if (trackManager == null || trackManager.isTrackManagerDestroyed())
                {
                    trackManager = new TrackManager();
                    TrackPainter painter = new TrackPainter();
                    painter.setTrackPool(trackManager.getTrackPool());
                    trackManager.getTrackPool().addTrackProcessor(painter);
                }
                
                trackManager.setDisplaySequence(sequence);
            }
        });
        
        Overlay overlay = new Overlay("Manual Tracking", OverlayPriority.TOPMOST)
        {
            private final TrackSegment track     = new TrackSegment();
            
            private final String       groupName = trackGroupName.getValue();
            
            private SwimmingObject     swimmer;
            
            @Override
            public void mouseClick(MouseEvent e, Double imagePoint, IcyCanvas canvas)
            {
                e.consume();
                
                track.addDetection(new Detection(imagePoint.x, imagePoint.y, imagePoint.z, canvas.getPositionT()));
                canvas.setPositionT(canvas.getPositionT() + 1);
                
                if (track.getDetectionList().size() == 1)
                {
                    // this is the first detection => add it to the track group
                    
                    TrackGroup trackGroup = trackGroups.get(groupName);
                    
                    if (trackGroup == null)
                    {
                        trackGroup = new TrackGroup(sequence);
                        trackGroup.setDescription(groupName + " (manual tracking)");
                        trackGroups.put(groupName, trackGroup);
                        
                        // throw it in the pool
                        swimmer = new SwimmingObject(trackGroup);
                        Icy.getMainInterface().getSwimmingPool().add(swimmer);
                    }
                    
                    trackGroup.addTrackSegment(track);
                }
                
                // refresh the track manager
                trackManager.getTrackPool().fireTrackEditorProcessorChange();
            }
        };
        
        sequence.addOverlay(overlay);
        
        while (!Thread.currentThread().isInterrupted())
            ThreadUtil.sleep(100);
        
        sequence.removeOverlay(overlay);
    }
}
