package plugins.adufour.trackprocessors.speed;

import icy.gui.dialog.ConfirmDialog;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileOutputStream;
import java.util.ArrayList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JPanel;
import javax.vecmath.Vector3d;

import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;

import plugins.adufour.vars.gui.swing.SwingVarEditor;
import plugins.adufour.vars.lang.VarWorkbook;
import plugins.fab.trackmanager.PluginTrackManagerProcessor;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.spot.Detection;

public class RelativeMotion extends PluginTrackManagerProcessor implements ActionListener
{
    private JPanel    options      = new JPanel();
    private JCheckBox useRealUnits = new JCheckBox("Use real units");
    private JCheckBox showSpeed    = new JCheckBox("Show speed instead of displacement");
    
    public RelativeMotion()
    {
        setName("Relative motion");
        
        // options panel
        useRealUnits.setSelected(false);
        useRealUnits.addActionListener(this);
        useRealUnits.setEnabled(getActiveSequence() != null);
        
        showSpeed.setSelected(false);
        showSpeed.addActionListener(this);
        
        // assemble the main interface
        panel.setLayout(new BorderLayout());
        panel.add(options, BorderLayout.CENTER);
    }
    
    @Override
    public void Compute()
    {
        panel.setVisible(false);
        panel.removeAll();
        panel.setLayout(new BorderLayout());
        
        final VarWorkbook wb = new VarWorkbook("Relative Motion", "Relative motion");
        SwingVarEditor<Workbook> viewer = (SwingVarEditor<Workbook>) wb.createVarViewer();
        panel.add(viewer.getEditorComponent(), BorderLayout.CENTER);
        
        Sheet sheet = wb.getValue().getSheetAt(0);
        int rowIndex = 0;
        
        String unit = showSpeed.isSelected() ? "speed" : "disp.";
        if (showSpeed.isSelected())
        {
            unit += useRealUnits.isSelected() ? " (um/sec.)" : " (px./frame)";
        }
        else
        {
            unit += useRealUnits.isSelected() ? " (um)" : " (px.)";
        }
        
        Row headerRow = sheet.createRow(rowIndex++);
        headerRow.createCell(0).setCellValue("Frame");
        headerRow.createCell(1).setCellValue("Group");
        headerRow.createCell(2).setCellValue("Track");
        headerRow.createCell(3).setCellValue("Raw " + unit);
        headerRow.createCell(4).setCellValue("Ref. group");
        headerRow.createCell(5).setCellValue("Ref. track");
        headerRow.createCell(6).setCellValue("Relative " + unit);
        
        ArrayList<TrackGroup> trackGroups = trackPool.getTrackGroupList();
        
        for (int trackGroupIndexA = 0; trackGroupIndexA < trackGroups.size(); trackGroupIndexA++)
        {
            TrackGroup tgA = trackGroups.get(trackGroupIndexA);
            ArrayList<TrackSegment> trackSegmentsA = tgA.getTrackSegmentList();
            
            for (int trackGroupIndexB = 0; trackGroupIndexB < trackGroups.size(); trackGroupIndexB++)
            {
                TrackGroup tgB = trackGroups.get(trackGroupIndexB);
                ArrayList<TrackSegment> trackSegmentsB = tgB.getTrackSegmentList();
                
                for (int trackSegmentIndexA = 0; trackSegmentIndexA < trackSegmentsA.size(); trackSegmentIndexA++)
                {
                    TrackSegment tsA = trackSegmentsA.get(trackSegmentIndexA);
                    int trackIDA = tgA.getTrackSegmentList().indexOf(tsA);
                    ArrayList<Detection> detsA = tsA.getDetectionList();
                    
                    for (int trackSegmentIndexB = 0; trackSegmentIndexB < trackSegmentsB.size(); trackSegmentIndexB++)
                    {
                        TrackSegment tsB = trackSegmentsB.get(trackSegmentIndexB);
                        int trackIDB = tgB.getTrackSegmentList().indexOf(tsB);
                        ArrayList<Detection> detsB = tsB.getDetectionList();
                        
                        // don't compare the same track segment...
                        
                        if (tsA == tsB) continue;
                        
                        // measure the relative displacement between tsA and tsB
                        
                        for (int a = 1; a < detsA.size(); a++)
                        {
                            Detection a1 = detsA.get(a - 1);
                            Detection a2 = detsA.get(a);
                            
                            int currentFrame = a2.getT();
                            
                            Vector3d a1a2 = new Vector3d(a2.getX() - a1.getX(), a2.getY() - a1.getY(), a2.getZ() - a1.getZ());
                            
                            for (int b = 1; b < detsB.size(); b++)
                            {
                                Detection b1 = detsB.get(b - 1);
                                Detection b2 = detsB.get(b);
                                
                                // compare only simultaneous detections...
                                if (b2.getT() != currentFrame) continue;
                                
                                Vector3d b1b2 = new Vector3d(b2.getX() - b1.getX(), b2.getY() - b1.getY(), b2.getZ() - b1.getZ());
                                
                                Row row = sheet.createRow(rowIndex++);
                                row.createCell(0).setCellValue(currentFrame);
                                row.createCell(1).setCellValue(tgA.getDescription());
                                row.createCell(2).setCellValue("" + trackIDA);
                                
                                if (useRealUnits.isSelected())
                                {
                                    a1a2.x *= trackPool.getDisplaySequence().getPixelSizeX();
                                    a1a2.y *= trackPool.getDisplaySequence().getPixelSizeY();
                                    a1a2.z *= trackPool.getDisplaySequence().getPixelSizeZ();
                                    
                                    b1b2.x *= trackPool.getDisplaySequence().getPixelSizeX();
                                    b1b2.y *= trackPool.getDisplaySequence().getPixelSizeY();
                                    b1b2.z *= trackPool.getDisplaySequence().getPixelSizeZ();
                                }
                                
                                double rawValue = a1a2.length();
                                // subtract [B1 B2] to the motion vector [A1 A2]
                                a1a2.sub(b1b2);
                                double netValue = a1a2.length();
                                
                                if (useRealUnits.isSelected() && showSpeed.isSelected())
                                {
                                    rawValue /= trackPool.getDisplaySequence().getTimeInterval();
                                    netValue /= trackPool.getDisplaySequence().getTimeInterval();
                                }
                                
                                row.createCell(3).setCellValue(rawValue);
                                row.createCell(4).setCellValue(tgB.getDescription());
                                row.createCell(5).setCellValue("" + trackIDB);
                                row.createCell(6).setCellValue(netValue);
                            }
                        }
                    }
                }
            }
        }
        // notify the viewer that the value has changed
        viewer.valueChanged(wb, null, wb.getValue());
        
        JPanel options = new JPanel();
        
        options.setLayout(new BoxLayout(options, BoxLayout.X_AXIS));
        options.add(useRealUnits);
        options.add(showSpeed);
        options.add(Box.createHorizontalGlue());
        
        JButton bExport = new JButton("Export to Excel");
        bExport.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                JFileChooser f = new JFileChooser();
                if (f.showOpenDialog(getPanel().getParent()) != JFileChooser.APPROVE_OPTION) return;
                
                File file = f.getSelectedFile();
                if (!file.getPath().endsWith(".xls") && !file.getPath().endsWith(".xlsx")) file = new File(file.getPath() + ".xls");
                
                if (!file.exists() || ConfirmDialog.confirm("Overwrite " + file.getPath() + "?"))
                {
                    try
                    {
                        wb.getValue().write(new FileOutputStream(file, false));
                    }
                    catch (Exception e1)
                    {
                        throw new RuntimeException(e1);
                    }
                }
            }
        });
        options.add(bExport);
        
        panel.add(options, BorderLayout.NORTH);
        panel.setVisible(true);
    }
    
    @Override
    public void actionPerformed(ActionEvent arg0)
    {
        trackPool.fireTrackEditorProcessorChange();
    }
    
    @Override
    public void displaySequenceChanged()
    {
        useRealUnits.setEnabled(getActiveSequence() != null);
    }
    
    @Override
    public void Close()
    {
        
    }
    
}
