/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.process.process3D;

import icy.gui.frame.progress.ProgressFrame;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import plugins.big.bigsnakeutils.process.process1D.Convolver1D;

/**
 * Fast 2D LoG filter (multithreaded).
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class LoG {

	// ----------------------------------------------------------------------------
	// INPUT IMAGE

	/** Input image stack. */
	private final double[][] images_;

	/** Width of the image stack to filter. */
	private final int width_;
	/** Height of the image stack to filter. */
	private final int height_;
	/** Depth of the image stack to filter. */
	private final int depth_;

	// ----------------------------------------------------------------------------
	// KERNELS

	/**
	 * Contains the kernel
	 * cst*(x^2/(sigmaX^4)-1/(sigmaX^2))*exp(-(x^2)/(sigmaX^2)).
	 */
	private final double kernelFactX_[];
	/**
	 * Contains the kernel
	 * cst*(y^2/(sigmaY^4)-1/(sigmaY^2))*exp(-(y^2)/(sigmaY^2)).
	 */
	private final double kernelFactY_[];
	/**
	 * Contains the kernel
	 * cst*(z^2/(sigmaZ^4)-1/(sigmaZ^2))*exp(-(z^2)/(sigmaZ^2)).
	 */
	private final double kernelFactZ_[];
	/** Contains the kernel exp(-(x^2)/(sigmaX^2)). */
	private final double kernelBaseX_[];
	/** Contains the kernel exp(-(y^2)/(sigmaY^2)). */
	private final double kernelBaseY_[];
	/** Contains the kernel exp(-(z^2)/(sigmaZ^2)). */
	private final double kernelBaseZ_[];

	// ----------------------------------------------------------------------------
	// CONSTANTS

	/**
	 * This number indicates the number of 2*sigmas that are considered in the
	 * filtering.
	 */
	private final static double GAUSSIAN_TRUNCATION = 5.0;

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public LoG(double[][] array, int width, int height, int depth, double sigma) {
		images_ = new double[depth][width * height];
		for (int z = 0; z < depth; z++) {
			double[] imZ = array[z];
			for (int i = 0; i < imZ.length; i++) {
				images_[z][i] = imZ[i];
			}
		}

		double s = sigma > 0 ? sigma : 1.0;
		double cst = 1.0 / (s * s * s * Math.pow(2.0 * Math.PI, 3.0 / 2.0));

		width_ = width;
		height_ = height;
		depth_ = depth;

		kernelFactX_ = createKernelLoG_Fact(sigma, cst);
		kernelBaseX_ = createKernelLoG_Base(sigma);
		kernelFactY_ = createKernelLoG_Fact(sigma, cst);
		kernelBaseY_ = createKernelLoG_Base(sigma);
		kernelFactZ_ = createKernelLoG_Fact(sigma, cst);
		kernelBaseZ_ = createKernelLoG_Base(sigma);
	}

	// ----------------------------------------------------------------------------

	/** Applies the LoG filter. */
	public double[][] filter() {
		ProgressFrame pFrame = new ProgressFrame("Applying LoG filter");
		pFrame.setLength(3 * (width_ + height_ + depth_));

		LoGThreadX loGThreadXrays = new LoGThreadX(images_, kernelBaseX_,
				kernelBaseY_, kernelFactZ_, width_, height_, depth_, pFrame);
		LoGThreadY loGThreadYrays = new LoGThreadY(images_, kernelBaseX_,
				kernelFactY_, kernelBaseZ_, width_, height_, depth_, pFrame);
		LoGThreadZ loGThreadZrays = new LoGThreadZ(images_, kernelFactX_,
				kernelBaseY_, kernelBaseZ_, width_, height_, depth_, pFrame);

		ExecutorService executor = Executors.newFixedThreadPool(3);
		executor.execute(loGThreadXrays);
		executor.execute(loGThreadYrays);
		executor.execute(loGThreadZrays);
		executor.shutdown();
		while (!executor.isTerminated()) {
		}

		double[][] output = loGThreadXrays.getOutput();
		double[][] outputY = loGThreadYrays.getOutput();
		double[][] outputZ = loGThreadZrays.getOutput();

		for (int z = 0; z < depth_; z++) {
			for (int i = 0; i < width_ * height_; i++) {
				output[z][i] += outputY[z][i] + outputZ[z][i];
			}
		}

		loGThreadXrays = null;
		loGThreadYrays = null;
		loGThreadZrays = null;

		pFrame.close();
		return output;
	}

	// ============================================================================
	// PRIVATE METHODS

	/**
	 * Constructs the kernel
	 * cst*(x^2/(sigma^4)-1/(sigma^2))*exp(-(x^2)/(sigma^2)).
	 */
	private static double[] createKernelLoG_Fact(double sigma, double cst) {
		if (sigma <= 0.0) {
			double[] kernel = new double[1];
			kernel[0] = 1.0;
			return kernel;
		}

		double s2 = sigma * sigma;
		double s4 = s2 * s2;
		double dem = 2.0 * s2;
		int size = (int) Math
				.round(((int) (sigma * GAUSSIAN_TRUNCATION)) * 2.0 + 1);
		int size2 = size / 2;
		double[] kernel = new double[size];

		double x;
		for (int k = 0; k < size; k++) {
			x = (k - size2) * (k - size2);
			kernel[k] = -cst * (x / s4 - 1.0 / s2) * Math.exp(-x / dem);
		}
		return kernel;
	}

	// ----------------------------------------------------------------------------

	/** Constructs the kernel exp(-(x^2)/(sigma^2)). */
	private static double[] createKernelLoG_Base(double sigma) {
		if (sigma <= 0.0) {
			double[] kernel = new double[1];
			kernel[0] = 1.0;
			return kernel;
		}

		double s2 = sigma * sigma;
		double dem = 2.0 * s2;
		int size = (int) Math
				.round(((int) (sigma * GAUSSIAN_TRUNCATION)) * 2.0 + 1);
		int size2 = size / 2;
		double[] kernel = new double[size];

		double x;
		for (int k = 0; k < size; k++) {
			x = (k - size2) * (k - size2);
			kernel[k] = Math.exp(-x / dem);
		}
		return kernel;
	}

	// ============================================================================
	// INNER CLASSES

	/**
	 * Class that performs the 2D LoG filter in the 1st dimension in a single
	 * thread.
	 * 
	 * @version April 26, 2013
	 * 
	 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
	 */
	private class LoGThreadX implements Runnable {

		/** Input image stack. */
		private final double[][] images_;

		/** Output image stack. */
		private double[][] output_;

		/** Width of the image stack to filter. */
		private final int width_;

		/** Height of the image stack to filter. */
		private final int height_;

		/** Depth of the image stack to filter. */
		private final int depth_;

		/**
		 * Contains the kernel
		 * cst*(z^2/(sigmaZ^4)-1/(sigmaZ^2))*exp(-(z^2)/(sigmaZ^2)).
		 */
		private final double kernelFactZ_[];

		/** Contains the kernel exp(-(x^2)/(sigmaX^2)). */
		private final double kernelBaseX_[];

		/** Contains the kernel exp(-(y^2)/(sigmaY^2)). */
		private final double kernelBaseY_[];

		/** Progress bar that indicated the remaining time. */
		private final ProgressFrame pFrame_;

		// ============================================================================
		// PUBLIC METHODS

		/** Constructor. */
		public LoGThreadX(double[][] images, double[] kernelBaseX,
				double[] kernelBaseY, double[] kernelFactZ, int width,
				int height, int depth, ProgressFrame pFrame) {
			images_ = images;
			kernelBaseX_ = kernelBaseX;
			kernelBaseY_ = kernelBaseY;
			kernelFactZ_ = kernelFactZ;
			width_ = width;
			height_ = height;
			depth_ = depth;
			pFrame_ = pFrame;
		}

		// ----------------------------------------------------------------------------

		/** Performs the filtering. */
		@Override
		public void run() {
			output_ = new double[depth_][width_ * height_];

			for (int z = 0; z < depth_; z++) {
				System.arraycopy(images_[z], 0, output_[z], 0, width_ * height_);
			}

			double vinZ[] = new double[depth_];
			double voutZ[] = new double[depth_];
			for (int y = 0; y < height_; y++) {
				pFrame_.incPosition();
				for (int x = 0; x < width_; x++) {
					ImageArrayUtils.getZ(output_, width_, height_, depth_, x,
							y, 0, vinZ);
					Convolver1D.convolve(vinZ, voutZ, kernelFactZ_);
					ImageArrayUtils.putZ(output_, width_, height_, depth_, x,
							y, 0, voutZ);
				}
			}

			double vinY[] = new double[height_];
			double voutY[] = new double[height_];
			for (int x = 0; x < width_; x++) {
				pFrame_.incPosition();
				for (int z = 0; z < depth_; z++) {
					ImageArrayUtils.getY(output_, width_, height_, depth_, x,
							0, z, vinY);
					Convolver1D.convolve(vinY, voutY, kernelBaseY_);
					ImageArrayUtils.putY(output_, width_, height_, depth_, x,
							0, z, voutY);
				}
			}

			double vinX[] = new double[width_];
			double voutX[] = new double[width_];
			for (int z = 0; z < depth_; z++) {
				pFrame_.incPosition();
				for (int y = 0; y < height_; y++) {
					ImageArrayUtils.getX(output_, width_, height_, depth_, 0,
							y, z, vinX);
					Convolver1D.convolve(vinX, voutX, kernelBaseX_);
					ImageArrayUtils.putX(output_, width_, height_, depth_, 0,
							y, z, voutX);
				}
			}
		}

		// ----------------------------------------------------------------------------

		/** Returns the output image stack. */
		public double[][] getOutput() {
			return output_;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Class that performs the 2D LoG filter in the 2nd dimension in a single
	 * thread.
	 * 
	 * @version April 26, 2013
	 * 
	 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
	 */
	private class LoGThreadY implements Runnable {

		/** Input image stack. */
		private final double[][] images_;

		/** Output image stack. */
		private double[][] output_;

		/** Width of the image stack to filter. */
		private final int width_;

		/** Height of the image stack to filter. */
		private final int height_;

		/** Depth of the image stack to filter. */
		private final int depth_;

		/**
		 * Contains the kernel
		 * cst*(y^2/(sigmaY^4)-1/(sigmaY^2))*exp(-(y^2)/(sigmaY^2)).
		 */
		private final double kernelFactY_[];

		/** Contains the kernel exp(-(x^2)/(sigmaX^2)). */
		private final double kernelBaseX_[];

		/** Contains the kernel exp(-(z^2)/(sigmaZ^2)). */
		private final double kernelBaseZ_[];

		/** Progress bar that indicated the remaining time. */
		private final ProgressFrame pFrame_;

		// ============================================================================
		// PUBLIC METHODS

		/** Constructor. */
		public LoGThreadY(double[][] images, double[] kernelBaseX,
				double[] kernelFactY, double[] kernelBaseZ, int width,
				int height, int depth, ProgressFrame pFrame) {
			images_ = images;
			kernelBaseX_ = kernelBaseX;
			kernelFactY_ = kernelFactY;
			kernelBaseZ_ = kernelBaseZ;
			width_ = width;
			height_ = height;
			depth_ = depth;
			pFrame_ = pFrame;
		}

		// ----------------------------------------------------------------------------

		/** Performs the filtering. */
		@Override
		public void run() {
			output_ = new double[depth_][width_ * height_];

			for (int z = 0; z < depth_; z++) {
				System.arraycopy(images_[z], 0, output_[z], 0, width_ * height_);
			}

			double vinZ[] = new double[depth_];
			double voutZ[] = new double[depth_];
			for (int y = 0; y < height_; y++) {
				pFrame_.incPosition();
				for (int x = 0; x < width_; x++) {
					ImageArrayUtils.getZ(output_, width_, height_, depth_, x,
							y, 0, vinZ);
					Convolver1D.convolve(vinZ, voutZ, kernelBaseZ_);
					ImageArrayUtils.putZ(output_, width_, height_, depth_, x,
							y, 0, voutZ);
				}
			}

			double vinY[] = new double[height_];
			double voutY[] = new double[height_];
			for (int x = 0; x < width_; x++) {
				pFrame_.incPosition();
				for (int z = 0; z < depth_; z++) {
					ImageArrayUtils.getY(output_, width_, height_, depth_, x,
							0, z, vinY);
					Convolver1D.convolve(vinY, voutY, kernelFactY_);
					ImageArrayUtils.putY(output_, width_, height_, depth_, x,
							0, z, voutY);
				}
			}

			double vinX[] = new double[width_];
			double voutX[] = new double[width_];
			for (int z = 0; z < depth_; z++) {
				pFrame_.incPosition();
				for (int y = 0; y < height_; y++) {
					ImageArrayUtils.getX(output_, width_, height_, depth_, 0,
							y, z, vinX);
					Convolver1D.convolve(vinX, voutX, kernelBaseX_);
					ImageArrayUtils.putX(output_, width_, height_, depth_, 0,
							y, z, voutX);
				}
			}
		}

		// ----------------------------------------------------------------------------

		/** Returns the output image stack. */
		public double[][] getOutput() {
			return output_;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Class that performs the 2D LoG filter in the 3rd dimension in a single
	 * thread.
	 * 
	 * @version April 26, 2013
	 * 
	 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
	 */
	private class LoGThreadZ implements Runnable {

		/** Input image stack. */
		private final double[][] images_;

		/** Output image stack. */
		private double[][] output_;

		/** Width of the image stack to filter. */
		private final int width_;

		/** Height of the image stack to filter. */
		private final int height_;

		/** Depth of the image stack to filter. */
		private final int depth_;

		/**
		 * Contains the kernel
		 * cst*(x^2/(sigmaX^4)-1/(sigmaX^2))*exp(-(x^2)/(sigmaX^2)).
		 */
		private final double kernelFactX_[];

		/** Contains the kernel exp(-(y^2)/(sigmaY^2)). */
		private final double kernelBaseY_[];

		/** Contains the kernel exp(-(z^2)/(sigmaZ^2)). */
		private final double kernelBaseZ_[];

		/** Progress bar that indicated the remaining time. */
		private final ProgressFrame pFrame_;

		// ============================================================================
		// PUBLIC METHODS

		/** Constructor. */
		public LoGThreadZ(double[][] images, double[] kernelFactX,
				double[] kernelBaseY, double[] kernelBaseZ, int width,
				int height, int depth, ProgressFrame pFrame) {
			images_ = images;
			kernelFactX_ = kernelFactX;
			kernelBaseY_ = kernelBaseY;
			kernelBaseZ_ = kernelBaseZ;
			width_ = width;
			height_ = height;
			depth_ = depth;
			pFrame_ = pFrame;
		}

		// ----------------------------------------------------------------------------

		/** Performs the filtering. */
		@Override
		public void run() {
			output_ = new double[depth_][width_ * height_];

			for (int z = 0; z < depth_; z++) {
				System.arraycopy(images_[z], 0, output_[z], 0, width_ * height_);
			}

			double vinZ[] = new double[depth_];
			double voutZ[] = new double[depth_];
			for (int y = 0; y < height_; y++) {
				pFrame_.incPosition();
				for (int x = 0; x < width_; x++) {
					ImageArrayUtils.getZ(output_, width_, height_, depth_, x,
							y, 0, vinZ);
					Convolver1D.convolve(vinZ, voutZ, kernelBaseZ_);
					ImageArrayUtils.putZ(output_, width_, height_, depth_, x,
							y, 0, voutZ);
				}
			}

			double vinY[] = new double[height_];
			double voutY[] = new double[height_];
			for (int x = 0; x < width_; x++) {
				pFrame_.incPosition();
				for (int z = 0; z < depth_; z++) {
					ImageArrayUtils.getY(output_, width_, height_, depth_, x,
							0, z, vinY);
					Convolver1D.convolve(vinY, voutY, kernelBaseY_);
					ImageArrayUtils.putY(output_, width_, height_, depth_, x,
							0, z, voutY);
				}
			}

			double vinX[] = new double[width_];
			double voutX[] = new double[width_];
			for (int z = 0; z < depth_; z++) {
				pFrame_.incPosition();
				for (int y = 0; y < height_; y++) {
					ImageArrayUtils.getX(output_, width_, height_, depth_, 0,
							y, z, vinX);
					Convolver1D.convolve(vinX, voutX, kernelFactX_);
					ImageArrayUtils.putX(output_, width_, height_, depth_, 0,
							y, z, voutX);
				}
			}
		}

		// ----------------------------------------------------------------------------

		/** Returns the output image stack. */
		public double[][] getOutput() {
			return output_;
		}
	}

}
