/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.gui.curve;

import java.awt.geom.Point2D;
import java.util.Vector;

/**
 * Class describing a polynomial Curve. If the Curve owns N control points, the
 * PolynomialCurve is a polynom of degree N-1 interpolating the control points.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public class PolynomialCurve extends Curve {

	/** Default constructor */
	public PolynomialCurve() {
		this(0, 1);
	}

	/**
	 * Constructs the curve specifying the position of the first and last
	 * points.
	 */
	public PolynomialCurve(double y0, double y1) {
		controlPoints_.add(new Point2D.Double(0, y0));
		controlPoints_.add(new Point2D.Double(1, y1));
	}

	/**
	 * Default constructor.
	 * 
	 * @param points
	 *            Control points
	 */
	public PolynomialCurve(final Vector<Point2D> points) {
		setControlPoints(points);
	}

	/** Copy constructor */
	public PolynomialCurve(final PolynomialCurve other) {
		this(other.controlPoints_);
	}

	@Override
	public void setControlPoints(Vector<Point2D> points) {
		if (controlPoints_ == points) {
			return;
		}

		if (!points.isEmpty()) {
			if (points.size() < 2) {
				throw new IllegalArgumentException(
						"Curve must contains at least 2 points");
			}
			if (points.get(0).getX() != 0) {
				throw new IllegalArgumentException(
						"First point of the curve must be 0");
			}
			if (points.get(points.size() - 1).getX() != 1) {
				throw new IllegalArgumentException(
						"Last point of the curve must be 1");
			}

			controlPoints_.clear();
			controlPoints_.addAll(points);
		} else {
			clear();
		}
	}

	@Override
	public void insertControlPoint(int index, Point2D point) {
		checkPointValidity(index, point);
		controlPoints_.insertElementAt(point, index);
	}

	@Override
	public void removeControlPoint(int index) {
		controlPoints_.remove(index);
	}

	@Override
	public void setControlPoint(int index, Point2D point) {
		checkPointValidity(index, point);
		controlPoints_.set(index, point);
	}

	@Override
	public void clear() {
		controlPoints_.clear();
		controlPoints_.add(new Point2D.Double(0, 0));
		controlPoints_.add(new Point2D.Double(1, 1));
	}

	@Override
	public double valueAt(double x) {
		double res = 0;

		for (Point2D pt : controlPoints_) {
			double a = pt.getY();
			for (Point2D pt2 : controlPoints_) {
				if (pt != pt2) {
					a *= (x - pt2.getX()) / (pt.getX() - pt2.getX());
				}
			}
			res += a;
		}

		return res;
	}

	@Override
	public Curve clone() {
		return new PolynomialCurve(this);
	}

}
