package plugins.adufour.distancetransforms;

/**
 * Pseudo-Euclidean distance transform based on a two-pass sweeping algorithm using a 3x3 (in 2D) or
 * 3x3x3 (in 3D) chamfer mask. <br/>
 * The chamfer mask defines distances as follows:
 * 
 * <pre>
 *                  b a b
 *  Z = 0 (or 2D):  a x a
 *                  b a b
 * 
 *                  e d e
 *  Z = +/- 1:      d c d
 *                  e d e
 * </pre>
 * 
 * @author Alexandre Dufour
 */
public class Chamfer3 extends ChamferDistanceTransform
{
    public void updateUnsignedChamferDistance2D(float[] map, int width, float planarLateralDistance)
    {
        double xy2 = planarLateralDistance * planarLateralDistance;
        float planarDiagonalDistance = (float) Math.sqrt(xy2 + xy2);
        
        int height = map.length / width;
        int i, j, index = 0;
        float aDist, bDist;
        
        // First pass
        
        for (j = 0; j < height; j++)
            for (i = 0; i < width; i++, index++)
            {
                aDist = map[index] + planarLateralDistance;
                
                if (i < width - 1 && map[index + 1] > aDist) map[index + 1] = aDist;
                
                if (j < height - 1)
                {
                    bDist = aDist - planarLateralDistance + planarDiagonalDistance;
                    
                    if (map[index + width] > aDist) map[index + width] = aDist;
                    
                    if (i < width - 1 && map[index + width + 1] > bDist) map[index + width + 1] = bDist;
                    
                    if (i > 0 && map[index + width - 1] > bDist) map[index + width - 1] = bDist;
                }
            }
        
        index--;
        
        // Second pass
        
        for (j = height - 1; j >= 0; j--)
            for (i = width - 1; i >= 0; i--, index--)
            {
                aDist = map[index] + planarLateralDistance;
                
                if (i > 0 && map[index - 1] > aDist) map[index - 1] = aDist;
                
                if (j > 0)
                {
                    bDist = aDist - planarLateralDistance + planarDiagonalDistance;
                    
                    if (map[index - width] > aDist) map[index - width] = aDist;
                    
                    if (i > 0 && map[index - width - 1] > bDist) map[index - width - 1] = bDist;
                    
                    if (i < width - 1 && map[index - width + 1] > bDist) map[index - width + 1] = bDist;
                }
            }
    }
    
    public void updateUnsignedChamferDistance3D(float[][] map, int width, float planarLateralDistance, float axialLateralDistance)
    {
        double xy2 = planarLateralDistance * planarLateralDistance;
        double z2 = axialLateralDistance * axialLateralDistance;
        float xyDiag = (float) Math.sqrt(xy2 + xy2);
        float xzDiag = (float) Math.sqrt(xy2 + z2);
        float xyzDiag = (float) Math.sqrt(xy2 + xy2 + z2);
        
        int height = map[0].length / width;
        int depth = map.length;
        
        int i, j, k, offset = 0;
        
        // First pass
        
        for (k = 0; k < depth - 1; k++)
        {
            float[] thisSlice = map[k];
            float[] nextSlice = (k < depth - 1) ? map[k + 1] : null;
            
            offset = 0;
            
            for (j = 0; j < height; j++)
                for (i = 0; i < width; i++, offset++)
                {
                    float val = thisSlice[offset];
                    float xyLatDist = val + planarLateralDistance;
                    float xzLatDist = val + axialLateralDistance;
                    float xyDiagDist = val + xyDiag;
                    float xzDiagDist = val + xzDiag;
                    float xyzDiagDist = val + xyzDiag;
                    
                    if (k < depth - 1)
                    {
                        if (nextSlice[offset] > xzLatDist) nextSlice[offset] = xzLatDist;
                        
                        if (j < height - 1)
                        {
                            if (nextSlice[offset + width] > xzDiagDist) nextSlice[offset + width] = xzDiagDist;
                            
                            if (i > 0 && nextSlice[offset + width - 1] > xyzDiagDist) nextSlice[offset + width - 1] = xyzDiagDist;
                            
                            if (i < width - 1 && nextSlice[offset + width + 1] > xyzDiagDist) nextSlice[offset + width + 1] = xyzDiagDist;
                        }
                        
                        if (j > 0)
                        {
                            if (nextSlice[offset - width] > xzDiagDist) nextSlice[offset - width] = xzDiagDist;
                            
                            if (i > 0 && nextSlice[offset - width - 1] > xyzDiagDist) nextSlice[offset - width - 1] = xyzDiagDist;
                            
                            if (i < width - 1 && nextSlice[offset - width + 1] > xyzDiagDist) nextSlice[offset - width + 1] = xyzDiagDist;
                        }
                        
                        if (i > 0 && nextSlice[offset - 1] > xzDiagDist) nextSlice[offset - 1] = xzDiagDist;
                        
                        if (i < width - 1 && nextSlice[offset + 1] > xzDiagDist) nextSlice[offset + 1] = xzDiagDist;
                    }
                    
                    if (j < height - 1)
                    {
                        if (thisSlice[offset + width] > xyLatDist) thisSlice[offset + width] = xyLatDist;
                        
                        if (i < width - 1 && thisSlice[offset + width + 1] > xyDiagDist) thisSlice[offset + width + 1] = xyDiagDist;
                        
                        if (i > 0 && thisSlice[offset + width - 1] > xyDiagDist) thisSlice[offset + width - 1] = xyDiagDist;
                    }
                    
                    if (i < width - 1 && thisSlice[offset + 1] > xyLatDist) thisSlice[offset + 1] = xyLatDist;
                }
        }
        
        // Second pass
        
        for (k = depth - 1; k > 0; k--)
        {
            float[] thisSlice = map[k];
            float[] prevSlice = (k > 0) ? map[k - 1] : null;
            
            offset = thisSlice.length - 1;
            
            for (j = height - 1; j >= 0; j--)
                for (i = width - 1; i >= 0; i--, offset--)
                {
                    float val = thisSlice[offset];
                    float xyLatDist = val + planarLateralDistance;
                    float xzLatDist = val + axialLateralDistance;
                    float xyDiagDist = val + xyDiag;
                    float xzDiagDist = val + xzDiag;
                    
                    if (k > 0)
                    {
                        float xyzDiagDist = val + xyzDiag;
                        
                        if (prevSlice[offset] > xzLatDist) prevSlice[offset] = xzLatDist;
                        
                        if (j > 0)
                        {
                            if (prevSlice[offset - width] > xzDiagDist) prevSlice[offset - width] = xzDiagDist;
                            
                            if (i > 0 && prevSlice[offset - width - 1] > xyzDiagDist) prevSlice[offset - width - 1] = xyzDiagDist;
                            
                            if (i < width - 1 && prevSlice[offset - width + 1] > xyzDiagDist) prevSlice[offset - width + 1] = xyzDiagDist;
                        }
                        
                        if (j < height - 1)
                        {
                            if (prevSlice[offset + width] > xzDiagDist) prevSlice[offset + width] = xzDiagDist;
                            
                            if (i > 0 && prevSlice[offset + width - 1] > xyzDiagDist) prevSlice[offset + width - 1] = xyzDiagDist;
                            
                            if (i < width - 1 && prevSlice[offset + width + 1] > xyzDiagDist) prevSlice[offset + width + 1] = xyzDiagDist;
                        }
                        
                        if (i > 0 && prevSlice[offset - 1] > xzDiagDist) prevSlice[offset - 1] = xzDiagDist;
                        
                        if (i < width - 1 && prevSlice[offset + 1] > xzDiagDist) prevSlice[offset + 1] = xzDiagDist;
                    }
                    
                    if (j > 0)
                    {
                        if (thisSlice[offset - width] > xyLatDist) thisSlice[offset - width] = xyLatDist;
                        
                        if (i > 0 && thisSlice[offset - width - 1] > xyDiagDist) thisSlice[offset - width - 1] = xyDiagDist;
                        
                        if (i < width - 1 && thisSlice[offset - width + 1] > xyDiagDist) thisSlice[offset - width + 1] = xyDiagDist;
                    }
                    
                    if (i > 0 && thisSlice[offset - 1] > xyLatDist) thisSlice[offset - 1] = xyLatDist;
                }
        }
    }
    
}
