/*
*   Class  Strings
*
*   Methods for editing Strings
*
*   WRITTEN BY: Dr Michael Thomas Flanagan
*
*   DATE:       March 2013
*   REVISED:    April 2013, 2 May 2013, 30 November 2013
*
*   DOCUMENTATION:
*   See Michael Thomas Flanagan's Java library on-line web page:
*   http://www.ee.ucl.ac.uk/~mflanaga/java/Strings.html
*   http://www.ee.ucl.ac.uk/~mflanaga/java/
*
*   Copyright (c)2013 Michael Thomas Flanagan
*
*
***************************************************************************************/

package flanagan.util;

import java.util.ArrayList;

import flanagan.math.Fmath;
import flanagan.io.PrintToScreen;

public class Strings{
        
    private String enteredString = null;                        // Entered string for editing
    private int nEntered = 0;                                   // Number of characters in the entered string
    private String editedString = null;                         // Edited string 
    private int nEdited = 0;                                    // Number of characters in the edited string
    private boolean[] whiteSpaces = null;                       // true if corresponding char is a white space
    private String[] tokens = null;                             // tokens separated by white space/s
    private int nTokens = 0;                                    // number of tokens
    private int[] tokenInitialIndices = null;                   // initial index of each token
    private int[] tokenFinalIndices = null;                     // final index of each token
    private boolean tokensDone = false;                         // = true when tokens found
    private String[] principalTokens = null;                    // tokens, other tan the first token, that are not articles, prepostions or conjunctions
    private int nPrincipalTokens = 0;                           // number of principal tokens
    private int[] principalTokenInitialIndices = null;          // initial index of each principal token
    private int[] principalTokenFinalIndices = null;            // final index of each principaltoken
   
    
    // Accent  arrays
    private String[] symbolsA = {"", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ""};

    private String[] replSymbolsA = {"A", "A", "A", "A", "A", "A", "AE", "C", "E", "E", "E", "E", "I", "I", "I", "I", "DH", "N", "O", "O", "O", "O", "O", "O", "U", "U", "U", "U", "Y", "TH", "ss", "a", "a", "a", "a", "a", "a", "ae", "c", "e", "e", "e", "e", "i", "i", "i", "i", "dh", "n", "o", "o", "o", "o", "o", "o", "u", "u", "u", "u", "y", "th", "y", "OE", "oe", "s", "s", "y", "f"};

    private String[] htmlNamesA = {"&Agrave;", "&Aacute;", "&Acirc;", "&Atilde;", "&Auml;", "&Aring;", "&AElig;", "&Ccedil;", "&Egrave;", "&Eacute;", "&Ecirc;", "&Euml;", "&Igrave;", "&Iacute;", "&Icirc;", "&Iuml;", "&ETH;", "&Ntilde;", "&Ograve;", "&Oacute;", "&Ocirc;", "&Otilde;", "&Ouml;", "&Oslash;", "&Ugrave;", "&Uacute;", "&Ucirc;", "&Uuml;", "&Yacute;", "&THORN;", "&szlig;", "&agrave;", "&aacute;", "&acirc;", "&atilde;", "&auml;", "&aring;", "&aelig;", "&ccedil;", "&egrave;", "&eacute;", "&ecirc;", "&euml;", "&igrave;", "&iacute;", "&icirc;", "&iuml;", "&eth;", "&ntilde;", "&ograve;", "&oacute;", "&ocirc;", "&otilde;", "&ouml;", "&oslash;", "&ugrave;", "&uacute;", "&ucirc;", "&uuml;", "&yacute;", "&thorn;", "&yuml;", "&OE;", "&oe;", "&Scaron;", "&scaron;", "&Ydia;", "&fhook;"}; 

    private String[] htmlNumbersA = {"&#192;", "&#193;", "&#194;", "&#195;", "&#196;", "&#197;", "&#198;", "&#199;", "&#200;", "&#201;", "&#202;", "&#203;", "&#204;", "&#205;", "&#206;", "&#207;", "&#208;", "&#209;", "&#210;", "&#211;", "&#212;", "&#213;", "&#214;", "&#216;", "&#217;", "&#218;", "&#219;", "&#220;", "&#221;", "&#222;", "&#223;", "&#224;", "&#225;", "&#226;", "&#227;", "&#228;", "&#229;", "&#230;", "&#231;", "&#232;", "&#233;", "&#234;", "&#235;", "&#236;", "&#237;", "&#238;", "&#239;", "&#240;", "&#241;", "&#242;", "&#243;", "&#244;", "&#245;", "&#246;", "&#248;", "&#249;", "&#250;", "&#251;", "&#252;", "&#253;", "&#254;", "&#255;", "&#338;", "&#339;", "&#352;", "&#353;", "&#376;", "&#402;"}; 

    private int[] decNumbersA = {192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 248, 249, 250, 251, 252, 253, 254, 255, 338, 339, 352, 353, 376, 402}; 

    private int nSymbolsA = symbolsA.length;
    
    // Quotation mark  arrays   
    private String[] symbolQ = {"\"", "'", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ""};
    
    private String[] htmlNamesQ = {"&quot;", "&none", "&sbquo;", "&dbquo;", "&lsaquo;", "&lsquo;", "&rsquo;", "&ldquo;", "&rdquo;", "&rsaquo;", "&laquo;", "&raquo;", "&none;", "&none;", "&none;", "&none;", "&none;", "&none;"};
    
    private String[] htmlNumbersQ = {"&#34;", "&#39;", "&#130;", "&#132;", "&#139;", "&#145;", "&#146;", "&#147;", "&#148;", "&#155;", "&#171;", "&#187;", "&#8216;", "&#8217;", "&#8218;", "&#8220;", "&#8221;", "&#8222;"};
                    
    private int[] decNumbersQ = {34, 39, 130, 132, 139, 145, 146, 147, 148, 155, 171, 187, 8216, 8217, 8218, 8220, 8221, 8222};  
    
    private int nSymbolsQ = symbolQ.length;
    
    
    // Single quotation mark arrays   
    private String[] symbolsSQ = {"'", "", "", "", ""};
    
    private String[] htmlNamesSQ = {"'", "&lsquo;", "&rsquo;", "&none;", "&none;"};
    
    private String[] htmlNumbersSQ = {"&#39;", "&#145;", "&#146;", "&#8216;", "&#8217;"};
                    
    private int[] decNumbersSQ = {39, 145, 146, 8216, 8217};  
    
    private int nSymbolsSQ = symbolsSQ.length;
    
    // dash arrays
    private String[] symbolsD = {"-", "", "", "", "", "", ""};
    
    private String[] htmlNamesD = {"&none;", "&ndash;", "&mdash;", "", "&oline;", "&none;", "&none;"};
    
    private String[] htmlNumbersD = {"&#45;", "&#150;", "&#151;", "&#8213;", "&#8254;", "&#8211;", "&#8212;"};
    
    private int[] decNumbersD = {45, 150, 151, 8213, 8254, 8211, 8212};
    
    private int nSymbolsD = symbolsD.length;
    
    // Articles
    private String[] articles = {"a", "an", "the"};
    private int nArticles = articles.length;
    private String[] prepositions = {"about", "above", "across", "after", "against", "along", "among", "around", "at", "before", "behind", "below", "beneath", "beside", "between", "beyond", "but", "by", "despite", "down", "during", "except", "for", "from", "in", "inside", "into", "like", "near", "of", "off", "on", "onto", "out", "outside", "over", "past", "since", "through", "throughout", "till", "to", "toward", "under", "underneath", "until", "up", "upon", "with", "within", "without"};
    private int nPrepositions = prepositions.length;
    private String[] conjunctions = {"and", "&amp", "&", "but", "or", "nor", "for", "so", "yet", "not", "only", "also", "either", "neither", "although", "because", "since", "unless", "until", "while"};
    private int nConjunctions = conjunctions.length;
    
    // Constructors 
    public Strings(String string){
        this.enteredString = string;
        this.editedString = string;
        this.nEntered = string.length();
        this.nEdited = this.nEntered;
    }
    
    public Strings(){
        // for use with Updater
    }
    
    // Return edited String 
    public String getEditedString(){
        return this.editedString;
    }
    
    // Return number of characters in the edited String 
    public int getEditedStringLength(){
        return this.nEdited;
    }
    
   // Return enteredString 
    public String getEnteredString(){
        return this.enteredString;
    }
    
    // Return number of characters in the entered String 
    public int getEnteredStringLength(){
        return this.nEntered;
    }
    
    // Return array of single quotation mark numbers
    // instance method
    public String[] getSingleQuotationMarkHtmlNumbers(){
        return this.htmlNumbersSQ;
    }
    
       
    // converts all the string to upper case
    public String toUpperCase(){
        this.editedString = this.editedString.toUpperCase();
        return this.editedString;
    }
    
    // converts to the character at i to upper case
    public String toUpperCase(int i){
        this.editedString = this.editedString.substring(0,i) + this.editedString.substring(i,i+1).toUpperCase() + this.editedString.substring(i+1);
        return this.editedString;
    }
    
    // converts to lower case
    public String toLowerCase(){
        this.editedString = this.editedString.toLowerCase();
        return this.editedString;
    }
    
    // converts to the character at i to lower case
    public String toLowerCase(int i){
        this.editedString = this.editedString.substring(0,i) + this.editedString.substring(i,i+1).toLowerCase() + this.editedString.substring(i+1);
        return this.editedString;
    }
    
    // Return string in full title case
    // instance method
    public String toAllTitleCase(){
        
        this.editedString = this.editedString.toLowerCase();
        this.tokens();
        
        int jj = 0;
        for(int i=0; i<this.nTokens; i++){
            jj = this.tokenInitialIndices[i];
            if(Chars.isQuotationMark(this.editedString.charAt(jj)) && jj<this.nEdited)jj++;
            if(!this.whiteSpaces[jj]){
                this.toUpperCase(jj);
            }   
        }     
        return this.editedString;
    }
    
    // Return string in full title case
    // static method
    public static String toAllTitleCase(String ss){
       Strings strs = new Strings(ss);
       return strs.toAllTitleCase();
    }
    
    // Return string in title case, i.e. only prncipal words with initial upper case
    // instance method
    public String toTitleCase(){
        this.editedString = this.editedString.toLowerCase();
        this.principalTokens();
        
        int jj = 0;
        for(int i=0; i<this.nPrincipalTokens; i++){
            jj = this.principalTokenInitialIndices[i];
            if(Chars.isQuotationMark(this.editedString.charAt(jj)) && jj<this.nEdited)jj++;
            if(!this.whiteSpaces[jj]){
                this.toUpperCase(jj);
            }   
        }     
        return this.editedString;
    }
    
    // Return string in title case, i.e. only prncipal words with initial upper case
    // static method
    public static String toTitleCase(String ss){
       Strings strs = new Strings(ss);
       return strs.toTitleCase();
    }
    
    
    // Converts to sentence case
    // instance method
    public String toSentenceCase(){
        this.editedString = this.editedString.toLowerCase();
        this.tokens();
        
        int jj = this.tokenInitialIndices[0];
        int kk = -1;
        if(Chars.isQuotationMark(this.editedString.charAt(jj)) && jj<this.nEdited)jj++;
        if(!this.whiteSpaces[jj]){
            this.toUpperCase(jj);
        }
        for(int i=0;i<this.nTokens-1; i++){
            jj = this.tokenFinalIndices[i];
            if(Chars.isQuotationMark(this.editedString.charAt(jj)))jj--;
            if(this.editedString.charAt(jj)=='.'){
                kk = this.tokenInitialIndices[i+1];
                if(Chars.isQuotationMark(this.editedString.charAt(kk)))kk++;
                if(!this.whiteSpaces[kk]){
                    this.toUpperCase(kk);
                }
            }
        }
        return this.editedString;
    }
    
    // Converts to sentence case
    // static method
      public static String toSentenceCase(String ss){
       Strings strs = new Strings(ss);
       return strs.toSentenceCase();
    }
      
    // Checks for white spaces
    // instance method
    public boolean[] whiteSpaces(){
        this.whiteSpaces = new boolean[this.nEdited];
        boolean test = true;
        int iFirst = -1;
        for(int i=0; i<this.nEdited; i++){
            whiteSpaces[i] = false;
            if(Character.isWhitespace(this.editedString.charAt(i))){
                whiteSpaces[i] = true;
            }
        }
        return this.whiteSpaces;
    }
   
    // Checks for white spaces
    // static method
    public static boolean[] whiteSpaces(String ss){
       Strings strs = new Strings(ss);
       return strs.whiteSpaces();
    } 
   
    // Split the string into tokens
    // instance method
    public String[] tokens(){
        this.whiteSpaces();
        ArrayList<Integer> al = new ArrayList<Integer>();
        boolean test0 = true;
        boolean test1 = false;
        for(int i=0; i<this.nEdited; i++){
            if(test0){
                if(!this.whiteSpaces[i]){
                    al.add(i);
                    test0 = false;
                    test1 = true;
                }
            }
            else{
                if(test1){
                    if(this.whiteSpaces[i]){
                        al.add(i);
                        test1 = false;
                        test0 = true;
                    }
                }
            }
        }

        int n = al.size();
        if(Fmath.isOdd(n)){
            al.add(nEdited);
            n++;
        }
       
        this.nTokens = n/2;
        this.tokens = new String[this.nTokens];
        this.tokenInitialIndices = new int[this.nTokens];
        this.tokenFinalIndices = new int[this.nTokens];
        int j = -1;
        for(int i=0; i<this.nTokens; i++){
            
            int pos0 = al.get(++j);
            int pos1 = al.get(++j);
            this.tokens[i] = this.editedString.substring(pos0,pos1);
            this.tokenInitialIndices[i] = pos0;
            this.tokenFinalIndices[i] = pos1-1;
        }  
        this.tokensDone = true;
        return this.tokens;
    }
    
    // Split the string into tokens
    // static method
    public static String[] tokens(String ss){
       Strings strs = new Strings(ss);
       return strs.tokens();
    }
    
    // Return token initial indices
    // instance method
    public int[] tokenInitialIndices(){
        if(!this.tokensDone)this.tokens();
        return this.tokenInitialIndices;
    }
    
    
    // Return token initial indices
    // static method
    public static int[] tokenInitialIndices(String ss){
        Strings strs = new Strings(ss);
        return strs.tokenInitialIndices();
    }
    
    // Return token final indices
    // instance method
    public int[] tokenFinalIndices(){
        if(!this.tokensDone)this.tokens();
        return this.tokenFinalIndices;
    }
    
    // Return token final indices
    // static method
    public static int[] tokenFinalIndices(String ss){
        Strings strs = new Strings(ss);
        return strs.tokenFinalIndices();
    }
    
    // Split the string into principal tokens
    // instance method
    public String[] principalTokens(){
        if(!this.tokensDone)this.tokens();
        ArrayList<Integer> al = new ArrayList<Integer>();
        al.add(0);
        
        for(int i=1; i<this.nTokens; i++){
            String tt = Strings.removeQuotationMarks(this.tokens[i]);
            boolean test = true;
            for(int j=0; j<this.nArticles; j++){
                if(tt.equalsIgnoreCase(this.articles[j])){
                    test = false;
                    break;
                }
            }
            if(test){
                for(int j=0; j<this.nPrepositions; j++){
                    if(tt.equalsIgnoreCase(this.prepositions[j])){
                        test = false;
                        break;
                    }
                }
            }
            if(test){
                for(int j=0; j<this.nConjunctions; j++){
                    if(tt.equalsIgnoreCase(this.conjunctions[j])){
                        test = false;
                        break;
                    }
                }
            }
            if(test)al.add(i);
        }
        
        this.nPrincipalTokens = al.size();
        this.principalTokens = new String[this.nPrincipalTokens];
        this.principalTokenInitialIndices = new int[this.nPrincipalTokens];
        this.principalTokenFinalIndices = new int[this.nPrincipalTokens];
        for(int i=0; i<this.nPrincipalTokens; i++){
            int j = al.get(i);
            this.principalTokens[i] = this.tokens[j];
            this.principalTokenInitialIndices[i] = this.tokenInitialIndices[j];
            this.principalTokenFinalIndices[i] = this.tokenFinalIndices[j];
        }
        
        return this.principalTokens;
    }
    
    // Split the string into principal tokens
    // static method
    public static String[] principalTokens(String ss){
        Strings strs = new Strings(ss);
        return strs.principalTokens();
    }
    
    // Split the string into principal tokens ignoring obligatory  initial principal token
    // instance method
    public String[] principalTokensZero(){
        if(!this.tokensDone)this.tokens();
        ArrayList<Integer> al = new ArrayList<Integer>();
        
        for(int i=0; i<this.nTokens; i++){
            String tt = Strings.removeQuotationMarks(this.tokens[i]);
            boolean test = true;
            for(int j=0; j<this.nArticles; j++){
                if(tt.equalsIgnoreCase(this.articles[j])){
                    test = false;
                    break;
                }
            }
            if(test){
                for(int j=0; j<this.nPrepositions; j++){
                    if(tt.equalsIgnoreCase(this.prepositions[j])){
                        test = false;
                        break;
                    }
                }
            }
            if(test){
                for(int j=0; j<this.nConjunctions; j++){
                    if(tt.equalsIgnoreCase(this.conjunctions[j])){
                        test = false;
                        break;
                    }
                }
            }
            if(test)al.add(i);
        }
        
        this.nPrincipalTokens = al.size();
        this.principalTokens = new String[this.nPrincipalTokens];
        this.principalTokenInitialIndices = new int[this.nPrincipalTokens];
        this.principalTokenFinalIndices = new int[this.nPrincipalTokens];
        for(int i=0; i<this.nPrincipalTokens; i++){
            int j = al.get(i);
            this.principalTokens[i] = this.tokens[j];
            this.principalTokenInitialIndices[i] = this.tokenInitialIndices[j];
            this.principalTokenFinalIndices[i] = this.tokenFinalIndices[j];
        }
        
        return this.principalTokens;
    }
    
    // Split the string into principal tokens ignoring obligatory  initial principal token
    // static method
    public static String[] principalTokensZero(String ss){
        Strings strs = new Strings(ss);
        return strs.principalTokensZero();
    }
    
    
    // Removes white spaces
    // instance method
    public String removeWhiteSpaces(){
        String ret = "";
        for(int i=0; i<this.nEdited; i++){
            char cc = this.editedString.charAt(i);
            if(!Character.isWhitespace(cc)){
                ret += cc;
            }
        } 
        return ret.trim();
    }
    
    // Removes white spaves 
    // static method
    public static String removeWhiteSpaces(String line){
        Strings ss = new Strings(line);
        return ss.removeWhiteSpaces();
    }
    
    // Removes all accents and replaces the accented letter with its non-accented equivalent 
    // instance method
    public String removeAccents(){
        
        String ss = this.editedString;
        
        // Remove html names and replace with unaccented equivalent
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesA[i].length();
                    String hold0 = ss.substring(0,pos0) + this.replSymbolsA[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers and replace with unaccented equivalent
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersA[i].length();
                    String hold0 = ss.substring(0,pos0) + this.replSymbolsA[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalent with unaccented equivalent
        for(int i=0; i<this.nSymbolsA; i++){
            for(int j=0; j<this.nEdited; j++){
               if((int)ss.charAt(j)==this.decNumbersA[i]){
                   ss = ss.substring(0,j) + this.replSymbolsA[i] + ss.substring(j+1);
                   this.nEdited = ss.length();
               }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Removes all accents and replaces the accented letter with its non-accented equivalent 
    // static method
    public static String removeAccents(String ss){
        Strings strs = new Strings(ss);
        return strs.removeAccents();
    }
    
    // Convert all quotation mark to html number representation 
    // instance method
    public String quotationMarksToHtmlNumbers(){
        
        String ss = this.editedString;
        
        // Convert html names to html numbers
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0) + this.htmlNumbersQ[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<this.nEdited; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsQ; j++){
                if(cc==this.decNumbersQ[j]){
                    String hold1 = ss.substring(0,i) + this.htmlNumbersQ[j];
                    if(i+1<this.nEdited)hold1 += ss.substring(i+1);
                    ss = hold1;
                    this.nEdited = ss.length();
                    break;
                }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Convert all quotation mark to html number representation 
    // static method
    public static String quotationMarksToHtmlNumbers(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksToHtmlNumbers();
    }
    
    // Convert all accents to html number representation 
    // instance method
    public String accentsToHtmlNumbers(){
        
        String ss = this.editedString;
        
        // Convert html names to html numbers
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesA[i].length();
                    String hold0 = ss.substring(0,pos0) + this.htmlNumbersA[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<this.nEdited; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsA; j++){
                if(cc==this.decNumbersA[j]){
                    String hold1 = ss.substring(0,i) + this.htmlNumbersA[j];
                    if(i+1<this.nEdited)hold1 += ss.substring(i+1);
                    ss = hold1;
                    this.nEdited = ss.length();
                    break;
                }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Convert all quotation mark to html number representation 
    // static method
    public static String accentsToHtmlNumbers(String ss){
        Strings strs = new Strings(ss);
        return strs.accentsToHtmlNumbers();
    }
    
    // Convert all dashes to html number representation 
    // instance method
    public String dashesToHtmlNumbers(){
        
        String ss = this.editedString;
        
        // Convert html names to html numbers
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0) + this.htmlNumbersD[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<this.nEdited; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsD; j++){
                if(cc==this.decNumbersD[j]){
                    String hold1 = ss.substring(0,i) + this.htmlNumbersD[j];
                    if(i+1<this.nEdited)hold1 += ss.substring(i+1);
                    ss = hold1;
                    this.nEdited = ss.length();
                    break;
                }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Convert all dashes to html number representation 
    // static method
    public static String dashesToHtmlNumbers(String ss){
        Strings strs = new Strings(ss);
        return strs.dashesToHtmlNumbers();
    }
    
    
    // Convert all accent html name and html number representations to decimal integer representation 
    // instance method
    public String accentsToDec(){
        
        String ss = this.editedString;
        
        // Convert html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesA[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersA[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert html numbers 
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersA[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersA[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Convert all quotation mark html name and html number representations to decimal integer representation 
    // static method
    public static String accentsToDec(String ss){
        Strings strs = new Strings(ss);
        return strs.accentsToDec();
    }
    
    // Convert all dash html name and html number representations to decimal integer representation 
    // instance method
    public String dashesToDec(){
        
        String ss = this.editedString;
        
        // Convert html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersD[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert html numbers 
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersD[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersD[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Convert all dash html name and html number representations to decimal integer representation 
    // static method
    public static String dashesToDec(String ss){
        Strings strs = new Strings(ss);
        return strs.dashesToDec();
    }
    
    // Convert all quotation mark html name and html number representations to decimal integer representation 
    // instance method
    public String quotationMarksToDec(){
        
        String ss = this.editedString;
        
        // Convert html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersQ[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersQ[i];
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Convert all quotation mark html name and html number representations to decimal integer representation 
    // static method
    public static String quotationMarksToDec(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksToDec();
    }
    
    // Convert all quotation mark integer dec representations to html name representation 
    // instance method
    public String quotationMarksDecToName(){
              
        // Convert to html names 
        for(int i=0; i<this.nEdited;i++){
            int cc = (int)this.editedString.charAt(i);
            for(int j=0; j<this.nSymbolsQ; j++){
                if(cc==this.decNumbersQ[j]){
                    String hold0 = this.editedString.substring(0,i) + this.htmlNamesQ[j];
                    if(i<this.nEdited-1)hold0 += this.editedString.substring(i+1);
                    this.editedString = hold0;
                    this.nEdited = this.editedString.length();
                    break;
                }
            }   
        }
        return this.editedString;
    }
    
    // Convert all quotation mark integer dec representations to html name representation 
    // static method
    public static String quotationMarksDecToName(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksDecToName();
    }
    
    // Convert all quotation mark integer dec representations to html number representation 
    // instance method
    public String quotationMarksDecToHtmlNumber(){
              
        // Convert to html names 
        for(int i=0; i<this.nEdited;i++){
            int cc = (int)this.editedString.charAt(i);
            for(int j=0; j<this.nSymbolsQ; j++){
                if(cc==this.decNumbersQ[j]){
                    String hold0 = this.editedString.substring(0,i) + this.htmlNumbersQ[j];
                    if(i<this.nEdited-1)hold0 += this.editedString.substring(i+1);
                    this.editedString = hold0;
                    this.nEdited = this.editedString.length();
                    break;
                }
            }   
        }
        return this.editedString;
    }
    
    // Convert all quotation mark integer dec representations to html name representation 
    // static method
    public static String quotationMarksDecToHtmlNumber(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksDecToHtmlNumber();
    }
    
    // Replace dashes with spaces
    // instance method
    public String replaceDashesWithSpaces(){
        
        String ss = this.editedString;
        
        // Replace html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0) + " ";
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace html numbers 
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersD[i].length();
                    String hold0 = ss.substring(0,pos0) + " ";
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsD; i++){
            for(int j=0; j<this.nEdited; j++){
               if((int)ss.charAt(j)==this.decNumbersD[i]){
                   ss = ss.substring(0,j) + " " + ss.substring(j+1);
                   this.nEdited = ss.length();
               }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Replace dashes with spaces 
    // static method
    public static String replaceDashesWithSpaces(String ss){
        Strings strs = new Strings(ss);
        return strs.replaceDashesWithSpaces();
    }

    
    // Removes all quotation marks
    // instance method
    public String removeQuotationMarks(){
        
        String ss = this.editedString;
        
        // Remove html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsQ; i++){
            for(int j=0; j<this.nEdited; j++){
               if((int)ss.charAt(j)==this.decNumbersQ[i]){
                   ss = ss.substring(0,j) + ss.substring(j+1);
                   this.nEdited = ss.length();
               }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Removes all quotation marks 
    // static method
    public static String removeQuotationMarks(String ss){
        Strings strs = new Strings(ss);
        return strs.removeQuotationMarks();
    }
    
    // Replace all quotation marks by spaces
    // instance method
    public String replaceQuotationMarksBySpaces(){
        
        String ss = this.editedString;
        
        // Replace html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<this.nEdited)hold0 += (" " + ss.substring(pos1));
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<this.nEdited)hold0 += (" " + ss.substring(pos1));
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsQ; i++){
            for(int j=0; j<this.nEdited; j++){
               if((int)ss.charAt(j)==this.decNumbersQ[i]){
                   ss = ss.substring(0,j) + " " + ss.substring(j+1);
                   this.nEdited = ss.length();
               }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Replace all quotation marks by spaces
    // instance method
    public String replaceQuotationMarksWithSpaces(){
        
        String ss = this.editedString;
        
        // Remove html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0) + " ";
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0) + " ";
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsQ; i++){
            for(int j=0; j<this.nEdited; j++){
               if((int)ss.charAt(j)==this.decNumbersQ[i]){
                   ss = ss.substring(0,j) + " " + ss.substring(j+1);
                   this.nEdited = ss.length();
               }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Replace all quotation marks by spaces
    // static method
    public static String replaceQuotationMarksWithSpaces(String ss){
        Strings strs = new Strings(ss);
        return strs.replaceQuotationMarksWithSpaces();
    }
    
    // Removes all quotation marks and 's
    // instance method
    public String removeQuotationMarksPlus(){
        
        String ss = this.editedString;
        
        // Remove html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    if(pos1<this.nEdited && pos0>0){
                        char ccc = ss.charAt(pos0-1); 
                        if(!Character.isWhitespace(ccc)){
                            if(ss.charAt(pos1)=='s' || ss.charAt(pos1)=='S')pos1++;
                        }
                    }
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    if(pos1<this.nEdited && pos0>0){
                        char ccc = ss.charAt(pos0-1); 
                        if(!Character.isWhitespace(ccc)){
                            if(ss.charAt(pos1)=='s' || ss.charAt(pos1)=='S')pos1++;
                        }
                    }
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsQ; i++){
            for(int j=0; j<this.nEdited; j++){
                pos1 = j+1;
                if(pos1<this.nEdited && j>0){
                    char ccc = ss.charAt(j-1); 
                    if(!Character.isWhitespace(ccc)){
                        if(ss.charAt(pos1)=='s' || ss.charAt(pos1)=='S')pos1++;
                    }
                }
                if((int)ss.charAt(j)==this.decNumbersQ[i]){
                    ss = ss.substring(0,j) + ss.substring(pos1);
                    this.nEdited = ss.length();
                }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Removes all quotation marks plus's
    // static method
    public static String removeQuotationMarksPlus(String ss){
        Strings strs = new Strings(ss);
        return strs.removeQuotationMarksPlus();
    }
    
        // Removes all quotation marks
    // instance method
    public String removeDashes(){
       
        String ss = this.editedString;
        
        // Remove html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersD[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<this.nEdited)hold0 += ss.substring(pos1);
                    ss = hold0;
                    this.nEdited = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsD; i++){
            for(int j=0; j<this.nEdited; j++){
               if((int)ss.charAt(j)==this.decNumbersD[i]){
                   ss = ss.substring(0,j) + ss.substring(j+1);
                   this.nEdited = ss.length();
               }
            }
        }
        
        this.editedString = ss;
        return ss;
    }
    
    // Removes all dashes 
    // static method
    public static String removeDashes(String ss){
        Strings strs = new Strings(ss);
        return strs.removeDashes();
    }
    
    // Display integer character equivalents
    // instance method
    public int[] displayIntegers(){
        int[] ints = new int[this.nEdited];
        for(int i=0; i<this.nEdited; i++)ints[i] = (int)this.editedString.charAt(i);
        PrintToScreen.print(ints);
        
        return ints; 
    }
    
    // Display integer character equivalents
    // static method
    public static int[] displayIntegers(String ss){
        Strings strs = new Strings(ss);
        return strs.displayIntegers();
    }
    
    // Return index of the first occurence of a dash
    // instance method
    public int indexOfDash(){
        int ii = -1;
        int iis = -1;
        
        boolean found = false;
        for(int i=0; i<this.nEdited; i++){
            String cc = this.editedString.substring(i,i+1);
            for(int j=0; j<this.nSymbolsD; j++){
                if(cc.equals(symbolsD[j])){
                    ii = i;
                    iis = i;
                    found = true;
                    break;
                }
            }
            if(found)break;
        }
        
        String[] namesD = {"&ndash;", "&mdash;", "&oline;"};
        int pos0 = -1;
        int iiss = -1;
        for(int j=0; j<3; j++){
           pos0 = this.editedString.indexOf(namesD[j]);
           if(pos0!=-1){
               ii = pos0;
               if(iiss==-1){
                   iiss = pos0; 
               }
               else{
                   if(iiss<pos0){
                       ii = iiss;
                   }
                   else{
                       iiss = pos0;
                   }
               }
            }
        }  
        if(iis!=-1){
            if(iiss!=-1){
                ii = Math.min(iis, iiss);
                iis = ii;
            }
            else{
                // no action
            }
        }
        else{
            if(iiss!=-1){
                ii = iiss;
                iis = iiss;
            }
            else{
                // no action 
            }           
        }
        
        pos0 = -1;
        iiss = -1;
        for(int j=0; j<this.nSymbolsD; j++){
           pos0 = this.editedString.indexOf(htmlNumbersD[j]);
           if(pos0!=-1){
               ii = pos0;
               if(iiss==-1){
                   iiss = pos0; 
               }
               else{
                   if(iiss<pos0){
                       ii = iiss;
                   }
                   else{
                       iiss = pos0;
                   }
               }
            }
        }  
        if(iis!=-1){
            if(iiss!=-1){
                ii = Math.min(iis, iiss);
                iis = ii;
            }
            else{
                // no action
            }
        }
        else{
            if(iiss!=-1){
                ii = iiss;
                iis = iiss;
            }
            else{
                // no action 
            }           
        }

        return ii;
    }
    
    // Return index of the first occurence of a dash
    // static method
    public static int indexOfDash(String ss){
        Strings strs = new Strings(ss);
        return strs.indexOfDash();
    }
    
    // Returns true if a fraction (arguent identicalFraction) of the shorter string is common to both strings
    // instance method
    public boolean nearlyEquals(String ss1, double identicalFraction){
       
        boolean ret = false;
        String[] ss = new String[2];
        ss[0] = this.editedString;
        ss[1] = ss1;
        if(ss[0].equalsIgnoreCase(ss[1])){
            ret = true;
        }
        else{
            int n0 = ss[0].length();
            int n1 = ss[1].length();
            int nMin = Math.min(n0,n1);
            int nMax = Math.max(n0,n1);
            int minIndex = 0;
            int maxIndex = 1;
            if(n0==nMax){
                minIndex = 1;
                maxIndex = 0;
            }
            int nLimit = (int)Math.round(identicalFraction*nMin);
            int nExcess = nMax - nMin;
            for(int i=0; i<nExcess+1; i++){
                int nSame = 0;
                for(int j=0; j<nMin; j++){
                    if(ss[minIndex].charAt(j)==ss[maxIndex].charAt(j+i))nSame++;
                }
                if(nSame>=nLimit){
                    ret = true;
                    break;
                }
            }
        }
        return ret;
    }
    
    // Returns true if a continuous fraction of the shorter string is common to both strings
    // The length of the putative common string is the length of the shorter string times identicalFraction rounded to the nearest integer 
    // static  method
    public static boolean nearlyEquals(String ss0, String ss1, double identicalFraction){
        Strings ss = new Strings(ss1);
        return ss.nearlyEquals(ss0, identicalFraction);
    }
    
    
}

    