/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 ******************************************************************************/
package plugins.big.shapedesigner.splinecurve;

import icy.sequence.Sequence;
import icy.util.XMLUtil;

import java.awt.Color;
import java.awt.Polygon;
import java.awt.geom.Point2D;

import org.w3c.dom.Element;

import plugins.big.bigsnakeutils.icy.snake2D.Snake2D;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DScale;
import plugins.big.bigsnakeutils.process.process1D.BSplineBasis;
import plugins.big.bigsnakeutils.process.process1D.BSplineBasis.BSplineBasisType;
import plugins.big.bigsnakeutils.process.process1D.Filters;
import plugins.big.bigsnakeutils.shape.utils.Geometry2D;
import plugins.kernel.roi.roi2d.ROI2DEllipse;
import plugins.kernel.roi.roi2d.ROI2DPolygon;
import plugins.kernel.roi.roi2d.ROI2DRectangle;

/**
 * Spline curve.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class SplineCurve implements Snake2D {

	/** Spline curve defining nodes. */
	private Snake2DNode[] coef_ = null;

	// ----------------------------------------------------------------------------
	// SPLINE CURVE CONTOUR

	/** Samples of the x coordinates of the contour. */
	private double[] xSplineCurveContour_ = null;
	/** Samples of the y coordinates of the contour. */
	private double[] ySplineCurveContour_ = null;

	/** Number of samples at which each segment of the contour is discretized. */
	private int nSamplesPerSegment_ = 500;

	// ----------------------------------------------------------------------------
	// SPLINE CURVE STATUS FIELDS

	/** Signed area of the region enclosed by the spline curve. */
	private double area_ = 0;
	/**
	 * If <code>true</code>, the spline curve has gone through the
	 * initialization process.
	 */
	protected boolean isInitialized_ = false;

	// ----------------------------------------------------------------------------
	// SPLINE CURE OPTION FIELDS

	/** Number of spline vector coefficients. */
	private int M_ = 0;

	/** Initial contour. */
	private Polygon initialContour_ = null;

	// ----------------------------------------------------------------------------
	// SPLINE LUTS

	/** Basis function of the curve. */
	private static final BSplineBasis.BSplineBasisType BASIS_FUNCTION = BSplineBasisType.ESPLINE3;
	/** LUT with the samples of the B-spline basis function. */
	private double[] bSplineLUT_ = null;
	/** LUT with the samples of the autocorrelation function. */
	private double[] bSplineAutocorrelationLUT_ = null;
	/**
	 * LUT with the values of
	 * $Q(i,j)=\int_{0}^{M}\,\phi_M(t-j)\,\phi'_M(t-i)\,\mathrm{d}t$.
	 */
	private double[][] qLUT_ = null;

	// ----------------------------------------------------------------------------
	// IMAGE FIELDS

	/** Image to which the curve is attached to. */
	private Sequence image_ = null;

	// ----------------------------------------------------------------------------
	// DISPLAY

	/** Color of the scale that represents the contour of the spline curve. */
	private final Color SPLINE_CURVE_CONTOUR_COLOR = Color.RED;
	/**
	 * Color of the scale that represents the control polygon of the spline
	 * curve.
	 */
	private final Color SPLINE_CURVE_POLYGON_COLOR = new Color(255, 255, 0, 128);

	// ----------------------------------------------------------------------------
	// AUXILIARY FIELDS AND LUTS

	/** PI/M. */
	private double PIM_ = 0;
	/** 2*PI/M. */
	private double PI2M_ = 0;
	/**
	 * Support of the basis function multiplied by
	 * <code>nSamplesPerSegment_</code>.
	 */
	private int NR_ = 0;
	/** Number of control points multiplied by <code>nSamplesPerSegment_</code>. */
	private int MR_ = 0;

	/** LUT with samples of one period of a sine. */
	private double[] sinLUT_ = null;
	/** LUT with samples of one period of a cosine. */
	private double[] cosLUT_ = null;

	// ----------------------------------------------------------------------------
	// XML TAGS

	/**
	 * Label of the XML tag containing the list of snake-defining control
	 * points.
	 */
	public static final String ID_CONTROL_POINTS = "control_points";
	/**
	 * Label of the XML tag containing the a single snake-defining control
	 * point.
	 */
	public static final String ID_CONTROL_POINT = "control_point";

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public SplineCurve(Sequence seq, SplineCurveParameters parameters,
			Object initialContour) {
		if (seq == null) {
			System.err.println("Image not properly loaded.");
			return;
		}
		image_ = seq;
		if (parameters == null) {
			System.err.println("Spline curve parameters not properly loaded.");
			return;
		}
		M_ = parameters.getM();
		if (M_ < 3) {
			System.err
					.println("The minimum number of knots for this basis function is three.");
			return;
		}
		nSamplesPerSegment_ = (int) Math.ceil((double) nSamplesPerSegment_
				/ (double) M_);

		if (initialContour instanceof Polygon) {
			initialContour_ = (Polygon) initialContour;
		} else if (initialContour instanceof ROI2DPolygon) {
			initialContour_ = ((ROI2DPolygon) initialContour).getPolygon();
		} else if (initialContour instanceof ROI2DEllipse) {
			double x = ((ROI2DEllipse) initialContour).getEllipse()
					.getCenterX();
			double y = ((ROI2DEllipse) initialContour).getEllipse()
					.getCenterY();
			double h = ((ROI2DEllipse) initialContour).getEllipse().getHeight();
			double w = ((ROI2DEllipse) initialContour).getEllipse().getWidth();

			Polygon p = new Polygon();
			for (int i = 0; i < 100; i++) {
				p.addPoint((int) (x + w * Math.cos(2.0 * Math.PI * i / 100.0)
						/ 2.0),
						(int) (y + h * Math.sin(2.0 * Math.PI * i / 100.0)
								/ 2.0));
			}
			initialContour_ = p;
		} else if (initialContour instanceof ROI2DRectangle) {
			Polygon p = new Polygon();
			p.addPoint((int) ((ROI2DRectangle) initialContour).getBounds()
					.getMinX(), (int) ((ROI2DRectangle) initialContour)
					.getBounds().getMinY());
			p.addPoint((int) ((ROI2DRectangle) initialContour).getBounds()
					.getMinX(), (int) ((ROI2DRectangle) initialContour)
					.getBounds().getMaxY());
			p.addPoint((int) ((ROI2DRectangle) initialContour).getBounds()
					.getMaxX(), (int) ((ROI2DRectangle) initialContour)
					.getBounds().getMaxY());
			p.addPoint((int) ((ROI2DRectangle) initialContour).getBounds()
					.getMaxX(), (int) ((ROI2DRectangle) initialContour)
					.getBounds().getMinY());
		} else {
			initialContour_ = null;
		}

		initialize(true);
	}

	// ----------------------------------------------------------------------------
	// SNAKE2D METHODS

	/**
	 * The purpose of this method is to compute the energy of the snake. Since
	 * the spline curve is not supposed to be optimized, the returned value is
	 * always zero.
	 */
	@Override
	public double energy() {
		return 0;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a point with the position of the center of gravity of the scales.
	 */
	@Override
	public Point2D.Double getCentroid() {
		Point2D.Double centroid = new Point2D.Double();
		for (Snake2DNode element : coef_) {
			centroid.x += element.x;
			centroid.y += element.y;
		}
		centroid.x /= coef_.length;
		centroid.y /= coef_.length;
		return centroid;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to compute the gradient of the snake energy
	 * with respect to the snake-defining nodes. Since the spline curve is not
	 * supposed to be optimized, the returned value is always <code>null</code>.
	 */
	@Override
	public Point2D.Double[] getEnergyGradient() {
		return null;
	}

	// ----------------------------------------------------------------------------

	/**
	 * This method provides an accessor to the snake-defining nodes.
	 */
	@Override
	public Snake2DNode[] getNodes() {
		return coef_;
	}

	// ----------------------------------------------------------------------------

	/** This method returns the quantity of snake-defining nodes. */
	@Override
	public int getNumNodes() {
		return coef_.length;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to determine what to draw on screen, given
	 * the current configuration of nodes. Collectively, the array of scales
	 * forms the skin of the snake.
	 */
	@Override
	public Snake2DScale[] getScales() {
		int numScales = getNumScales();
		Snake2DScale[] skin = new Snake2DScale[numScales];

		double[] xpoints = new double[M_];
		double[] ypoints = new double[M_];
		for (int k = 0; k < M_; k++) {
			xpoints[k] = coef_[k].x;
			ypoints[k] = coef_[k].y;
		}
		skin[0] = new Snake2DScale(xSplineCurveContour_, ySplineCurveContour_,
				MR_, SPLINE_CURVE_CONTOUR_COLOR, true);

		skin[1] = new Snake2DScale(xpoints, ypoints, M_,
				SPLINE_CURVE_POLYGON_COLOR, true);

		return skin;
	}

	// ----------------------------------------------------------------------------

	@Override
	public void initialize(boolean initShape) {
		switch (BASIS_FUNCTION) {
		case ESPLINE3:
			NR_ = BSplineBasis.ESPLINE3SUPPORT * nSamplesPerSegment_;
			break;
		case ESPLINE4:
			NR_ = BSplineBasis.ESPLINE4SUPPORT * nSamplesPerSegment_;
			break;
		case LINEARBSPLINE:
			NR_ = BSplineBasis.LINEARBSPLINESUPPORT * nSamplesPerSegment_;
			break;
		case QUADRATICBSPLINE:
			NR_ = BSplineBasis.QUADRATICBSPLINESUPPORT * nSamplesPerSegment_;
			break;
		case CUBICBSPLINE:
			NR_ = BSplineBasis.CUBICBSPLINESUPPORT * nSamplesPerSegment_;
			break;
		case MSPLINE:
			NR_ = BSplineBasis.MSPLINESUPPORT * nSamplesPerSegment_;
			break;
		}

		MR_ = M_ * nSamplesPerSegment_;
		PIM_ = Math.PI / M_;
		PI2M_ = 2 * PIM_;

		xSplineCurveContour_ = new double[MR_];
		ySplineCurveContour_ = new double[MR_];

		qLUT_ = new double[M_][M_];

		buildLUTs();
		if (initShape) {
			initializeDefaultShape();
		}
		updateSplineCurveSkin();
		updateArea();
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to monitor the status of the snake.
	 */
	@Override
	public boolean isAlive() {
		return true;
	}

	// ----------------------------------------------------------------------------

	/** Returns <code>true</code> if the snake has been initialized. */
	@Override
	public boolean isInitialized() {
		return isInitialized_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Sets the status of the snake to alive, and restores the maximum number
	 * iterations to the original one.
	 */
	@Override
	public void reviveSnake() {
	}

	// ----------------------------------------------------------------------------

	/** Saves the curve-defining parameters in an XML file. */
	@Override
	public void saveToXML(Element node) {
		getSplineCurveParameters().saveToXML(node);
		Element controlPointsElement = XMLUtil.addElement(node,
				ID_CONTROL_POINTS);
		for (Snake2DNode controlPoint : coef_) {
			Element controlPointElement = XMLUtil.addElement(
					controlPointsElement, ID_CONTROL_POINT);
			controlPoint.saveToXML(controlPointElement);
		}
	}

	// ----------------------------------------------------------------------------

	/** This method provides a mutator to the curve-defining nodes. */
	@Override
	public void setNodes(Snake2DNode[] node) {
		for (int i = 0; i < M_; i++) {
			coef_[i].x = node[i].x;
			coef_[i].y = node[i].y;
		}

		updateSplineCurveSkin();
		updateArea();
	}

	// ----------------------------------------------------------------------------
	// SPLINECURVE METHODS

	/** Retrieves the area under the curve determined by the spline curve. */
	public double getArea() {
		return Math.abs(area_);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the number of scales provided by the method
	 * <code>getScales()</code>.
	 */
	public int getNumScales() {
		return 2;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a new container with the information of the execution parameters
	 * of the spline curve.
	 */
	public SplineCurveParameters getSplineCurveParameters() {
		return new SplineCurveParameters(M_);
	}

	// ----------------------------------------------------------------------------

	/** Sets the parameters of the spline curve. */
	public void setSplineCurveParameters(SplineCurveParameters parameters) {
		int M = parameters.getM();
		if (M != M_) {
			setNumNodes(1.5, M);
		}
	}

	// ============================================================================
	// PRIVATE METHODS

	/** Initializes all look-up tables of the class. */
	private void buildLUTs() {
		double currentVal;
		bSplineLUT_ = new double[NR_];
		sinLUT_ = new double[MR_];
		cosLUT_ = new double[MR_];

		double PI2MR = PI2M_ / nSamplesPerSegment_;
		for (int i = 0; i < MR_; i++) {
			sinLUT_[i] = Math.sin(PI2MR * i + 1.5);
			cosLUT_[i] = Math.cos(PI2MR * i + 1.5);
		}

		for (int i = 0; i < NR_; i++) {
			currentVal = (double) i / (double) nSamplesPerSegment_;
			switch (BASIS_FUNCTION) {
			case ESPLINE3:
				bSplineLUT_[i] = BSplineBasis.ESpline3(currentVal, PI2M_);
				break;
			case ESPLINE4:
				bSplineLUT_[i] = BSplineBasis.ESpline4(currentVal, PI2M_);
				break;
			case LINEARBSPLINE:
				bSplineLUT_[i] = BSplineBasis.LinearBSpline(currentVal);
				break;
			case QUADRATICBSPLINE:
				bSplineLUT_[i] = BSplineBasis.QuadraticSpline(currentVal);
				break;
			case CUBICBSPLINE:
				bSplineLUT_[i] = BSplineBasis.CubicBSpline(currentVal);
				break;
			case MSPLINE:
				bSplineLUT_[i] = BSplineBasis.MSpline(currentVal, PI2M_);
				break;
			}
		}

		int qSize = 2 * BSplineBasis.ESPLINE3SUPPORT - 1;
		bSplineAutocorrelationLUT_ = new double[qSize];
		for (int i = 0; i < qSize; i++) {
			bSplineAutocorrelationLUT_[i] = BSplineBasis.correlationESpline(i
					- BSplineBasis.ESPLINE3SUPPORT + 1, PI2M_);
		}

		for (int i = 0; i < M_; i++) {
			for (int j = 0; j < M_; j++) {
				qLUT_[i][j] = computeQ(i, j);
			}
		}
	}

	// ----------------------------------------------------------------------------

	/** Initializes the spline curve control points with a predefined shape. */
	private void initializeDefaultShape() {
		coef_ = new Snake2DNode[M_];
		if (initialContour_ != null) {
			if (initialContour_.npoints != 0) {
				Point2D.Double[] resampledContour = Geometry2D
						.arcLengthResampling(initialContour_, M_);
				coef_ = getSplineKnots(resampledContour, M_);
				return;
			}
		}

		double xg = image_.getWidth() / 2;
		double yg = image_.getHeight() / 2;
		double rad = 0.3 * Math.min(image_.getWidth(), image_.getHeight());

		for (int i = 0; i < coef_.length; i++) {
			coef_[i] = new Snake2DNode(rad * Math.cos(PI2M_ * i) + xg, rad
					* Math.sin(PI2M_ * i) + yg);
		}

	}

	// ----------------------------------------------------------------------------

	/**
	 * Computes the points of the contour of the snake curve from the control
	 * points and stores it in a LUT.
	 */
	private void updateSplineCurveSkin() {
		int index;

		double aux, xPosVal, yPosVal;
		for (int i = 0; i < MR_; i++) {
			xPosVal = 0.0;
			yPosVal = 0.0;
			for (int k = 0; k < M_; k++) {
				index = i - k * nSamplesPerSegment_;

				while (index < 0) {
					index += MR_;
				}
				while (index >= MR_) {
					index -= MR_;
				}
				if (index >= NR_) {
					continue;
				} else {
					aux = bSplineLUT_[index];
				}
				xPosVal += coef_[k].x * aux;
				yPosVal += coef_[k].y * aux;
			}
			xSplineCurveContour_[i] = xPosVal;
			ySplineCurveContour_[i] = yPosVal;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Computes the area of the curve defined by the spline curve, the enclosing
	 * ellipse and the area ratio.
	 */
	private void updateArea() {
		double area = 0.0;
		int l_p;
		for (int k = 0; k < M_; k++) {
			int kN = k + BSplineBasis.ESPLINE3SUPPORT;
			for (int l = k - BSplineBasis.ESPLINE3SUPPORT + 1; l < kN; l++) {
				l_p = l;
				while (l_p < 0) {
					l_p += M_;
				}

				while (l_p >= M_) {
					l_p -= M_;
				}

				area += coef_[k].y * coef_[l_p].x
						* bSplineAutocorrelationLUT_[kN - l - 1];
			}
		}
		area_ = area;
	}

	// ----------------------------------------------------------------------------
	// ENERGY METHODS

	/**
	 * $\int_{0}^{M}\,\varphi_{M}(t-k_2)\,\varphi'_{M}(t-k_1)\,\mathrm{d}t$.
	 */
	private double computeQ(int k1, int k2) {
		double val = 0;
		for (int i = 0; i < MR_; i++) {

			int index1 = i - k2 * nSamplesPerSegment_;

			while (index1 < 0) {
				index1 += MR_;
			}
			while (index1 >= MR_) {
				index1 -= MR_;
			}
			int index2 = i - k1 * nSamplesPerSegment_;

			while (index2 < 0) {
				index2 += MR_;
			}
			while (index2 >= MR_) {
				index2 -= MR_;
			}
			val += BSplineBasis.ESpline3((double) index1
					/ (double) nSamplesPerSegment_, PI2M_)
					* BSplineBasis.ESpline3_Prime((double) index2
							/ (double) nSamplesPerSegment_, PI2M_);
		}
		val /= nSamplesPerSegment_;
		return val;
	}

	// ----------------------------------------------------------------------------

	// ----------------------------------------------------------------------------
	// RESAMPLING METHODS

	private Snake2DNode[] getSplineKnots(Point2D.Double[] contour, int M) {
		double[] knotsX = new double[M];
		double[] knotsY = new double[M];

		double b = 2.0 / 3.0;

		for (int i = 0; i < M; i++) {
			knotsX[i] = contour[i].x;
			knotsY[i] = contour[i].y;
		}

		double[] pole = { (-b + Math.sqrt(2 * b - 1)) / (1 - b) };
		knotsX = Filters.prescaledPeriodic(knotsX, pole);
		knotsY = Filters.prescaledPeriodic(knotsY, pole);

		Snake2DNode[] newCoeff = new Snake2DNode[M];
		for (int i = 0; i < M; i++) {
			newCoeff[i] = new Snake2DNode(knotsX[i], knotsY[i]);
		}

		return newCoeff;
	}

	// ----------------------------------------------------------------------------

	/** Updates the number of control points. */
	private void setNumNodes(double initialSample, int newM) {
		double c = BSplineBasis.correlationOfTwoESpline(3.0 / newM,
				(2 * Math.PI) / newM, (2 * Math.PI) / newM, newM, newM);
		double b = BSplineBasis.correlationOfTwoESpline(4.0 / newM,
				(2 * Math.PI) / newM, (2 * Math.PI) / newM, newM, newM);
		double a = BSplineBasis.correlationOfTwoESpline(5.0 / newM,
				(2 * Math.PI) / newM, (2 * Math.PI) / newM, newM, newM);

		double alpha1 = (b + Math.sqrt(b * b - 4 * a * c + 8 * a * a))
				/ (2 * a);
		double alpha2 = (b - Math.sqrt(b * b - 4 * a * c + 8 * a * a))
				/ (2 * a);

		double z1 = (-alpha1 + Math.sqrt(alpha1 * alpha1 - 4)) / 2;
		if (Math.abs(z1) > 1) {
			z1 = 1 / z1;
		}

		double z2 = (-alpha2 + Math.sqrt(alpha2 * alpha2 - 4)) / 2;
		if (Math.abs(z2) > 1) {
			z2 = 1 / z2;
		}

		double[] invMatrix = new double[newM];
		double cz1 = 1 / ((1 - Math.pow(z1, newM)) * (z1 - 1 / z1) * a * (z1
				+ 1 / z1 - z2 - 1 / z2));
		double cz1i = -1
				/ ((1 - Math.pow(z1, -newM)) * (z1 - 1 / z1) * a * (z1 + 1 / z1
						- z2 - 1 / z2));
		double cz2 = -1
				/ ((1 - Math.pow(z2, newM)) * (z2 - 1 / z2) * a * (z1 + 1 / z1
						- z2 - 1 / z2));
		double cz2i = 1 / ((1 - Math.pow(z2, -newM)) * (z2 - 1 / z2) * a * (z1
				+ 1 / z1 - z2 - 1 / z2));
		double z1k = 1;
		double z2k = 1;
		for (int k = 0; k < newM; k++) {
			invMatrix[k] = cz1 * z1k + cz1i * (1 / z1k) + cz2 * z2k + cz2i
					* (1 / z2k);
			z1k *= z1;
			z2k *= z2;
		}

		Point2D.Double[] vec = new Point2D.Double[newM];
		double x, y;
		for (int k = 0; k < newM; k++) {
			x = 0.0;
			y = 0.0;
			for (int i = -BSplineBasis.ESPLINE3SUPPORT; i < M_
					+ BSplineBasis.ESPLINE3SUPPORT; i++) {
				x += coef_[(M_ + i) % M_].x
						* BSplineBasis.correlationOfTwoESpline(
								((3.0 + i) / M_ - (double) k / (double) newM),
								(2 * Math.PI) / M_, (2 * Math.PI) / newM, M_,
								newM);
				y += coef_[(M_ + i) % M_].y
						* BSplineBasis.correlationOfTwoESpline(
								((3.0 + i) / M_ - (double) k / (double) newM),
								(2 * Math.PI) / M_, (2 * Math.PI) / newM, M_,
								newM);
			}
			vec[k] = new Point2D.Double(x, y);
		}

		Snake2DNode[] newCoef = new Snake2DNode[newM];
		for (int k = 0; k < newM; k++) {
			x = 0.0;
			y = 0.0;
			for (int i = 0; i < newM; i++) {
				x += vec[i].x * invMatrix[(newM + i - k) % newM];
				y += vec[i].y * invMatrix[(newM + i - k) % newM];
			}
			newCoef[k] = new Snake2DNode(x, y);
		}

		coef_ = newCoef;
		M_ = newM;
		initialize(false);
	}
}
