package plugins.stef.roi.bloc.convert;

import icy.image.IcyBufferedImage;
import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginBundled;
import icy.plugin.interface_.PluginLibrary;
import icy.roi.ROI;
import icy.roi.ROIUtil;
import icy.sequence.Sequence;
import icy.sequence.SequenceDataIterator;
import icy.type.DataIteratorUtil;
import icy.type.DataType;
import icy.type.collection.CollectionUtil;
import icy.type.rectangle.Rectangle5D;
import icy.util.ShapeUtil.BooleanOperator;

import java.util.ArrayList;
import java.util.List;

import plugins.adufour.blocks.tools.roi.ROIBlock;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.lang.VarDouble;
import plugins.adufour.vars.lang.VarEnum;
import plugins.adufour.vars.lang.VarInteger;
import plugins.adufour.vars.lang.VarROIArray;
import plugins.adufour.vars.lang.VarSequence;
import plugins.stef.roi.bloc.RoiBlocks;

/**
 * Block (Protocols) to convert a ROI (Region Of Interest) into a binary image / sequence.<br>
 * <br>
 * It takes a ROI (or a list of ROI) as input and generate a binary sequence from it.<br/>
 * The content of the ROI area is filled with the specified value while background remains at 0.<br/>
 * Size informations are used to define the dimension of the output sequence, a value of 0 indicate
 * to use the ROI dimension instead.
 * 
 * @author Stephane
 */
public class RoiToSequence extends Plugin implements ROIBlock, PluginLibrary, PluginBundled
{
    final VarROIArray inputROI;
    final VarInteger sizeX;
    final VarInteger sizeY;
    final VarInteger sizeC;
    final VarInteger sizeZ;
    final VarInteger sizeT;
    final VarEnum<DataType> dataType;
    final VarDouble fillValue;
    final VarSequence varOut;

    public RoiToSequence()
    {
        super();

        inputROI = new VarROIArray("ROI(s)");
        sizeX = new VarInteger("Size X", 0);
        sizeY = new VarInteger("Size Y", 0);
        sizeC = new VarInteger("Size C", 1);
        sizeZ = new VarInteger("Size Z", 1);
        sizeT = new VarInteger("Size T", 1);
        dataType = new VarEnum<DataType>("Data type", DataType.UBYTE);
        fillValue = new VarDouble("Fill value", 255.0);
        varOut = new VarSequence("Sequence", null);
    }

    @Override
    public void declareInput(VarList inputMap)
    {
        inputMap.add("rois", inputROI);
        inputMap.add("sizeX", sizeX);
        inputMap.add("sizeY", sizeY);
        inputMap.add("sizeC", sizeC);
        inputMap.add("sizeZ", sizeZ);
        inputMap.add("sizeT", sizeT);
        inputMap.add("dataType", dataType);
        inputMap.add("fillValue", fillValue);
    }

    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("Sequence", varOut);
    }

    @Override
    public void run()
    {
        final List<ROI> rois = new ArrayList<ROI>();
        final Rectangle5D bounds = new Rectangle5D.Double();

        try
        {
            // compute the union of all ROI
            final ROI roi = ROIUtil.merge(CollectionUtil.asList(inputROI.getValue()), BooleanOperator.OR);
            // get bounds of result
            bounds.add(roi.getBounds5D());
            // add this single ROI to list
            rois.add(roi);
        }
        catch (Exception e)
        {
            for (ROI roi : inputROI.getValue())
            {
                // compute global bounds
                if (roi != null)
                {
                    bounds.add(roi.getBounds5D());
                    rois.add(roi);
                }
            }
        }

        int sX = sizeX.getValue().intValue();
        int sY = sizeY.getValue().intValue();
        int sC = sizeC.getValue().intValue();
        int sZ = sizeZ.getValue().intValue();
        int sT = sizeT.getValue().intValue();

        if (sX == 0)
            sX = (int) bounds.getSizeX();
        if (sY == 0)
            sY = (int) bounds.getSizeY();
        if (sC == 0)
            sC = (bounds.isInfiniteC() ? 1 : (int) bounds.getSizeC());
        if (sZ == 0)
            sZ = (bounds.isInfiniteZ() ? 1 : (int) bounds.getSizeZ());
        if (sT == 0)
            sT = (bounds.isInfiniteT() ? 1 : (int) bounds.getSizeT());

        // empty base dimension and empty result --> generate a empty 320x240 image
        if (sX == 0)
            sX = 320;
        if (sY == 0)
            sY = 240;
        if (sC == 0)
            sC = 1;
        if (sZ == 0)
            sZ = 1;
        if (sT == 0)
            sT = 1;

        final Sequence out = new Sequence("ROI conversion");
        final DataType type = dataType.getValue();

        out.beginUpdate();
        try
        {
            for (int t = 0; t < sT; t++)
                for (int z = 0; z < sZ; z++)
                    out.setImage(t, z, new IcyBufferedImage(sX, sY, sC, type));

            final double fill = fillValue.getValue().doubleValue();

            // set value from ROI(s)
            for (ROI roi : rois)
                if (!roi.getBounds5D().isEmpty())
                    DataIteratorUtil.set(new SequenceDataIterator(out, roi), fill);

            // notify data changed
            out.dataChanged();
        }
        finally
        {
            out.endUpdate();
        }

        varOut.setValue(out);
    }

    @Override
    public String getMainPluginClassName()
    {
        return RoiBlocks.class.getName();
    }
}
