/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 ******************************************************************************/
package plugins.big.steerablej;

import icy.plugin.abstract_.Plugin;
import icy.sequence.Sequence;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.lang.VarDouble;
import plugins.adufour.vars.lang.VarEnum;
import plugins.adufour.vars.lang.VarSequence;
import plugins.big.steerablej.core.AlphaValues;
import plugins.big.steerablej.core.ParameterSet;
import plugins.big.steerablej.process.Image2ArrayConverter;
import plugins.big.steerablej.process.SteerableDetector;

/**
 * Protocol block that implements 2D feature detectors. These detectors are a
 * class of steerable functions based on the optimization of a Canny-like
 * criterion. In contrast with previous computational designs, this approach is
 * truly 2D and provides filters that have closed-form expressions. It also
 * yields operators that have a better orientation selectivity than the
 * classical gradient or Hessian-based detectors.
 * 
 * @version April 24, 2013
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class FeatureDetectorBlock extends Plugin implements Block {

	/** Steerable filter detector. */
	private SteerableDetector detector_ = null;

	/** Stores all parameters used for the steerable templates. */
	private ParameterSet[][] pSet_ = null;
	/**
	 * Array of weights that control the linear combination of Gaussian
	 * derivatives which constitute the feature template.
	 */
	private AlphaValues alphaValues_ = null;
	/** Order of the feature template. */
	private int M_ = 1;
	/**
	 * Standard deviation of the Gaussian on which the steerable templates are
	 * based.
	 */
	private double sigma_ = 2.0;

	/** Enum encoding the type of detector (edge or ridge). */
	private enum detectionTypeEnum {
		EDGE, RIDGE
	}

	/** Enum encoding the quality of the detector (normal, high or highest). */
	private enum qualityEnum {
		NORMAL, HIGH, HIGHEST
	}

	// ----------------------------------------------------------------------------
	// BLOCK INPUTS

	private VarSequence inputSequenceVar_ = new VarSequence("Input Sequence",
			null);
	private VarDouble sigmaVar_ = new VarDouble("Feature width", sigma_);
	private VarEnum<detectionTypeEnum> detectionMethodVar_ = new VarEnum<detectionTypeEnum>(
			"Detector type", detectionTypeEnum.EDGE);
	private VarEnum<qualityEnum> qualityVar_ = new VarEnum<qualityEnum>(
			"Quality", qualityEnum.NORMAL);

	// ----------------------------------------------------------------------------
	// BLOCK OUTPUTS

	private VarSequence orientationMapSequence_ = new VarSequence(
			"Orientation Map Sequence", null);
	private VarSequence responseSequence_ = new VarSequence(
			"Filter Response Sequence", null);

	// ============================================================================
	// PUBLIC METHODS

	/** This will be called to run the block. */
	@Override
	public void run() {

		// Loading parameters
		alphaValues_ = new AlphaValues();
		pSet_ = ParameterSet.loadParameterSets(this.getClass()
				.getResourceAsStream("rsc/parameters.xml"));
		alphaValues_.setDefault(M_, 1, pSet_);

		// Processing input image
		Sequence focusedSequence = inputSequenceVar_.getValue();
		if (focusedSequence == null) {
			System.err.println("No image opened.");
			return;
		}

		sigma_ = sigmaVar_.getValue();

		switch (detectionMethodVar_.getValue()) {
		case EDGE:
			switch (qualityVar_.getValue()) {
			case NORMAL:
				M_ = 1;
				break;
			case HIGH:
				M_ = 3;
				break;
			case HIGHEST:
				M_ = 5;
				break;
			}
			break;
		case RIDGE:
			switch (qualityVar_.getValue()) {
			case NORMAL:
				M_ = 2;
				break;
			case HIGH:
				System.out
						.println("Combination RIDGE detection with HIGH quality not allowed.");
				System.out
						.println("Automatically swtiching to RIDGE detection with HIGHEST quality.");
				M_ = 4;
				break;
			case HIGHEST:
				M_ = 4;
				break;
			}
			break;
		}

		alphaValues_.setDefault(M_, M_ % 2 == 1 ? 1 : 2, pSet_);
		detector_ = new SteerableDetector(
				Image2ArrayConverter.seqToDoubleArray(focusedSequence, 0),
				focusedSequence.getSizeX(), focusedSequence.getSizeY(),
				focusedSequence.getSizeT(), sigma_, M_, alphaValues_.getAlpha(
						1, sigma_, M_));

		detector_.run();
		orientationMapSequence_.setValue(detector_.getOrientation());
		responseSequence_.setValue(detector_.getResponse());
	}

	// ----------------------------------------------------------------------------

	/** This code is called to create the input map. */
	@Override
	public void declareInput(VarList inputMap) {
		inputMap.add(inputSequenceVar_);
		inputMap.add(detectionMethodVar_);
		inputMap.add(qualityVar_);
		inputMap.add(sigmaVar_);
	}

	// ----------------------------------------------------------------------------

	/** This code is called to create the output map. */
	@Override
	public void declareOutput(VarList outputMap) {
		outputMap.add(orientationMapSequence_);
		outputMap.add(responseSequence_);
	}
}
