package plugins.nchenouard.particletracking;

import icy.file.FileUtil;
import icy.gui.frame.IcyFrame;
import icy.gui.frame.IcyInternalFrame;
import icy.gui.frame.progress.AnnounceFrame;
import icy.gui.frame.progress.ProgressFrame;
import icy.image.ImageUtil;
import icy.main.Icy;
import icy.plugin.abstract_.PluginActionable;
import icy.resource.ResourceUtil;
import icy.sequence.Sequence;
import icy.swimmingPool.SwimmingObject;
import icy.system.SystemUtil;
import icy.util.XMLUtil;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Vector;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.ButtonGroup;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JEditorPane;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileFilter;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackManager;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.particletracking.MHTracker.HMMMHTracker;
import plugins.nchenouard.particletracking.filtering.IMM2D;
import plugins.nchenouard.particletracking.filtering.IMM3D;
import plugins.nchenouard.particletracking.filtering.KF2dDirected;
import plugins.nchenouard.particletracking.filtering.KF2dRandomWalk;
import plugins.nchenouard.particletracking.filtering.KF3dDirected;
import plugins.nchenouard.particletracking.filtering.KF3dRandomWalk;
import plugins.nchenouard.particletracking.filtering.Predictor;
import plugins.nchenouard.particletracking.filtering.Predictor2D;
import plugins.nchenouard.particletracking.filtering.Predictor3D;
import plugins.nchenouard.particletracking.gui.TrackingMainPanel;
import plugins.nchenouard.particletracking.legacytracker.StoppableTracker;
import plugins.nchenouard.particletracking.legacytracker.Tracker;
import plugins.nchenouard.particletracking.legacytracker.associationMethod.InstantaneousTracker;
import plugins.nchenouard.particletracking.legacytracker.associationMethod.SolveMLAssociation;
import plugins.nchenouard.particletracking.legacytracker.associationMethod.Track;
import plugins.nchenouard.particletracking.legacytracker.gui.PanelTracking;
import plugins.nchenouard.particletracking.simplifiedMHT.SimplifiedMHTPanel;
import plugins.nchenouard.spot.Detection;
import plugins.nchenouard.spot.DetectionResult;
import plugins.nchenouard.spot.Spot;

/**
 * Particle tracking plugin for ICY that uses the multiple hypothesis tracking algorithm
 * described in:
 * Nicolas Chenouard, Isabelle Bloch, Jean-Christophe Olivo-Marin,
 * "Multiple Hypothesis Tracking for Cluttered Biological Image Sequences," IEEE Transactions on Pattern Analysis and Machine Intelligence, vol. 35, no. 11, pp. 2736-3750, Nov., 2013
 * Pubmed link: http://www.ncbi.nlm.nih.gov/pubmed/23689865
 * 
 * Part of the Spot Tracking plugin for ICY: http://icy.bioimageanalysis.org/plugin/Spot_Tracking
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @version 3.1
 * @date 2013-11-13
 * @license gpl v3.0
 */

public class SpotTrackingPlugin extends PluginActionable implements ActionListener
{
	public static String MHT_PLUGIN_CONFIGURATION = "MHTconfiguration";
	private enum GUIStateEnum {ADVANCED, SIMPLE, LEGACY};
	GUIStateEnum currentGUISate = GUIStateEnum.SIMPLE;

	IcyFrame mainFrame;
	JMenuBar menuBar;
	JMenu guiMenu, fileMenu, infoMenu;
	JMenuItem menuItem;

	JPanel mainPanel;

	final String SIMPLEGUI = "Simple interface";
	final String ADVANCEDGUI = "Avanced interface";
	final String LEGACYGUI = "Legacy plugin";

	TrackingMainPanel advancedGUIPanel;
	SimplifiedMHTPanel simpleGUIPanel;
	PanelTracking legacyGUIPanel;

	LegacyTrackerManager legacyTrackerManager;
	public MHTrackerManager mhTrackerManager;
	MHTparameterSet mhtParameterSet = null;
	boolean defaultParameterSet = true;

	Thread computationThread = null;
	ProgressFrame trackingAnnounce;

	public static boolean optimizationLibraryLoaded = false;

	boolean useMultithreading = true;
	boolean useLPSolve = true;

	public SpotTrackingPlugin()
	{
		if (!optimizationLibraryLoaded)
		{
			try
			{
				if (!loadLibrary("lpsolve55"))
				{
					System.err.println("Library cannot be loaded");
					throw (new Exception("lpsolve55 could not be loaded"));
				}		        
				if (extractLibrary("lpsolve55j") == null)
				{
					System.err.println("Library cannot be extracted");
					throw (new Exception("lpsolve55 could not be loaded"));
				}
			}
			catch (Exception e)
			{
				e.printStackTrace();
				optimizationLibraryLoaded = false;
				JOptionPane.showMessageDialog(Icy.getMainInterface().getMainFrame(),
						"The lpsolve55 optimization library has failed to load.\n" +
								"The plugin is now using a slower library.\n" +
								"Please, consider switching to another platform (eg. windows) for improved performance,\nor contact the developper.",
								"Library loading error",
								JOptionPane.ERROR_MESSAGE);
				return;
			}
			optimizationLibraryLoaded = true;
		}
	}
	
	public File extractLibrary(String libName)
	{
		try
		{
			// get mapped library name
			String mappedlibName = System.mapLibraryName(libName);
			// get base resource path for native library
			final String basePath = getResourceLibraryPath() + ResourceUtil.separator;

			// search for library in resource
			URL libUrl = getResource(basePath + mappedlibName);

			// not found ?
			if (libUrl == null)
			{
				// jnilib extension may not work, try with "dylib" extension instead
				if (mappedlibName.endsWith(".jnilib"))
				{
					mappedlibName = mappedlibName.substring(0, mappedlibName.length() - 7) + ".dylib";
					libUrl = getResource(basePath + mappedlibName);
				}
			}

			// resource not found --> error
			if (libUrl == null)
				throw new IOException("Couldn't find resource " + basePath + mappedlibName);

			// extract resource
			return extractResource(SystemUtil.getTempLibraryDirectory() + FileUtil.separator + mappedlibName, libUrl);
		}
		catch (IOException e)
		{
			System.err.println("Error while loading packed library " + libName + ": " + e);
		}

		return null;
	}

	@Override
	public void run() {
		mhTrackerManager = new MHTrackerManager();

		mainPanel = new JPanel();
		mainFrame = new IcyFrame("Spot Tracking", true, true, true, true);
		mainFrame.getContentPane().add(mainPanel);

		// set up the tracking plugin panels

		mainPanel.setLayout(new BorderLayout());

		int iconSize = 100;
		ImageIcon detectionIcon = ResourceUtil.getImageIcon( ImageUtil.load( getResourceAsStream( "plugins/nchenouard/particletracking/simplifiedMHT/detectionIcon.png" ) ), iconSize );
		ImageIcon mhtIcon = ResourceUtil.getImageIcon( ImageUtil.load( getResourceAsStream( "plugins/nchenouard/particletracking/simplifiedMHT/MHTIcon.png" ) ), iconSize );
		ImageIcon outputIcon = ResourceUtil.getImageIcon( ImageUtil.load( getResourceAsStream( "plugins/nchenouard/particletracking/simplifiedMHT/trackPoolIcon.png" ) ), iconSize );
		ImageIcon workingIcon = ResourceUtil.getImageIcon( ImageUtil.load( getResourceAsStream( "plugins/nchenouard/particletracking/simplifiedMHT/workingIcon.png" ) ), iconSize );

		simpleGUIPanel = new SimplifiedMHTPanel(detectionIcon, mhtIcon, outputIcon, workingIcon);
		mainPanel.add(simpleGUIPanel, BorderLayout.CENTER);
		simpleGUIPanel.loadParametersButton.addActionListener(this);
		simpleGUIPanel.runTrackingButton.addActionListener(mhTrackerManager);
		simpleGUIPanel.detectionChooser.addItemListener(new ItemListener() {			
			@Override
			public void itemStateChanged(ItemEvent e) {
				if (simpleGUIPanel.detectionChooser.getSelectedDetectionResult() != null && mhtParameterSet != null)
					mhtParameterSet.detectionResults = simpleGUIPanel.detectionChooser.getSelectedDetectionResult();
			}
		});
		simpleGUIPanel.trackGroupNameTF.addKeyListener(new KeyListener() {
			@Override
			public void keyTyped(KeyEvent e) {
			}

			@Override
			public void keyReleased(KeyEvent e) {
				mhtParameterSet.trackGroupName = simpleGUIPanel.trackGroupNameTF.getText();
			}

			@Override
			public void keyPressed(KeyEvent e) {
			}
		});
		simpleGUIPanel.estimateParametersButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent arg0)
			{
				estimateTrackingParameters();
			}
		});

		advancedGUIPanel = new TrackingMainPanel();
		advancedGUIPanel.startTrackingButton.addActionListener(mhTrackerManager);
		advancedGUIPanel.configFilePanel.exportConfigurationFileButton.addActionListener(new ActionListener() {			
			@Override
			public void actionPerformed(ActionEvent e)
			{
				mhtParameterSet = advancedGUIPanel.getParameterSet();
				exportConfiguration();
			}
		});
		advancedGUIPanel.configFilePanel.loadConfigurationFileButton.addActionListener(new ActionListener() {			
			@Override
			public void actionPerformed(ActionEvent e)
			{
				loadMHTParameters();
			}
		});
		//		advancedGUIPanel.configFilePanel.runMultipleConfigurationFileButton.addActionListener(new ActionListener() {			
		//			@Override
		//			public void actionPerformed(ActionEvent e)
		//			{
		//			}
		//		});
		refreshMainPanel(currentGUISate);

		// legacy tracker

		legacyGUIPanel = new PanelTracking();
		legacyTrackerManager = new LegacyTrackerManager(legacyGUIPanel);

		// set up the menu bar
		menuBar = new JMenuBar();
		mainFrame.setJMenuBar(menuBar);

		fileMenu = new JMenu("File");
		menuBar.add(fileMenu);

		JMenuItem saveConfigurationItem = new JMenuItem("Save parameters");
		fileMenu.add(saveConfigurationItem);
		saveConfigurationItem.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				exportConfiguration();
			}
		});

		JMenuItem loadConfigurationItem = new JMenuItem("Load parameters");
		fileMenu.add(loadConfigurationItem);
		loadConfigurationItem.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				loadMHTParameters();
			}
		});

		JMenuItem configurationItem = new JMenuItem("Plugin configuration");
		fileMenu.add(configurationItem);
		configurationItem.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				String title = "Plugin configuration";
				boolean resizable = false;
				boolean closable = true;
				boolean maximizable = false;
				boolean iconifiable = true;
				final IcyInternalFrame configFrame = new IcyInternalFrame(title, resizable, closable, maximizable, iconifiable);				
				JPanel configFramePanel = new JPanel();
				configFrame.setContentPane(configFramePanel);
				configFramePanel.setLayout(new GridLayout(3, 1));
				final JCheckBox multithreadingBox = new JCheckBox("Multithreaded computation.");
				multithreadingBox.setSelected(useMultithreading);
				final JCheckBox lpSolveBox = new JCheckBox("Optimization library in C (faster).");
				lpSolveBox.setSelected(useLPSolve);
				final JButton closeButton = new JButton("Apply");
				configFramePanel.add(multithreadingBox);
				configFramePanel.add(lpSolveBox);
				configFramePanel.add(closeButton);

				closeButton.addActionListener(new ActionListener(){
					@Override
					public void actionPerformed(ActionEvent e) {
						useMultithreading = multithreadingBox.isSelected();
						useLPSolve = lpSolveBox.isSelected();
						configFrame.close(false);
					}});
				configFrame.pack();
				configFrame.setVisible(true);
				Icy.getMainInterface().addToDesktopPane(configFrame);
			}
		});

		guiMenu = new JMenu("Interface");
		menuBar.add(guiMenu);

		JMenuItem simplifiedGUIItem = new JMenuItem(SIMPLEGUI);
		guiMenu.add(simplifiedGUIItem);
		simplifiedGUIItem.addActionListener(this);

		JMenuItem advancedGUIItem = new JMenuItem(ADVANCEDGUI);
		guiMenu.add(advancedGUIItem);
		advancedGUIItem.addActionListener(this);

		JMenuItem legacyInterfaceItem = new JMenuItem(LEGACYGUI);
		guiMenu.add(legacyInterfaceItem);
		legacyInterfaceItem.addActionListener(this);

		infoMenu = new JMenu("Info");
		menuBar.add(infoMenu);

		JMenuItem manualItem = new JMenuItem("Manual");
		infoMenu.add(manualItem);
		manualItem.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e)
			{
				IcyFrame manualFrame = new IcyFrame("Spot Tracking Plugin Manual");
				manualFrame.setResizable(true);
				manualFrame.setClosable(true);
				JEditorPane descriptionPane = new JEditorPane();
				descriptionPane.setEditable(false);
				descriptionPane.setContentType("text/html");
				descriptionPane.setText( "<html><p>For a full description, please read the <a href=http://icy.bioimageanalysis.org/plugin/Spot_Tracking#documentation>online manual</a> on ICY website:<br> http://icy.bioimageanalysis.org/plugin/Spot_Tracking#documentation"
						+ "<p><b>Quick description:</b>"
						+"<br><html>The usual way of using the software is to follow indications given by the interface from top to bottom.<br> Tool tip text on each graphical element gives useful information about the use." +
						"<br>In general, the following sequence of operations will be performed: <ol><li>Select a set of locations in space and over time (detections) for the spots to track.<br> This can be done with the Spot Detector plugin. (Section 1)</li> <li>Estimate parameters for the tracking steps or load a set of existing parameters from a local file. (Section 2)</li><li>Specify a unique name for the tracking results. (Section 3)</li><li>Run the track construction process. (Section 4)</li><li>Analyze and save tracking results with the Track Manager plugin.</li><li>Save the tracking parameters via the File menu of the plugin for future re-use.</li></ol>"
						+ "<br>Exhaustive control of the tracking parameters is accessible by switching to the advanced interface.<br> (Menu Interface/Advanced Interface).</html>");
				manualFrame.setContentPane(descriptionPane);
				manualFrame.pack();
				addIcyFrame(manualFrame);
				manualFrame.setVisible(true);
			}
		});

		JMenuItem aboutItem = new JMenuItem("About");
		infoMenu.add(aboutItem);
		aboutItem.addActionListener(new ActionListener() {			
			@Override
			public void actionPerformed(ActionEvent arg0) {
				IcyFrame aboutFrame = new IcyFrame("About Spot Tracking Plugin");
				aboutFrame.setResizable(true);
				aboutFrame.setClosable(true);
				JEditorPane descriptionPane = new JEditorPane();
				descriptionPane.setEditable(false);
				descriptionPane.setContentType("text/html");
				descriptionPane.setText("<html><p>The Spot Tracking Plugin is a essentially an implementation of the methods described in the article:" +
						"<br>Nicolas Chenouard, Isabelle Bloch, Jean-Christophe Olivo-Marin,<br> <b>Multiple Hypothesis Tracking for Cluttered Biological Image Sequences</b>, <br>IEEE Transactions on Pattern Analysis and Machine Intelligence, vol. 35, no. 11, pp. 2736-3750, Nov., 2013" +
						"<br>Pubmed link: http://www.ncbi.nlm.nih.gov/pubmed/23689865" +
						"<br><p>By using the software for data analysis you agree to properly cite and reference this work<br> in any communication regarding results obtained by using it." +
						"<br><p>Author: Nicolas Chenouard. Institut Pasteur, Paris, France." +
						"<br>Version 3.1, 2013-11-13.</html>");
				aboutFrame.setContentPane(descriptionPane);
				aboutFrame.pack();
				addIcyFrame(aboutFrame);
				aboutFrame.setVisible(true);
			}
		});

		mainFrame.pack();
		mainFrame.addToDesktopPane();
		mainFrame.center();
		mainFrame.setVisible(true);
		mainFrame.requestFocus();

		defaultParameterSet = true;
		simpleGUIPanel.setUsingDefaultParameters(defaultParameterSet);
	}


	private void refreshMHTParameterSetFromGUI()
	{
		if (advancedGUIPanel != null)
		{
			MHTparameterSet parameters = advancedGUIPanel.getParameterSet();
			if (parameters != null)
			{
				mhtParameterSet = parameters;
				simpleGUIPanel.setParameter(mhtParameterSet);
				defaultParameterSet = false;
				simpleGUIPanel.setUsingDefaultParameters(defaultParameterSet);
			}
		}
	}

	private void setMHTParameterSetToGUI()
	{
		if (advancedGUIPanel != null && mhtParameterSet != null)
		{
			advancedGUIPanel.setParameterset(mhtParameterSet);
			simpleGUIPanel.setParameter(mhtParameterSet);
			defaultParameterSet = false;
		}
	}

	private void refreshMainPanel(final GUIStateEnum guiState)
	{
		mainPanel.removeAll();
		switch(guiState)
		{
		case ADVANCED:
			setMHTParameterSetToGUI(); // set the current parameters to the advanced GUI
			mainPanel.add(advancedGUIPanel, BorderLayout.CENTER);
			break;
		case LEGACY:
			mainPanel.add(legacyGUIPanel, BorderLayout.CENTER);
			break;
		case SIMPLE:
			refreshMHTParameterSetFromGUI(); // load parameters from advanced GUI
			mainPanel.add(simpleGUIPanel, BorderLayout.CENTER);
			break;
		default:
			break;
		}
		mainFrame.pack();
		mainFrame.updateUI();
		currentGUISate = guiState;
	}


	@Override
	public void actionPerformed(ActionEvent e) {
		if (e.getSource() instanceof JMenuItem)
		{
			if (((JMenuItem)e.getSource()).getText().equals(SIMPLEGUI))
				refreshMainPanel(GUIStateEnum.SIMPLE); // switch to simple GUI
			if (((JMenuItem)e.getSource()).getText().equals(ADVANCEDGUI))
				refreshMainPanel(GUIStateEnum.ADVANCED); // switch to advanced GUI
			if (((JMenuItem)e.getSource()).getText().equals(LEGACYGUI))
				refreshMainPanel(GUIStateEnum.LEGACY); // switch to legacy GUI
		}
		if (e.getSource() == simpleGUIPanel.loadParametersButton)
			loadMHTParameters();
	}

	public void loadMHTParameters()
	{
		MHTparameterSet parameters = loadConfiguration();
		if (parameters != null)
		{
			mhtParameterSet = parameters;
			advancedGUIPanel.setParameterset(mhtParameterSet);
		}	
	}

	private void exportConfigurationToFile(File file) {
		Document document = XMLUtil.createDocument( true );
		Element configurationElement = document.createElement(MHT_PLUGIN_CONFIGURATION);
		XMLUtil.getRootElement( document ).appendChild(configurationElement);
		if (mhtParameterSet != null)
		{
			mhtParameterSet.saveToXML(configurationElement);
		}
		XMLUtil.saveDocument( document , file );
	}

	public void exportConfiguration() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileFilter(new FileFilter() {			
			@Override
			public String getDescription() {
				return "XML file filter";
			}
			@Override
			public boolean accept(File file) {
				if (file.isDirectory())
					return true;
				return file.getName().endsWith(".xml");
			}
		});
		fileChooser.setMultiSelectionEnabled(false);
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setName("Output xml file selection");
		int returnVal = fileChooser.showOpenDialog(mainPanel);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();
			if(!file.getName().endsWith(".xml"))
				file = new File(file.getAbsolutePath() + ".xml");
			exportConfigurationToFile(file);
		}
	}

	public MHTparameterSet loadConfiguration() {
		MHTparameterSet parameterSet = null;
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileFilter(new FileFilter() {			
			@Override
			public String getDescription() {
				return "XML file filter";
			}
			@Override
			public boolean accept(File file) {
				if (file.isDirectory())
					return true;
				return file.getName().endsWith(".xml");
			}
		});
		fileChooser.setMultiSelectionEnabled(false);
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setName("Input xml file selection");
		int returnVal = fileChooser.showOpenDialog(mainPanel);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();
			parameterSet = loadConfigurationFromFile(file);
		}
		return parameterSet;
	}

	private MHTparameterSet loadConfigurationFromFile(File file)
	{
		Document document = XMLUtil.loadDocument( file );
		Element root = XMLUtil.getRootElement( document );
		if ( root == null )
		{
			throw new IllegalArgumentException( "can't find: <root> tag." );
		}
		Element configurationElement = XMLUtil.getElements( root , MHT_PLUGIN_CONFIGURATION ).get( 0 );
		if ( configurationElement == null )
		{
			throw new IllegalArgumentException( "can't find: <root><" + MHT_PLUGIN_CONFIGURATION + "> tag." ) ;		
		}
		MHTparameterSet parameterSet = MHTparameterSet.loadFromXML(configurationElement);
		return parameterSet;
	}

	public static void sendTracksToPool(final TrackGroup trackGroup,
			final Sequence sequence) {
		SwingUtilities.invokeLater(new Runnable()
		{
			public void run() {
				// Add the given trackGroup
				SwimmingObject result = new SwimmingObject(trackGroup);
				Icy.getMainInterface().getSwimmingPool().add(result);
				TrackManager manager = new TrackManager();
				if (sequence != null)
					manager.setDisplaySequence(sequence );
				new AnnounceFrame("Tracking results exported to Track manager plugin");
			}
		});
	}

	double squaredDistance(Spot s1, Spot s2)
	{
		return (s1.mass_center.x - s2.mass_center.x)*(s1.mass_center.x - s2.mass_center.x) + (s1.mass_center.y - s2.mass_center.y)*(s1.mass_center.y - s2.mass_center.y) + (s1.mass_center.z - s2.mass_center.z)*(s1.mass_center.z - s2.mass_center.z);
	}

	public void estimateTrackingParameters()
	{
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				ParameterEstimationFrame estimationFrame = new ParameterEstimationFrame();
				addIcyFrame(estimationFrame);
				estimationFrame.setVisible(true);
			}
		});		
	}
		
	public class MHTrackerManager implements ActionListener
	{
		boolean isRunning = false;
		boolean stopTracking = false;
		boolean stopped = false;
		HMMMHTracker runningTracker;
		private Lock stopLock = new ReentrantLock();
		private Condition stopCondition = stopLock.newCondition();

		@Override
		public void actionPerformed(ActionEvent e)
		{
			if (e.getSource() == advancedGUIPanel.startTrackingButton || e.getSource() == simpleGUIPanel.runTrackingButton)
			{
				if (e.getSource() == advancedGUIPanel.startTrackingButton)
					mhtParameterSet = advancedGUIPanel.getParameterSet();
				startSoppableMHTracking();
			}
		} 

		public void startSoppableMHTracking()
		{
			if (isRunning)
			{
				stopLock.lock();
				if (runningTracker != null)
					runningTracker.stopComputing();
				stopTracking = true;
				try {
					while (isRunning)
						stopCondition.await();
				} catch (InterruptedException ie) {
					ie.printStackTrace();
				}
				finally
				{
					stopLock.unlock();
				}
				stopTracking = false;
			}
			else
			{
				enableTracking(false);
				startMHTracking();					
			}	
		}

		private void enableTracking(final boolean isEnableTracking)
		{
			SwingUtilities.invokeLater(new Runnable(){
				@Override
				public void run() {
					simpleGUIPanel.changeTrackingState(!isEnableTracking);				
					advancedGUIPanel.changeTrackingState(!isEnableTracking);
				}});
		}

		public HMMMHTracker buildMHTracker(DetectionResult dr)
		{
			HMMMHTracker tracker = null;
			int dim = 2;
			double volume = 256*256;
			if (dr.getSequence() != null)
			{
				if (dr.getSequence().getSizeZ() > 1)
					dim = 3;
				volume = dr.getSequence().getSizeX()*dr.getSequence().getSizeY()*dr.getSequence().getSizeZ();
			}
			else
			{
				// check detections to see if they all have the same z coordinate
				//				double minX = Double.MAX_VALUE;
				//				double minY = Double.MAX_VALUE;
				//				double minZ = Double.MAX_VALUE;
				double minX = 0; // assume the sequence coordinates start at 0
				double minY = 0;
				double minZ = 0;
				double maxX = 0;
				double maxY = 0;	
				double maxZ = 0;

				for (int t = dr.getFirstFrameTime(); t <= dr.getLastFrameTime(); t++)
				{
					for (Spot s:dr.getDetectionsAtT(t))
					{
						minX = Math.min(minX, s.mass_center.x);
						minY = Math.min(minY, s.mass_center.y);
						minZ = Math.min(minZ, s.mass_center.z);
						maxX = Math.max(maxX, s.mass_center.x);
						maxY = Math.max(maxY, s.mass_center.y);
						maxZ = Math.max(maxZ, s.mass_center.z);					
					}
				}
				if (maxZ > minZ)
				{
					dim = 3;
					volume = (maxX - minX +1)*(maxY - minY +1)*(maxZ - minZ +1);
				}
				else
				{
					dim = 2;
					volume = (maxX - minX +1)*(maxY - minY +1);
				}
			}
			try {
				tracker = advancedGUIPanel.buildTracker(dim, volume, useLPSolve && optimizationLibraryLoaded, useMultithreading);
			} catch (Exception e) {
				e.printStackTrace();
			}
			return tracker;
		}

		private void startMHTracking()
		{
			final DetectionResult dr = mhtParameterSet.detectionResults;// advancedGUIPanel.getSelectedDetectionResult();
			if (dr == null)
			{
				enableTracking(true);
				new AnnounceFrame("Tracking requires to specify a set of detections. You can use the Spot Detector plugin to create some.", 10);
				return;
			}
			if (dr.getNumberOfDetection() < 1)
			{
				enableTracking(true);
				new AnnounceFrame("A non empty set of detections needs to be used");
				return;
			}

			final HMMMHTracker mhtracker = buildMHTracker(dr);
			computationThread = new Thread(){
				@Override
				public void run()
				{
					trackingAnnounce.setLength(dr.getLastFrameTime());
					for (int t = dr.getFirstFrameTime(); t <= dr.getLastFrameTime(); t++)
					{
						if (!stopTracking)
						{
							trackingAnnounce.setMessage("Track extraction at frame " + t);
							trackingAnnounce.setPosition(t);
							mhtracker.track(t, dr.getDetectionsAtT(t));
						}
						else
						{
							stopped = true;
							break;
						}
					}
					trackingAnnounce.close();
					ArrayList<TrackSegment> tracks = mhtracker.getCompleteTracks();
					TrackGroup tg = new TrackGroup(dr.getSequence());
					tg.setDescription(mhtParameterSet.trackGroupName);
					for (TrackSegment ts:tracks)
						tg.addTrackSegment(ts);
					sendTracksToPool(tg, tg.getSequence());
					isRunning = false;
					enableTracking(true);
					if (stopped)
					{
						stopLock.lock();
						isRunning = false;
						stopCondition.signalAll();
						stopLock.unlock();
					}
					computationThread = null;
					runningTracker = null;
				}
			};
			trackingAnnounce = new ProgressFrame("Tracking progress started");
			computationThread.start();
		}
	}

	class LegacyTrackerManager implements ActionListener
	{
		private StoppableTracker runningTracker = null;
		private boolean stopTracking = false;
		private boolean isRunning = false;

		private Thread runningThread = null;

		private Lock stopLock = new ReentrantLock();
		private Condition stopCondition = stopLock.newCondition();

		PanelTracking panelTracking;

		public LegacyTrackerManager(PanelTracking legacyGUIPanel)
		{
			legacyGUIPanel.trackingStartButton.addActionListener(this);
			this.panelTracking = legacyGUIPanel;
		}

		private void enableTracking(final boolean isEnableTracking)
		{
			SwingUtilities.invokeLater(new Runnable(){
				@Override
				public void run() {
					panelTracking.changeTrackingState(!isEnableTracking);				
				}});
		}

		public void actionPerformed(ActionEvent e)
		{
			if (e.getSource() == panelTracking.trackingStartButton)
			{
				if (isRunning)
				{
					stopLock.lock();
					if (runningTracker != null)
						runningTracker.stopComputing();
					stopTracking = true;
					try {
						while (isRunning)
							stopCondition.await();
					} catch (InterruptedException ie) {
						ie.printStackTrace();
					}
					finally
					{
						stopLock.unlock();
					}
					stopTracking = false;
				}
				else
				{
					DetectionResult detections = panelTracking.getDetectionResults();
					if (detections != null) {
						enableTracking(false);
						trackingRun(detections, detections.getSequence());
					} else
						new AnnounceFrame("Please first select a detection set.");
				}
			}
		}

		private synchronized void trackingRun(final DetectionResult detections, final Sequence sequence) {
			if (detections != null) {
				isRunning = true;
				Thread trackThread = new Thread() {
					public void run() {
						int dim = 2;
						if (sequence.getSizeZ() > 1)
							dim = 3;
						Tracker tracker = panelTracking.buildTracker(dim);
						if (tracker instanceof StoppableTracker)
							runningTracker = (StoppableTracker)tracker;
						int firstT = detections.getFirstFrameTime();
						int lastT = detections.getLastFrameTime();
						String message = "Tracking at frame ";
						ProgressFrame announceFrame = new ProgressFrame("");
						announceFrame.setLength(lastT - firstT + 1);
						boolean stopped = false;
						for (int t = firstT; t <= lastT; t++) {
							if (!stopTracking)
							{
								announceFrame.setPosition(t);
								announceFrame.setMessage(message+" "+t);
								tracker.track(t, detections.getDetectionsAtT(t));
							}
							else
							{
								stopped = true;
								break;
							}
						}
						announceFrame.close();
						if (tracker instanceof InstantaneousTracker) {
							TrackGroup trackGroup = new TrackGroup(sequence);
							trackGroup.setDescription(panelTracking.getTrackGroupName());
							for (Track track : ((InstantaneousTracker) tracker)
									.getTracks()) {
								TrackSegment ts = convertTrack2TrackSegment(track,
										null);
								if (ts != null && ts.getDetectionList().size() > 1)
									trackGroup.addTrackSegment(ts);
							}
							sendTracksToPool(trackGroup, sequence);
							new AnnounceFrame("Tracks exported as "+trackGroup.getDescription()+" in TrackEditor");
						}
						isRunning = false;
						enableTracking(true);
						if (stopped)
						{
							stopLock.lock();
							isRunning = false;
							stopCondition.signalAll();
							stopLock.unlock();
						}
						runningThread = null;
						runningTracker = null;
					}
				};
				runningThread = trackThread;
				trackThread.start();
			}
		}

		public TrackSegment convertTrack2TrackSegment(Track track, Object detectionEditor) {
			int firstIndex = track.getFirstIndex();
			int lastIndex = track.getLastIndex();
			for (int i = lastIndex; i > firstIndex; i--)
			{
				if (track.isPredictionAtFrame(i))
					lastIndex = i-1;
				else
					break;
			}
			if (firstIndex <= lastIndex)
			{
				ArrayList<Detection> detections = new ArrayList<Detection>(lastIndex - firstIndex + 1);
				for (int i = firstIndex; i <= lastIndex; i++)
				{
					Spot s = track.getSpotAtFrame(i);
					Detection detect = new DetectionSpotTrack(s, i);
					if (track.isPredictionAtFrame(i))
						detect.setDetectionType(Detection.DETECTIONTYPE_VIRTUAL_DETECTION);
					detections.add(detect);
				}
				return new TrackSegment(detections);
			}
			else
				return null;
		}
	}

	public void estimateParameters(final boolean isDirectedMotion, final boolean isSingleMotion, final boolean isUpdateMotion)
	{
		
				mhtParameterSet.isDirectedMotion = isDirectedMotion;
				mhtParameterSet.isSingleMotion = isSingleMotion;
				mhtParameterSet.isUpdateMotion = isUpdateMotion;

				// first get an estimation of parameters based on analysis of distance between detections
				if (mhtParameterSet.detectionResults == null)
					return;
				// fill the detection distance list between subsequent detections
				int maxNumDetections = 10000;
				int numDetections = 0;
				int numT = 0;
				for (int t = mhtParameterSet.detectionResults.getFirstFrameTime(); t < mhtParameterSet.detectionResults.getLastFrameTime(); t++)
				{
					numDetections += mhtParameterSet.detectionResults.getDetectionsAtT(t).size();
					numT++;
					if (numDetections > maxNumDetections)
						break;
				}
				int t = mhtParameterSet.detectionResults.getFirstFrameTime();
				double[] distTabXY = new double[numDetections];
				double[] distTabZ = new double[numDetections];		
				int cntDist = 0;
				while (t < mhtParameterSet.detectionResults.getFirstFrameTime() + numT)
				{
					Vector<Spot> dList1 = mhtParameterSet.detectionResults.getDetectionsAtT(t);
					Vector<Spot> dList2 = mhtParameterSet.detectionResults.getDetectionsAtT(t + 1);
					if (!dList1.isEmpty() && !dList2.isEmpty())
					{
						for (Spot s1:dList1)
						{
							double minDist = Double.MAX_VALUE;
							Spot minSpot = null;
							for (Spot s2:dList2)
							{
								double d  = squaredDistance(s1, s2);
								if (d < minDist)
								{
									minDist = d;
									minSpot = s2;
								}
							}
							if (minDist < Double.MAX_VALUE)
							{
								///distTab[cntDist] = minDist;
								distTabXY[cntDist] = (s1.mass_center.x - minSpot.mass_center.x)*(s1.mass_center.x - minSpot.mass_center.x) + (s1.mass_center.y - minSpot.mass_center.y)*(s1.mass_center.y - minSpot.mass_center.y);
								distTabZ[cntDist] += (s1.mass_center.z - minSpot.mass_center.z)*(s1.mass_center.z - minSpot.mass_center.z);						
								cntDist++;
							}
						}
					}
					t++;
				}

				// get a first estimation of the diffusion as the underestimated median of the squared distances
				Arrays.sort(distTabXY);
				Arrays.sort(distTabZ);
				double varDiffXY = distTabXY[(int)(distTabXY.length*0.45)];
				double varDiffZ = distTabZ[(int)(distTabXY.length*0.45)];

				// then run the legacy tracker to get a refined estimation
				int maxConsecutivePred = 1;
				int gateFactor = 3;
				boolean gateLikelihood = true;
				Predictor predictor = null;
				if (isSingleMotion)
				{
					if (varDiffZ > 0)
					{
						if (isDirectedMotion)
							predictor = new KF3dDirected();
						else
							predictor = new KF3dRandomWalk();
						predictor.setTrackingCovariances(new double[]{varDiffXY, varDiffXY, varDiffZ});
					}
					else
					{
						if (isDirectedMotion)
							predictor = new KF2dDirected();
						else
							predictor = new KF2dRandomWalk();
						predictor.setTrackingCovariances(new double[]{varDiffXY, varDiffXY});
					}
				}
				else
				{
					if (varDiffZ > 0)
					{
						ArrayList<Predictor3D> predictors = new ArrayList<Predictor3D>();
						predictors.add(new KF3dRandomWalk());
						predictors.add(new KF3dDirected());
						predictors.get(0).setTrackingCovariances(new double[]{varDiffXY, varDiffXY, varDiffZ});
						predictors.get(1).setTrackingCovariances(new double[]{varDiffXY, varDiffXY, varDiffZ});							
						if (mhtParameterSet.useMostLikelyModel)
							predictor = new IMM3D(IMM3D.LikelihoodTypes.IMM_MAX_LIKELIHOOD, new Double(mhtParameterSet.immInertia), predictors);
						else
							predictor = new IMM3D(IMM3D.LikelihoodTypes.IMM_WEIGHTED_LIKELIHOOD, mhtParameterSet.immInertia, predictors);
					}
					else
					{
						ArrayList<Predictor2D> predictors = new ArrayList<Predictor2D>();
						predictors.add(new KF2dRandomWalk());
						predictors.add(new KF2dDirected());
						predictors.get(0).setTrackingCovariances(new double[]{varDiffXY, varDiffXY});
						predictors.get(1).setTrackingCovariances(new double[]{varDiffXY, varDiffXY});							
						if (mhtParameterSet.useMostLikelyModel)
							predictor = new IMM2D(IMM2D.LikelihoodTypes.IMM_MAX_LIKELIHOOD, new Double(mhtParameterSet.immInertia), predictors);
						else
							predictor = new IMM2D(IMM2D.LikelihoodTypes.IMM_WEIGHTED_LIKELIHOOD, mhtParameterSet.immInertia, predictors);
					}
				}
				SolveMLAssociation assignementSolver = new SolveMLAssociation(true);
				InstantaneousTracker tracker = new InstantaneousTracker(assignementSolver, predictor, gateLikelihood, gateFactor, maxConsecutivePred);
				for (int tt = mhtParameterSet.detectionResults.getFirstFrameTime(); tt < mhtParameterSet.detectionResults.getLastFrameTime(); tt++)
					tracker.track(tt, mhtParameterSet.detectionResults.getDetectionsAtT(tt));
				ArrayList<Track> tracks = tracker.getTracks();
				// consider tracks with length < 3 to be wrong
				ArrayList<Track> trueTracks = new ArrayList<Track>();
				ArrayList<Track> falseTracks = new ArrayList<Track>();
				for (Track tr:tracks)
				{
					if (tr.getLastIndex() - tr.getFirstIndex() > 2)
						trueTracks.add(tr);
					else
						falseTracks.add(tr);
				}
				if (trueTracks.isEmpty())
					return;
				// compute track length
				int sumTrackLength = 0;
				for (Track tr:trueTracks)
					sumTrackLength += (tr.getLastIndex() - tr.getFirstIndex() + 1);
				mhtParameterSet.meanTrackLength = sumTrackLength/trueTracks.size();
				// compute number of tracks at first frame, and number of new tracks in other frames
				int numTrackFirstFrame = 0;
				int numNewTracks = 0;
				for (Track tr:trueTracks)
				{
					if (tr.getFirstIndex() == 0)
						numTrackFirstFrame ++;
					else
						numNewTracks ++;
				}
				mhtParameterSet.numberInitialObjects = numTrackFirstFrame;
				mhtParameterSet.numberNewObjects = (numNewTracks/(1d + mhtParameterSet.detectionResults.getLastFrameTime() - mhtParameterSet.detectionResults.getFirstFrameTime()));

				// compute number of false detections per frame
				int numFalseDetections = 0;
				for (Track tr:falseTracks)
				{
					for (int tt = tr.getFirstIndex(); tt < tr.getLastIndex() + 1; tt++)
					{
						if (tr.isPredictionAtFrame(tt))
							numFalseDetections++;
					}
				}
				mhtParameterSet.numberOfFalseDetections = (int) (Math.round(numFalseDetections/(1d + mhtParameterSet.detectionResults.getLastFrameTime() - mhtParameterSet.detectionResults.getFirstFrameTime()))); 

				// re estimate diffusion //TODO: do this with MSDs to account for directed periods
				//					if (isSingleMotion && !isDirectedMotion)
				{
					// just use squared displacement
					double sumDistSq = 0;
					double sumDistSqZ = 0;					
					int numDist = 0;
					for (Track tr:trueTracks)
					{
						for (int f = tr.getFirstIndex(); f < tr.getLastIndex(); f++)
						{
							if(!tr.isPredictionAtFrame(f) && !tr.isPredictionAtFrame(f +1))
							{
								Spot s1 = tr.getSpotAtFrame(f);
								Spot s2 = tr.getSpotAtFrame(f + 1);
								sumDistSq += (s1.mass_center.x - s2.mass_center.x)*(s1.mass_center.x - s2.mass_center.x) + (s1.mass_center.y - s2.mass_center.y)*(s1.mass_center.y - s2.mass_center.y);
								sumDistSqZ += (s1.mass_center.z - s2.mass_center.z)*(s1.mass_center.z - s2.mass_center.z);
								numDist++;
								if (numDist > maxNumDetections)
									break;
							}
						}
						if (numDist > maxNumDetections)
							break;
					}
					mhtParameterSet.displacementXY = Math.sqrt(sumDistSq/((double) numDist));
					mhtParameterSet.displacementZ = Math.sqrt(sumDistSqZ/((double) numDist));
				}
				// estimate detection rate
				int numTrueDetections = 0;
				int numVirtualDetections = 0;
				for (Track tr:trueTracks)
				{
					for (int f = tr.getFirstIndex(); f < tr.getLastIndex(); f++)
						if (tr.isPredictionAtFrame(f))
							numVirtualDetections++;
						else
							numTrueDetections++;
				}
				mhtParameterSet.detectionRate = numTrueDetections/((double)numTrueDetections + numVirtualDetections);	
	}
	
	class ParameterEstimationFrame extends IcyFrame
	{
		public ParameterEstimationFrame()
		{
			super("Parameters estimation", true, true, true, true);
			JPanel mainPanel = new JPanel();
			this.setContentPane(mainPanel);
			mainPanel.setLayout(new GridBagLayout());
			GridBagConstraints gc = new GridBagConstraints();
			gc.fill = GridBagConstraints.HORIZONTAL;
			gc.gridy = 0;
			mainPanel.add(new JLabel("Target motion"), gc);
			gc.gridy++;
			final JRadioButton diffusiveButton = new JRadioButton("is mainly diffusive (default).");
			mainPanel.add(diffusiveButton, gc);
			gc.gridy++;
			final JRadioButton directedButton = new JRadioButton("is mainly directed.");
			mainPanel.add(directedButton, gc);
			gc.gridy++;
			final JRadioButton directedDiffusiveButton = new JRadioButton("is both diffusive and directed.");
			mainPanel.add(directedDiffusiveButton, gc);
			gc.gridy++;			
			ButtonGroup motionGroup = new ButtonGroup();
			motionGroup.add(diffusiveButton);
			motionGroup.add(directedButton);
			motionGroup.add(directedDiffusiveButton);
			diffusiveButton.setSelected(true);

			mainPanel.add(new JLabel("Parameters for target motion"), gc);
			gc.gridy++;
			final JRadioButton noUpdateMotionButton = new JRadioButton("are kept to their initial values (default).");
			mainPanel.add(noUpdateMotionButton, gc);
			gc.gridy++;
			final JRadioButton updateMotionButton = new JRadioButton("are re-estimated online.");
			mainPanel.add(updateMotionButton, gc);
			gc.gridy++;
			ButtonGroup updateGroup = new ButtonGroup();
			updateGroup.add(noUpdateMotionButton);
			updateGroup.add(updateMotionButton);
			noUpdateMotionButton.setSelected(true);

			JButton estimateParametersButton = new JButton("Run parameter estimation procedure");
			mainPanel.add(estimateParametersButton, gc);
			estimateParametersButton.addActionListener(new ActionListener() {
				@Override
				public void actionPerformed(ActionEvent arg0)
				{	
					computationThread = new Thread()
					{
						@Override
						public void run()
						{
							estimateParameters(directedButton.isSelected(), !directedDiffusiveButton.isSelected(), updateMotionButton.isSelected());
							SwingUtilities.invokeLater(new Runnable(){
								@Override
								public void run() {
									simpleGUIPanel.changeTrackingState(false);				
									advancedGUIPanel.changeTrackingState(false);
									if (trackingAnnounce != null)
										trackingAnnounce.close();
								}});	
						}
					};
					trackingAnnounce = new ProgressFrame("Parameter estimation in progress");
					simpleGUIPanel.changeTrackingState(true);				
					advancedGUIPanel.changeTrackingState(true);
					computationThread.start();
					close();
				}
			});
			this.pack();
		}		
	}
}
