/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 *     Daniel Schmitter (daniel.schmitter@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnake.gui;

import icy.gui.frame.progress.AnnounceFrame;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import plugins.big.bigsnake.BIGSnake;
import plugins.big.bigsnake.core.Settings;
import plugins.big.bigsnake.snake.ESnakeEnergyType;
import plugins.big.bigsnake.snake.ESnakeParameters;
import plugins.big.bigsnake.snake.ESnakePriorShapeType;
import plugins.big.bigsnake.snake.ESnakeTargetType;
import plugins.big.bigsnake.snake.ShapeSpaceType;
import plugins.big.bigsnakeutils.icy.gui.CollapsiblePanel;
import plugins.big.bigsnakeutils.icy.gui.DetailPanelMode;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Banana;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Boomerang;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.BrainWhite;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.BuddingYeast;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.CorpusCallosum;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Cross;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Custom;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.ElectricGuitar;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Femur;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Fly;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.MouseOrgan;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Rod;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Semicircle;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Square;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.UpperLip;

/**
 * Panel in which the user specifies the parameters of the E-Snake.
 * 
 * @version May 24, 2015
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Daniel Schmitter (daniel.schmitter@epfl.ch)
 */
public class SnakeSettingsPane extends CollapsiblePanel implements
ChangeListener, ItemListener {

	/**
	 * Determines if a de-serialized file is compatible with this class.
	 * 
	 * Maintainers must change this value if and only if the new version of this
	 * class is not compatible with old versions. See Sun docs for <a
	 * href=http://java.sun.com/products/jdk/1.1/docs/guide
	 * /serialization/spec/version.doc.html> details. </a>
	 * 
	 * Not necessary to include in first version of the class, but included here
	 * as a reminder of its importance.
	 */
	private static final long serialVersionUID = -2078162621447160540L;

	private final JComboBox targetBrightnessComboBox_ = new JComboBox();
	private final JSpinner numControlPointsSpinner_ = new JSpinner();
	private final JComboBox energyTypeComboBox_ = new JComboBox();
	private final JSpinner alphaSpinner_ = new JSpinner();
	private final JComboBox priorShapeComboBox_ = new JComboBox();
	private final JComboBox shapeSpaceComboBox_ = new JComboBox();
	private final JSpinner betaSpinner_ = new JSpinner();
	private final JSpinner maxIterationsSpinner_ = new JSpinner();
	private final JCheckBox immortalCheckbox_ = new JCheckBox("Immortal");

	private final JLabel targetBrightnessLabel_ = new JLabel(
			"Target brightness");
	private final JLabel numControlPointsLabel_ = new JLabel("Control points");
	private final JLabel energyTypeLabel_ = new JLabel("Energy type");
	private final JLabel alphaLabel_ = new JLabel("Alpha");
	private final JLabel priorShapeLabel_ = new JLabel("Prior shape");
	private final JLabel shapeSpaceLabel_ = new JLabel("Shape space");
	private final JLabel betaLabel_ = new JLabel("Beta");
	private final JLabel maxIterationsLabel_ = new JLabel("Max iterations");

	private final String[] energyTypeStrings_ = new String[] { "Contour",
			"Region", "Mixture" };
	private final String[] detectTypeStrings_ = new String[] { "Dark", "Bright" };
	private final String[] priorShapeStrings_ = new String[] { "None", "Rod",
			"Budding Yeast", "Banana", "Boomerang", "Brain white", "Circle",
			"Corpus callosum", "Cross", "Electric guitar", "Femur", "Fly",
			"Mouse organ", "Semicircle", "Square", "Upper lip", "Custom" };
	private final String[] shapeSpaceStrings_ = new String[] { "Similarity",
			"Affine" };

	private BIGSnake bigSnake_ = null;

	private Custom customPriorShape_ = null;

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public SnakeSettingsPane(String title, BIGSnake bigSnake) {
		super(title);
		bigSnake_ = bigSnake;

		GridBagLayout gridBagLayout = new GridBagLayout();
		gridBagLayout.columnWidths = new int[] { 120, 150, 0 };
		gridBagLayout.rowHeights = new int[] { 27, 28, 27, 28, 28, 28, 28, 28,
				23, 0 };
		gridBagLayout.columnWeights = new double[] { 0.0, 1.0, Double.MIN_VALUE };
		gridBagLayout.rowWeights = new double[] { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
				0.0, 0.0, 0.0, Double.MIN_VALUE };
		setLayout(gridBagLayout);

		JPanel targetBrightnessLabelPanel = new JPanel();
		GridBagConstraints gbc_targetBrightnessLabelPanel = new GridBagConstraints();
		gbc_targetBrightnessLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_targetBrightnessLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_targetBrightnessLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_targetBrightnessLabelPanel.gridx = 0;
		gbc_targetBrightnessLabelPanel.gridy = 0;
		add(targetBrightnessLabelPanel, gbc_targetBrightnessLabelPanel,
				DetailPanelMode.BASICS);
		targetBrightnessLabelPanel.setLayout(new GridLayout(0, 1, 0, 0));

		targetBrightnessLabelPanel.add(targetBrightnessLabel_);

		JPanel targetBrightnessComboBoxPanel = new JPanel();
		GridBagConstraints gbc_targetBrightnessComboBoxPanel = new GridBagConstraints();
		gbc_targetBrightnessComboBoxPanel.insets = new Insets(0, 0, 5, 0);
		gbc_targetBrightnessComboBoxPanel.fill = GridBagConstraints.BOTH;
		gbc_targetBrightnessComboBoxPanel.gridx = 1;
		gbc_targetBrightnessComboBoxPanel.gridy = 0;
		add(targetBrightnessComboBoxPanel, gbc_targetBrightnessComboBoxPanel,
				DetailPanelMode.BASICS);
		targetBrightnessComboBoxPanel.setLayout(new GridLayout(0, 1, 0, 0));

		targetBrightnessComboBoxPanel.add(targetBrightnessComboBox_);
		targetBrightnessComboBox_.setModel(new DefaultComboBoxModel(
				detectTypeStrings_));
		targetBrightnessComboBox_.setSelectedIndex(0);

		JPanel numControlPointsLabelPanel = new JPanel();
		GridBagConstraints gbc_numControlPointsLabelPanel = new GridBagConstraints();
		gbc_numControlPointsLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_numControlPointsLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_numControlPointsLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_numControlPointsLabelPanel.gridx = 0;
		gbc_numControlPointsLabelPanel.gridy = 1;
		add(numControlPointsLabelPanel, gbc_numControlPointsLabelPanel,
				DetailPanelMode.BASICS);
		numControlPointsLabelPanel.setLayout(new GridLayout(0, 1, 0, 0));

		numControlPointsLabelPanel.add(numControlPointsLabel_);

		JPanel numControlPointsSpinnerPanel = new JPanel();
		GridBagConstraints gbc_numControlPointsSpinnerPanel = new GridBagConstraints();
		gbc_numControlPointsSpinnerPanel.insets = new Insets(0, 0, 5, 0);
		gbc_numControlPointsSpinnerPanel.fill = GridBagConstraints.BOTH;
		gbc_numControlPointsSpinnerPanel.gridx = 1;
		gbc_numControlPointsSpinnerPanel.gridy = 1;
		add(numControlPointsSpinnerPanel, gbc_numControlPointsSpinnerPanel,
				DetailPanelMode.BASICS);
		numControlPointsSpinnerPanel.setLayout(new GridLayout(0, 1, 0, 0));

		numControlPointsSpinnerPanel.add(numControlPointsSpinner_);
		numControlPointsSpinner_.setModel(new SpinnerNumberModel(new Integer(
				Settings.M_DEFAULT), new Integer(3), new Integer(20),
				new Integer(1)));

		JPanel energyTypeLabelPanel = new JPanel();
		GridBagConstraints gbc_energyTypeLabelPanel = new GridBagConstraints();
		gbc_energyTypeLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_energyTypeLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_energyTypeLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_energyTypeLabelPanel.gridx = 0;
		gbc_energyTypeLabelPanel.gridy = 2;
		add(energyTypeLabelPanel, gbc_energyTypeLabelPanel,
				DetailPanelMode.ADVANCED);
		energyTypeLabelPanel.setLayout(new GridLayout(0, 1, 0, 0));

		energyTypeLabelPanel.add(energyTypeLabel_);

		JPanel energyTypeComboBoxPanel = new JPanel();
		GridBagConstraints gbc_energyTypeComboBoxPanel = new GridBagConstraints();
		gbc_energyTypeComboBoxPanel.insets = new Insets(0, 0, 5, 0);
		gbc_energyTypeComboBoxPanel.fill = GridBagConstraints.BOTH;
		gbc_energyTypeComboBoxPanel.gridx = 1;
		gbc_energyTypeComboBoxPanel.gridy = 2;
		add(energyTypeComboBoxPanel, gbc_energyTypeComboBoxPanel,
				DetailPanelMode.ADVANCED);
		energyTypeComboBoxPanel.setLayout(new GridLayout(0, 1, 0, 0));

		energyTypeComboBoxPanel.add(energyTypeComboBox_);
		energyTypeComboBox_.setModel(new DefaultComboBoxModel(
				energyTypeStrings_));
		energyTypeComboBox_.setSelectedIndex(1);

		JPanel alphaLabelPanel = new JPanel();
		GridBagConstraints gbc_alphaLabelPanel = new GridBagConstraints();
		gbc_alphaLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_alphaLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_alphaLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_alphaLabelPanel.gridx = 0;
		gbc_alphaLabelPanel.gridy = 3;
		add(alphaLabelPanel, gbc_alphaLabelPanel, DetailPanelMode.ADVANCED);
		alphaLabelPanel.setLayout(new GridLayout(0, 1, 0, 0));

		alphaLabelPanel.add(alphaLabel_);

		JPanel alphaSpinnerPanel = new JPanel();
		GridBagConstraints gbc_alphaSpinnerPanel = new GridBagConstraints();
		gbc_alphaSpinnerPanel.insets = new Insets(0, 0, 5, 0);
		gbc_alphaSpinnerPanel.fill = GridBagConstraints.BOTH;
		gbc_alphaSpinnerPanel.gridx = 1;
		gbc_alphaSpinnerPanel.gridy = 3;
		add(alphaSpinnerPanel, gbc_alphaSpinnerPanel, DetailPanelMode.ADVANCED);
		alphaSpinnerPanel.setLayout(new GridLayout(0, 1, 0, 0));

		alphaSpinnerPanel.add(alphaSpinner_);
		alphaSpinner_.setModel(new SpinnerNumberModel(Settings.ALPHA_DEFAULT,
				0.0, 1.0, 0.01));
		alphaSpinner_.setEnabled(energyTypeComboBox_.getSelectedItem()
				.toString().compareTo(energyTypeStrings_[2]) == 0);

		JPanel priorShapeLabelPanel = new JPanel();
		GridBagConstraints gbc_priorShapeLabelPanel = new GridBagConstraints();
		gbc_priorShapeLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_priorShapeLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_priorShapeLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_priorShapeLabelPanel.gridx = 0;
		gbc_priorShapeLabelPanel.gridy = 4;
		add(priorShapeLabelPanel, gbc_priorShapeLabelPanel,
				DetailPanelMode.ADVANCED);
		priorShapeLabelPanel.setLayout(new GridLayout(0, 1, 0, 0));

		priorShapeLabelPanel.add(priorShapeLabel_);

		JPanel priorShapeComboBoxPanel = new JPanel();
		GridBagConstraints gbc_priorShapeComboBoxPanel = new GridBagConstraints();
		gbc_priorShapeComboBoxPanel.insets = new Insets(0, 0, 5, 0);
		gbc_priorShapeComboBoxPanel.fill = GridBagConstraints.BOTH;
		gbc_priorShapeComboBoxPanel.gridx = 1;
		gbc_priorShapeComboBoxPanel.gridy = 4;
		add(priorShapeComboBoxPanel, gbc_priorShapeComboBoxPanel,
				DetailPanelMode.ADVANCED);
		priorShapeComboBoxPanel.setLayout(new GridLayout(0, 1, 0, 0));

		priorShapeComboBoxPanel.add(priorShapeComboBox_);
		priorShapeComboBox_.setModel(new DefaultComboBoxModel(
				priorShapeStrings_));
		priorShapeComboBox_.setSelectedIndex(0);

		// shape space label
		JPanel shapeSpaceLabelPanel = new JPanel();
		GridBagConstraints gbc_shapeSpaceLabelPanel = new GridBagConstraints();
		gbc_shapeSpaceLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_shapeSpaceLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_shapeSpaceLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_shapeSpaceLabelPanel.gridx = 0;
		gbc_shapeSpaceLabelPanel.gridy = 5;
		add(shapeSpaceLabelPanel, gbc_shapeSpaceLabelPanel,
				DetailPanelMode.ADVANCED);
		shapeSpaceLabelPanel.setLayout(new GridLayout(0, 1, 0, 0));

		shapeSpaceLabelPanel.add(shapeSpaceLabel_);

		// shape space panel
		JPanel shapeSpaceComboBoxPanel = new JPanel();
		GridBagConstraints gbc_shapeSpaceComboBoxPanel = new GridBagConstraints();
		gbc_shapeSpaceComboBoxPanel.insets = new Insets(0, 0, 5, 0);
		gbc_shapeSpaceComboBoxPanel.fill = GridBagConstraints.BOTH;
		gbc_shapeSpaceComboBoxPanel.gridx = 1;
		gbc_shapeSpaceComboBoxPanel.gridy = 5;
		add(shapeSpaceComboBoxPanel, gbc_shapeSpaceComboBoxPanel,
				DetailPanelMode.ADVANCED);
		shapeSpaceComboBoxPanel.setLayout(new GridLayout(0, 1, 0, 0));

		shapeSpaceComboBoxPanel.add(shapeSpaceComboBox_);
		shapeSpaceComboBox_.setModel(new DefaultComboBoxModel(
				shapeSpaceStrings_));
		shapeSpaceComboBox_.setSelectedIndex(0);
		shapeSpaceComboBox_
		.setEnabled(priorShapeComboBox_.getSelectedIndex() != 0);
		// -------------------------------------

		JPanel priorBetaPanel = new JPanel();
		GridBagConstraints gbc_betaLabelPanel = new GridBagConstraints();
		gbc_betaLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_betaLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_betaLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_betaLabelPanel.gridx = 0;
		gbc_betaLabelPanel.gridy = 6;
		add(priorBetaPanel, gbc_betaLabelPanel, DetailPanelMode.ADVANCED);
		priorBetaPanel.setLayout(new GridLayout(0, 1, 0, 0));

		priorBetaPanel.add(betaLabel_);

		JPanel betaSpinnerPanel = new JPanel();
		GridBagConstraints gbc_betaSpinnerPanel = new GridBagConstraints();
		gbc_betaSpinnerPanel.insets = new Insets(0, 0, 5, 0);
		gbc_betaSpinnerPanel.fill = GridBagConstraints.BOTH;
		gbc_betaSpinnerPanel.gridx = 1;
		gbc_betaSpinnerPanel.gridy = 6;
		add(betaSpinnerPanel, gbc_betaSpinnerPanel, DetailPanelMode.ADVANCED);
		betaSpinnerPanel.setLayout(new GridLayout(0, 1, 0, 0));

		betaSpinnerPanel.add(betaSpinner_);
		betaSpinner_.setModel(new SpinnerNumberModel(Settings.BETA_DEFAULT,
				0.0, Double.MAX_VALUE, 0.5));
		betaSpinner_.setEnabled(priorShapeComboBox_.getSelectedIndex() != 0);

		JPanel maxIterationsLabelPanel = new JPanel();
		GridBagConstraints gbc_maxIterationsLabelPanel = new GridBagConstraints();
		gbc_maxIterationsLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_maxIterationsLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_maxIterationsLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_maxIterationsLabelPanel.gridx = 0;
		gbc_maxIterationsLabelPanel.gridy = 7;
		add(maxIterationsLabelPanel, gbc_maxIterationsLabelPanel,
				DetailPanelMode.ADVANCED);
		maxIterationsLabelPanel.setLayout(new GridLayout(0, 1, 0, 0));

		maxIterationsLabelPanel.add(maxIterationsLabel_);

		JPanel maxIterationsSpinnerPanel = new JPanel();
		GridBagConstraints gbc_maxIterationsSpinnerPanel = new GridBagConstraints();
		gbc_maxIterationsSpinnerPanel.insets = new Insets(0, 0, 5, 0);
		gbc_maxIterationsSpinnerPanel.fill = GridBagConstraints.BOTH;
		gbc_maxIterationsSpinnerPanel.gridx = 1;
		gbc_maxIterationsSpinnerPanel.gridy = 7;
		add(maxIterationsSpinnerPanel, gbc_maxIterationsSpinnerPanel,
				DetailPanelMode.ADVANCED);
		maxIterationsSpinnerPanel.setLayout(new GridLayout(0, 1, 0, 0));

		maxIterationsSpinnerPanel.add(maxIterationsSpinner_);
		maxIterationsSpinner_
		.setModel(new SpinnerNumberModel(Settings.MAX_LIFE_DEFAULT,
				new Integer(1), null, new Integer(1)));

		JPanel immortalCheckboxPanel = new JPanel();
		immortalCheckboxPanel.setBorder(null);
		GridBagConstraints gbc_immortalCheckboxPanel = new GridBagConstraints();
		gbc_immortalCheckboxPanel.fill = GridBagConstraints.BOTH;
		gbc_immortalCheckboxPanel.gridx = 1;
		gbc_immortalCheckboxPanel.gridy = 8;
		add(immortalCheckboxPanel, gbc_immortalCheckboxPanel,
				DetailPanelMode.ADVANCED);
		immortalCheckboxPanel.setLayout(new GridLayout(0, 1, 0, 0));

		immortalCheckbox_.setSelected(false);
		maxIterationsSpinner_.setEnabled(!immortalCheckbox_.isSelected());

		immortalCheckboxPanel.add(immortalCheckbox_);

		setVisibility(activeMode_);

		targetBrightnessComboBox_.addItemListener(this);
		numControlPointsSpinner_.addChangeListener(this);
		energyTypeComboBox_.addItemListener(this);
		alphaSpinner_.addChangeListener(this);
		priorShapeComboBox_.addItemListener(this);
		shapeSpaceComboBox_.addItemListener(this);
		betaSpinner_.addChangeListener(this);
		maxIterationsSpinner_.addChangeListener(this);
		immortalCheckbox_.addItemListener(this);
	}

	// ----------------------------------------------------------------------------

	/** Returns the energy tradeoff parameter introduced by the user. */
	public Double getAlpha() {
		return (Double) alphaSpinner_.getValue();
	}

	// ----------------------------------------------------------------------------

	/** Returns the energy weight associated to the prior-shape energy. */
	public Double getBeta() {
		return (Double) betaSpinner_.getValue();
	}

	// ----------------------------------------------------------------------------

	public Custom getCustomPriorShape() {
		return customPriorShape_;
	}

	// ----------------------------------------------------------------------------

	/** Returns the energy type selected by the user. */
	public ESnakeEnergyType getEnergyType() {
		switch (energyTypeComboBox_.getSelectedIndex()) {
		case 0:
			return ESnakeEnergyType.CONTOUR;
		case 1:
			return ESnakeEnergyType.REGION;
		case 2:
			return ESnakeEnergyType.MIXTURE;
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	/** Returns the maximum number of iterations introduced by the user. */
	public Integer getMaxIterations() {
		return (Integer) maxIterationsSpinner_.getValue();
	}

	// ----------------------------------------------------------------------------

	/** Returns the number of control points introduced by the user. */
	public Integer getNumControlPoints() {
		return (Integer) numControlPointsSpinner_.getValue();
	}

	// ----------------------------------------------------------------------------

	/** Returns the energy type selected by the user. */
	public ESnakePriorShapeType getPriorShapeType() {
		switch (priorShapeComboBox_.getSelectedIndex()) {
		case 0:
			return ESnakePriorShapeType.NONE;
		case 1:
			return ESnakePriorShapeType.ROD;
		case 2:
			return ESnakePriorShapeType.BUDDINGYEAST;
		case 3:
			return ESnakePriorShapeType.BANANA;
		case 4:
			return ESnakePriorShapeType.BOOMERANG;
		case 5:
			return ESnakePriorShapeType.BRAINWHITE;
		case 6:
			return ESnakePriorShapeType.CIRCLE;
		case 7:
			return ESnakePriorShapeType.CORPUSCALLOSUM;
		case 8:
			return ESnakePriorShapeType.CROSS;
		case 9:
			return ESnakePriorShapeType.ELECTRICGUITAR;
		case 10:
			return ESnakePriorShapeType.FEMUR;
		case 11:
			return ESnakePriorShapeType.FLY;
		case 12:
			return ESnakePriorShapeType.MOUSEORGAN;
		case 13:
			return ESnakePriorShapeType.SEMICIRCLE;
		case 14:
			return ESnakePriorShapeType.SQUARE;
		case 15:
			return ESnakePriorShapeType.UPPERLIP;
		case 16:
			return ESnakePriorShapeType.CUSTOM;
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	/** Returns the shape space type selected by the user. */
	public ShapeSpaceType getShapeSpaceType() {
		switch (shapeSpaceComboBox_.getSelectedIndex()) {
		case 0:
			return ShapeSpaceType.SIMILARITY;
		case 1:
			return ShapeSpaceType.AFFINE;
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	/** Returns the brightness detection mode selected by the user. */
	public ESnakeTargetType getTargetBrightness() {
		switch (targetBrightnessComboBox_.getSelectedIndex()) {
		case 0:
			return ESnakeTargetType.DARK;
		case 1:
			return ESnakeTargetType.BRIGHT;
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns <code>true</code> if the user wants to ignore the maximum number
	 * of iterations, and evolve the snake till convergence. Returns
	 * <code>false</code> otherwise.
	 */
	public boolean isImmortal() {
		return immortalCheckbox_.isSelected();
	}

	// ----------------------------------------------------------------------------

	/** Sets the snake parameters to the interface. */
	public void setSnakeParameters(ESnakeParameters snakeParameters) {
		numControlPointsSpinner_.setValue(snakeParameters.getM());
		alphaSpinner_.setValue(snakeParameters.getAlpha());
		betaSpinner_.setValue(snakeParameters.getBeta());
		maxIterationsSpinner_.setValue(snakeParameters.getMaxLife());
		immortalCheckbox_.setSelected(snakeParameters.isImmortal());
		switch (snakeParameters.getEnergyType()) {
		case CONTOUR:
			energyTypeComboBox_.setSelectedIndex(0);
			break;
		case REGION:
			energyTypeComboBox_.setSelectedIndex(1);
			break;
		case MIXTURE:
			energyTypeComboBox_.setSelectedIndex(2);
			break;
		}
		switch (snakeParameters.getDetectType()) {
		case DARK:
			targetBrightnessComboBox_.setSelectedIndex(0);
			break;
		case BRIGHT:
			targetBrightnessComboBox_.setSelectedIndex(1);
			break;
		}
		switch (snakeParameters.getPriorShapeType()) {
		case NONE:
			priorShapeComboBox_.setSelectedIndex(0);
			break;
		case ROD:
			priorShapeComboBox_.setSelectedIndex(1);
			break;
		case BUDDINGYEAST:
			priorShapeComboBox_.setSelectedIndex(2);
			break;
		case BANANA:
			priorShapeComboBox_.setSelectedIndex(3);
			break;
		case BOOMERANG:
			priorShapeComboBox_.setSelectedIndex(4);
			break;
		case BRAINWHITE:
			priorShapeComboBox_.setSelectedIndex(5);
			break;
		case CIRCLE:
			priorShapeComboBox_.setSelectedIndex(6);
			break;
		case CORPUSCALLOSUM:
			priorShapeComboBox_.setSelectedIndex(7);
			break;
		case CROSS:
			priorShapeComboBox_.setSelectedIndex(8);
			break;
		case ELECTRICGUITAR:
			priorShapeComboBox_.setSelectedIndex(8);
			break;
		case FEMUR:
			priorShapeComboBox_.setSelectedIndex(10);
			break;
		case FLY:
			priorShapeComboBox_.setSelectedIndex(11);
			break;
		case MOUSEORGAN:
			priorShapeComboBox_.setSelectedIndex(12);
			break;
		case SEMICIRCLE:
			priorShapeComboBox_.setSelectedIndex(13);
			break;
		case SQUARE:
			priorShapeComboBox_.setSelectedIndex(14);
			break;
		case UPPERLIP:
			priorShapeComboBox_.setSelectedIndex(15);
			break;
		case CUSTOM:
			priorShapeComboBox_.setSelectedIndex(16);
			break;
		default:
			break;
		}
		switch (snakeParameters.getShapeSpaceType()) {
		case SIMILARITY:
			shapeSpaceComboBox_.setSelectedIndex(0);
			break;
		case AFFINE:
			shapeSpaceComboBox_.setSelectedIndex(1);
			break;
		default:
			break;
		}
	}

	// ============================================================================
	// PRIVATE METHODS

	private void updateNumControlPointsSpinner() {
		ESnakePriorShapeType shape = getPriorShapeType();
		switch (shape) {
		case NONE:
			break;
		case ROD:
			numControlPointsSpinner_.setValue(new Rod().getMinNumberNodes());
			break;
		case BUDDINGYEAST:
			numControlPointsSpinner_.setValue(new BuddingYeast()
			.getMinNumberNodes());
			break;
		case BANANA:
			numControlPointsSpinner_.setValue(new Banana().getMinNumberNodes());
			break;
		case BOOMERANG:
			numControlPointsSpinner_.setValue(new Boomerang()
			.getMinNumberNodes());
			break;
		case BRAINWHITE:
			numControlPointsSpinner_.setValue(new BrainWhite()
			.getMinNumberNodes());
			break;
		case CIRCLE:
			break;
		case CORPUSCALLOSUM:
			numControlPointsSpinner_.setValue(new CorpusCallosum()
			.getMinNumberNodes());
			break;
		case CROSS:
			numControlPointsSpinner_.setValue(new Cross().getMinNumberNodes());
			break;
		case ELECTRICGUITAR:
			numControlPointsSpinner_.setValue(new ElectricGuitar()
			.getMinNumberNodes());
			break;
		case FEMUR:
			numControlPointsSpinner_.setValue(new Femur().getMinNumberNodes());
			break;
		case FLY:
			numControlPointsSpinner_.setValue(new Fly().getMinNumberNodes());
			break;
		case MOUSEORGAN:
			numControlPointsSpinner_.setValue(new MouseOrgan()
			.getMinNumberNodes());
			break;
		case SEMICIRCLE:
			numControlPointsSpinner_.setValue(new Semicircle()
			.getMinNumberNodes());
			break;
		case SQUARE:
			numControlPointsSpinner_.setValue(new Square().getMinNumberNodes());
			break;
		case UPPERLIP:
			numControlPointsSpinner_.setValue(new UpperLip()
			.getMinNumberNodes());
			break;
		case CUSTOM:
			numControlPointsSpinner_.setValue(customPriorShape_
					.getMinNumberNodes());
			break;
		}

	}

	// ----------------------------------------------------------------------------
	// LISTENER METHODS

	/** Handles the events of the spinners. */
	@Override
	public void stateChanged(ChangeEvent e) {
		Object source = e.getSource();
		if (source == numControlPointsSpinner_ || source == alphaSpinner_
				|| source == betaSpinner_ || source == maxIterationsSpinner_) {
			bigSnake_.loadSnakeParametersFromInterface();
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Handles the events of the combo boxes and the checkbox. It fires an
	 * update of the active snake.
	 */
	@Override
	public void itemStateChanged(ItemEvent e) {
		if (e.getSource() == targetBrightnessComboBox_) {
			if (e.getStateChange() == ItemEvent.SELECTED) {
				bigSnake_.loadSnakeParametersFromInterface();
			}
		} else if (e.getSource() == energyTypeComboBox_) {
			if (e.getStateChange() == ItemEvent.SELECTED) {
				alphaSpinner_.setEnabled(energyTypeComboBox_.getSelectedItem()
						.toString().compareTo(energyTypeStrings_[2]) == 0);
				bigSnake_.loadSnakeParametersFromInterface();
			}
		} else if (e.getSource() == priorShapeComboBox_) {
			if (e.getStateChange() == ItemEvent.SELECTED) {
				betaSpinner_
				.setEnabled(priorShapeComboBox_.getSelectedIndex() != 0);
				shapeSpaceComboBox_.setEnabled(priorShapeComboBox_
						.getSelectedIndex() != 0);

				if (priorShapeComboBox_.getSelectedIndex() == 14) {
					try {
						customPriorShape_ = bigSnake_.getIOXMLUtils()
								.loadPriorShapeFromXML();
					} catch (Exception e1) {
						priorShapeComboBox_.setSelectedIndex(0);
						betaSpinner_.setEnabled(false);
						new AnnounceFrame("File not loaded.");
						return;
					}
					if (customPriorShape_ != null) {
						updateNumControlPointsSpinner();
						bigSnake_.loadSnakeParametersFromInterface();
					} else {
						priorShapeComboBox_.setSelectedIndex(0);
						betaSpinner_.setEnabled(false);
						new AnnounceFrame("File not loaded.");
						return;
					}
				} else {
					updateNumControlPointsSpinner();
					bigSnake_.loadSnakeParametersFromInterface();
				}
			}
		} else if (e.getSource() == shapeSpaceComboBox_) {
			if (e.getStateChange() == ItemEvent.SELECTED) {
				bigSnake_.loadSnakeParametersFromInterface();
			}
		} else if (e.getSource() == immortalCheckbox_) {
			maxIterationsSpinner_.setEnabled(!immortalCheckbox_.isSelected());
			bigSnake_.loadSnakeParametersFromInterface();
		}
	}
}
