/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnake.keeper;

import icy.roi.ROIEvent;
import icy.roi.ROIListener;
import icy.sequence.Sequence;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import plugins.big.bigsnake.BIGSnake;
import plugins.big.bigsnake.roi.ROI2DSnake;
import plugins.big.bigsnake.roi.SnakeEditMode;
import plugins.big.bigsnake.snake.ESnake;
import plugins.big.bigsnake.snake.ESnakeParameters;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DOptimizer;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DPowellOptimizer;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Custom;

/**
 * Class that takes care of the synchronization of the snake optimization and
 * the display.
 * 
 * @version May 3, 2014
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Masih Nilchian (masih_n85@yahoo.com)
 */
public class SnakeKeeper implements ROIListener, Observer {

	/** Pointer to the main class. */
	private BIGSnake mainPlugin_ = null;

	// ----------------------------------------------------------------------------
	// IMAGE-RELATIED FIELDS

	/** Image sequence where the snake lives. */
	private Sequence sequence_ = null;
	/**
	 * Flag that determines if the current <code>SnakeKeeper</code> is attached
	 * to a particular sequence of not.
	 */
	private boolean isAttachedToSequence_ = false;

	// ----------------------------------------------------------------------------
	// SNAKE-RELATIED FIELDS

	/** Unique identifier of the <code>SnakeKeeper</code>. */
	private String ID = null;
	/** The snake associated to this <code>SnakeKeeper</code>. */
	public ESnake snake_ = null;
	/** The ROI managing the snake-defining control points. */
	private ROI2DSnake roiNodes_ = null;

	// ----------------------------------------------------------------------------
	// OPTIMIZATION-RELATIED FIELDS

	/** Thread where the optimization of the snake takes place. */
	private OptimizationThread optimizationThread_ = null;
	/**
	 * Flag that determines if the current <code>SnakeKeeper</code> is
	 * optimizing the current snake not.
	 */
	private boolean isOptimizing_ = false;

	// ----------------------------------------------------------------------------
	// LOCKS

	/** Lock used to ensure that the snake is optimized as an atomic action. */
	private final Lock isOptimizingLock_ = new ReentrantLock();
	/** Lock used to ensure that the snake is modified as an atomic action. */
	private final Lock snakeLock_ = new ReentrantLock();
	/** Lock used to ensure that the ROI is modified as an atomic action. */
	private final Lock roiLock_ = new ReentrantLock();

	// ----------------------------------------------------------------------------
	// OTHER

	/** Array with the running threads related to the update of the interface. */
	private final ArrayList<Thread> runningThreadList_ = new ArrayList<Thread>();

	// ============================================================================
	// PUBLIC METHODS

	/** Default. */
	public SnakeKeeper(Sequence sequence, ESnake snake, BIGSnake mainPlugin) {
		if (sequence == null) {
			System.err.println("sequence is null");
			return;
		}
		if (snake == null) {
			System.err.println("snake is null");
			return;
		}
		sequence_ = sequence;
		isAttachedToSequence_ = true;
		mainPlugin_ = mainPlugin;

		snakeLock_.lock();
		snake_ = snake;
		snakeLock_.unlock();

		roiLock_.lock();
		roiNodes_ = new ROI2DSnake(snake_, this);
		roiLock_.unlock();

		sequence.addROI(roiNodes_);
		roiNodes_.addListener(this);
		sequence_.roiChanged();
	}

	// ----------------------------------------------------------------------------

	/** Activates the snake associate to this <code>SnakeKeeper</code>. */
	public void activateSnake() {
		mainPlugin_.activateSnake(this);
	}

	// ----------------------------------------------------------------------------

	/** Deactivates the snake associate to this <code>SnakeKeeper</code>. */
	public void deactivateSnake() {
		mainPlugin_.deactivateSnake(this);
	}

	// ----------------------------------------------------------------------------

	public Custom getCustomPriorShape() {
		return snake_.getCustomPriorShape();
	}

	// ----------------------------------------------------------------------------

	public ESnakeParameters getESnakeParameters() {
		return snake_.getSnakeParameters();
	}

	// ----------------------------------------------------------------------------

	/** Returns the unique identifier of this <code>SnakeKeeper</code>. */
	public String getID() {
		return ID;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a copy of the snake-defining nodes of this
	 * <code>SnakeKeeper</code>.
	 */
	public Snake2DNode[] getNodesCopy() {
		Snake2DNode[] snakeNodes = snake_.getNodes();
		Snake2DNode[] snakeNodesCopy = new Snake2DNode[snakeNodes.length];
		for (int i = 0; i < snakeNodes.length; i++) {
			snakeNodesCopy[i] = new Snake2DNode(snakeNodes[i].x,
					snakeNodes[i].y, snakeNodes[i].isFrozen(),
					snakeNodes[i].isHidden());
		}
		return snakeNodesCopy;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the <code>Sequence</code> object to which the keeper is attached
	 * to.
	 */
	public Sequence getSequence() {
		return sequence_;
	}

	// ----------------------------------------------------------------------------

	/** Removes the ROI from the <code>Sequence</code>. */
	public void removeFromSequence() {
		if (isAttachedToSequence_) {
			isAttachedToSequence_ = false;
			sequence_.removeROI(roiNodes_);
		}
	}

	// ----------------------------------------------------------------------------

	@Override
	public void roiChanged(ROIEvent event) {
		if (!isOptimizing_) {
			refreshSnakeFromViewer();
		}
	}

	// ----------------------------------------------------------------------------

	public void setID(String id) {
		ID = id;
		roiLock_.lock();
		try {
			roiNodes_.setName(ID);
		} finally {
			roiLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	public void setSelected(boolean selected) {
		roiNodes_.setEditable(selected);
	}

	// ----------------------------------------------------------------------------

	public void setSnakeEditMode(SnakeEditMode editingMode) {
		roiLock_.lock();
		try {
			roiNodes_.setEditMode(editingMode);
		} finally {
			roiLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	/** Updates the parameters of the snake. */
	public void setSnakeParameters(final ESnakeParameters snakeParameters,
			final Custom customPriorShape) {
		Thread thr = new Thread() {
			@Override
			public void run() {
				if (snakeLock_.tryLock()) {
					try {
						snake_.setCustomPriorShape(customPriorShape);
						snake_.setSnakeParameters(snakeParameters);
					} finally {
						snakeLock_.unlock();
					}
				}
				if (roiLock_.tryLock()) {
					try {
						roiNodes_.refreshROI();
					} finally {
						roiLock_.unlock();
					}
				}
			}
		};
		runningThreadList_.add(thr);
		thr.start();
	}

	// ----------------------------------------------------------------------------

	/** Applies a shift on the position of the snake. */
	public void shiftSnake(final int dx, final int dy) {
		Thread thr = new Thread() {
			@Override
			public void run() {
				if (snakeLock_.tryLock()) {
					try {
						Snake2DNode[] nodes = snake_.getNodes();
						for (int i = 0; i < nodes.length; i++) {
							nodes[i].x += dx;
							nodes[i].y += dy;
						}
						snake_.setNodes(nodes);
					} finally {
						snakeLock_.unlock();
					}
				}
				if (roiLock_.tryLock()) {
					try {
						roiNodes_.refreshROI();
					} finally {
						roiLock_.unlock();
					}
				}
			}
		};
		runningThreadList_.add(thr);
		thr.start();
	}

	// ----------------------------------------------------------------------------

	/** Launches the optimization procedure in a new thread. */
	public void startOptimization() {
		if (isAttachedToSequence_) {
			isOptimizingLock_.lock();
			try {
				if (!isOptimizing_) {
					isOptimizing_ = true;
					optimizationThread_ = new OptimizationThread();
					optimizationThread_.setPriority(Thread.MIN_PRIORITY);
					optimizationThread_.start();
				}
			} finally {
				isOptimizingLock_.unlock();
			}
		}
	}

	// ----------------------------------------------------------------------------

	/** Stops the optimization procedure. */
	public void stopOptimization() {
		if (optimizationThread_ != null) {
			optimizationThread_.optimizer_.stopOptimizing();
		}
	}

	// ----------------------------------------------------------------------------

	/** Update the ROIs from the scales. */
	@Override
	public void update(Observable observable, Object object) {
		if (observable instanceof Snake2DOptimizer) {
			if (((Snake2DOptimizer) observable).isCurrentBest) {
				refreshViewerFromSnake();
			}
		}
	}

	// ============================================================================
	// PRIVATE METHODS

	private void refreshViewerFromSnake() {
		Thread thr = new Thread() {
			@Override
			public void run() {
				Snake2DNode[] nodes = snake_.getNodes();
				double[] xPositions = new double[nodes.length];
				double[] yPositions = new double[nodes.length];
				for (int i = 0; i < nodes.length; i++) {
					xPositions[i] = nodes[i].x;
					yPositions[i] = nodes[i].y;
				}
				if (roiLock_.tryLock()) {
					try {
						roiNodes_.changePosition(xPositions, yPositions,
								snake_.getScales());
					} finally {
						roiLock_.unlock();
					}
				}
			}
		};
		runningThreadList_.add(thr);
		thr.start();
	}

	// ----------------------------------------------------------------------------

	private void refreshSnakeFromViewer() {
		Thread thr = new Thread() {
			@Override
			public void run() {
				if (snakeLock_.tryLock()) {
					try {
						if (roiLock_.tryLock()) {
							try {
								Snake2DNode[] nodes = new Snake2DNode[snake_
										.getNumNodes()];
								for (int i = 0; i < nodes.length; i++) {
									Point2D p = roiNodes_.getControlPoint(i);
									if (p != null) {
										nodes[i] = new Snake2DNode(p.getX(),
												p.getY());
									}
								}
								snake_.setNodes(nodes);
								roiNodes_.refreshScales();
								runningThreadList_.remove(this);
							} finally {
								roiLock_.unlock();
							}
						}
					} finally {
						snakeLock_.unlock();
					}
				}
			}
		};
		runningThreadList_.add(thr);
		thr.start();
	}

	// ============================================================================
	// OPTIMIZATION LAUNCHER INNER CLASS

	/**
	 * Class that takes care of the optimization of the snake.
	 * 
	 * @version November 5, 2013
	 * 
	 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
	 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
	 */
	private class OptimizationThread extends Thread {
		public Snake2DOptimizer optimizer_ = new Snake2DPowellOptimizer();

		@Override
		public void run() {
			roiNodes_.setEditable(false);
			snakeLock_.lock();

			final Snake2DNode[] youngSnake = snake_.getNodes();
			final int K = youngSnake.length;
			final Snake2DNode[] X = new Snake2DNode[K];
			for (int k = 0; k < K; k++) {
				X[k] = new Snake2DNode(youngSnake[k].x, youngSnake[k].y,
						youngSnake[k].isFrozen(), youngSnake[k].isHidden());
			}
			snake_.setNodes(X);
			snakeLock_.unlock();
			optimizer_.addObserver(SnakeKeeper.this);
			snake_.reviveSnake();
			optimizer_.optimize(snake_, X);

			isOptimizing_ = false;
			refreshViewerFromSnake();
			if (mainPlugin_.isActiveSnake(SnakeKeeper.this)) {
				roiNodes_.setEditable(true);
			}
		}
	}
}
