/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnake.snake;

import icy.util.XMLUtil;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import plugins.big.bigsnake.core.Settings;

/**
 * Class that wraps the parameters of the E-Snake.
 * 
 * @version October 27, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Daniel Schmitter (daniel.schmitter@epfl.ch)
 */
public class ESnakeParameters {

	/** Indicates the type of features to detect (bright or dark). */
	private ESnakeTargetType detectType_ = Settings.TARGET_TYPE_DEFAULT;
	/** Number of spline vector coefficients. */
	private int M_ = Settings.M_DEFAULT;
	/** Indicates the energy function of the snake. */
	private ESnakeEnergyType energyType_ = Settings.ENERGY_TYPE_DEFAULT;
	/** Energy tradeoff factor. */
	private double alpha_ = Settings.ALPHA_DEFAULT;
	/** Prior shape information. */
	private ESnakePriorShapeType priorShapeType_ = Settings.PRIOR_SHAPE_DEFAULT;
	/** Shape space information. */
	private ShapeSpaceType shapeSpaceType_ = Settings.SHAPE_SPACE_DEFAULT;
	/** Prior-shape energy tradeoff factor. */
	private double beta_ = Settings.BETA_DEFAULT;
	/**
	 * Maximum number of iterations allowed when the <code>immortal</code> flag
	 * is <code>false</code>.
	 */
	private int maxLife_ = Settings.MAX_LIFE_DEFAULT;
	/**
	 * If <code>true</code> indicates that the snake will keep iterating till
	 * the optimizer decides so.
	 */
	private boolean immortal_ = Settings.IMMORTAL_DEFAULT;

	// ----------------------------------------------------------------------------
	// XML TAG FIELDS

	/**
	 * Label of the XML tag informing about the brightness of the target to
	 * segment (brighter or darker than the background).
	 */
	public static final String ID_DETECT_TYPE = "detectType";
	/**
	 * Label of the XML tag containing the number of control points of the
	 * snake.
	 */
	public static final String ID_M = "M";
	/**
	 * Label of the XML tag informing about the type of energy the snake uses
	 * (contour, region or mixture).
	 */
	public static final String ID_ENERGY_TYPE = "energyType";
	/**
	 * Label of the XML tag containing the energy trade-off between the contour
	 * energy and the region energy.
	 */
	public static final String ID_ALPHA = "alpha";
	/**
	 * Label of the XML tag informing about the type of prior shape that is used
	 * in the energy.
	 */
	public static final String ID_PRIOR_SHAPE_TYPE = "priorShapeType";
	/**
	 * Label of the XML tag informing about the type of shape space that is used
	 * in the energy.
	 */
	public static final String ID_SHAPE_SPACE_TYPE = "priorShapeType";
	/**
	 * Label of the XML tag containing the energy weight of the prior shape
	 * energy.
	 */
	public static final String ID_BETA = "beta";
	/** Label of the XML tag containing the maximum number of iterations. */
	public static final String ID_MAX_LIFE = "maxLife";
	/**
	 * Label of the XML tag informing if the number of iterations of the snake
	 * is limited or not.
	 */
	public static final String ID_IMMORTAL = "immortal";

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public ESnakeParameters() {
	}

	// ----------------------------------------------------------------------------

	/** Constructor. */
	public ESnakeParameters(ESnakeTargetType detectType, int M,
			ESnakeEnergyType energyType, double alpha,
			ESnakePriorShapeType priorShapeType, ShapeSpaceType shapeSpaceType,
			double beta, int maxLife, boolean immortal) {
		setMaxLife(maxLife);
		setM(M);
		setAlpha(alpha);
		setImmortal(immortal);
		setDetectType(detectType);
		setEnergyType(energyType);
		setPriorShapeType(priorShapeType);
		setShapeSpaceType(shapeSpaceType);
		setBeta(beta);
	}

	// ----------------------------------------------------------------------------

	@Override
	public String toString() {
		return new String("[Snake parameters: " + ID_DETECT_TYPE + " "
				+ detectType_ + ID_M + " " + M_ + ID_ENERGY_TYPE + " "
				+ energyType_ + ID_ALPHA + " " + alpha_ + ID_PRIOR_SHAPE_TYPE
				+ " " + priorShapeType_ + ID_SHAPE_SPACE_TYPE + " "
				+ shapeSpaceType_ + ID_BETA + " " + beta_ + ID_MAX_LIFE + " "
				+ maxLife_ + ID_IMMORTAL + " " + immortal_ + "]");
	}

	// ----------------------------------------------------------------------------

	public void loadFromToXML(Node node) {

		Element detectTypeAsStringElement = XMLUtil.getElement(node,
				ID_DETECT_TYPE);
		if (detectTypeAsStringElement != null) {
			String detectTypeAsString = detectTypeAsStringElement
					.getTextContent();
			if (detectTypeAsString.compareToIgnoreCase("BRIGHT") == 0) {
				setDetectType(ESnakeTargetType.BRIGHT);
			} else if (detectTypeAsString.compareToIgnoreCase("DARK") == 0) {
				setDetectType(ESnakeTargetType.DARK);
			} else {
				setDetectType(Settings.TARGET_TYPE_DEFAULT);
			}
		} else {
			setDetectType(Settings.TARGET_TYPE_DEFAULT);
		}

		Element energyTypeAsStringElement = XMLUtil.getElement(node,
				ID_ENERGY_TYPE);
		if (energyTypeAsStringElement != null) {
			String energyTypeAsString = energyTypeAsStringElement
					.getTextContent();
			if (energyTypeAsString.compareToIgnoreCase("CONTOUR") == 0) {
				setEnergyType(ESnakeEnergyType.CONTOUR);
			} else if (energyTypeAsString.compareToIgnoreCase("REGION") == 0) {
				setEnergyType(ESnakeEnergyType.REGION);
			} else if (energyTypeAsString.compareToIgnoreCase("MIXTURE") == 0) {
				setEnergyType(ESnakeEnergyType.MIXTURE);
			} else {
				setEnergyType(Settings.ENERGY_TYPE_DEFAULT);
			}
		} else {
			setEnergyType(Settings.ENERGY_TYPE_DEFAULT);
		}

		Element priorShapeTypeAsStringElement = XMLUtil.getElement(node,
				ID_PRIOR_SHAPE_TYPE);
		if (priorShapeTypeAsStringElement != null) {
			String priorShapeTypeAsString = priorShapeTypeAsStringElement
					.getTextContent();
			if (priorShapeTypeAsString.compareToIgnoreCase("NONE") == 0) {
				setPriorShapeType(ESnakePriorShapeType.NONE);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("BANANA") == 0) {
				setPriorShapeType(ESnakePriorShapeType.BANANA);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("BOOMERANG") == 0) {
				setPriorShapeType(ESnakePriorShapeType.BOOMERANG);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("BRAINWHITE") == 0) {
				setPriorShapeType(ESnakePriorShapeType.BRAINWHITE);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("CIRCLE") == 0) {
				setPriorShapeType(ESnakePriorShapeType.CIRCLE);
			} else if (priorShapeTypeAsString
					.compareToIgnoreCase("CORPUSCALLOSUM") == 0) {
				setPriorShapeType(ESnakePriorShapeType.CORPUSCALLOSUM);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("CROSS") == 0) {
				setPriorShapeType(ESnakePriorShapeType.CROSS);
			} else if (priorShapeTypeAsString
					.compareToIgnoreCase("ELECTRICGUITAR") == 0) {
				setPriorShapeType(ESnakePriorShapeType.ELECTRICGUITAR);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("FEMUR") == 0) {
				setPriorShapeType(ESnakePriorShapeType.FEMUR);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("FLY") == 0) {
				setPriorShapeType(ESnakePriorShapeType.FLY);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("MOUSEORGAN") == 0) {
				setPriorShapeType(ESnakePriorShapeType.MOUSEORGAN);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("SEMICIRCLE") == 0) {
				setPriorShapeType(ESnakePriorShapeType.SEMICIRCLE);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("SQUARE") == 0) {
				setPriorShapeType(ESnakePriorShapeType.SQUARE);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("UPPERLIP") == 0) {
				setPriorShapeType(ESnakePriorShapeType.UPPERLIP);
			} else if (priorShapeTypeAsString.compareToIgnoreCase("CUSTOM") == 0) {
				setPriorShapeType(ESnakePriorShapeType.CUSTOM);
			} else {
				setPriorShapeType(Settings.PRIOR_SHAPE_DEFAULT);
			}
		} else {
			setPriorShapeType(Settings.PRIOR_SHAPE_DEFAULT);
		}

		Element shapeSpaceTypeAsStringElement = XMLUtil.getElement(node,
				ID_SHAPE_SPACE_TYPE);
		if (shapeSpaceTypeAsStringElement != null) {
			String shapeSpaceTypeAsString = shapeSpaceTypeAsStringElement
					.getTextContent();
			if (shapeSpaceTypeAsString.compareToIgnoreCase("SIMILARITY") == 0) {
				setShapeSpaceType(ShapeSpaceType.SIMILARITY);
			} else if (shapeSpaceTypeAsString.compareToIgnoreCase("AFFINE") == 0) {
				setShapeSpaceType(ShapeSpaceType.AFFINE);
			}
		} else {
			setShapeSpaceType(Settings.SHAPE_SPACE_DEFAULT);
		}

		setMaxLife(XMLUtil.getElementIntValue(node, ID_MAX_LIFE,
				Settings.MAX_LIFE_DEFAULT));
		setM(XMLUtil.getElementIntValue(node, ID_M, Settings.M_DEFAULT));
		setAlpha(XMLUtil.getElementDoubleValue(node, ID_ALPHA,
				Settings.ALPHA_DEFAULT));
		setBeta(XMLUtil.getElementDoubleValue(node, ID_BETA,
				Settings.BETA_DEFAULT));
		setImmortal(XMLUtil.getElementBooleanValue(node, ID_IMMORTAL,
				Settings.IMMORTAL_DEFAULT));
	}

	// ----------------------------------------------------------------------------

	public void saveToXML(Element node) {
		switch (detectType_) {
		case BRIGHT:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_DETECT_TYPE,
					"BRIGHT");
			break;
		case DARK:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_DETECT_TYPE,
					"DARK");
			break;
		}
		XMLUtil.setElementIntValue(node, ESnakeParameters.ID_M, M_);
		switch (energyType_) {
		case CONTOUR:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_ENERGY_TYPE,
					"CONTOUR");
			break;
		case REGION:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_ENERGY_TYPE,
					"REGION");
			break;
		case MIXTURE:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_ENERGY_TYPE,
					"MIXTURE");
			break;
		}
		XMLUtil.setElementDoubleValue(node, ESnakeParameters.ID_ALPHA, alpha_);
		switch (priorShapeType_) {
		case NONE:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"NONE");
			break;
		case BANANA:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"BANANA");
			break;
		case BOOMERANG:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"BOOMERANG");
			break;
		case BRAINWHITE:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"BRAINWHITE");
			break;
		case CIRCLE:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"CIRCLE");
			break;
		case CORPUSCALLOSUM:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"CORPUSCALLOSUM");
			break;
		case CROSS:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"CROSS");
			break;
		case ELECTRICGUITAR:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"ELECTRICGUITAR");
			break;
		case FEMUR:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"FEMUR");
			break;
		case FLY:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"FLY");
			break;
		case MOUSEORGAN:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"MOUSEORGAN");
			break;
		case SEMICIRCLE:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"SEMICIRCLE");
			break;
		case SQUARE:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"SQUARE");
			break;
		case UPPERLIP:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"UPPERLIP");
			break;
		case CUSTOM:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_PRIOR_SHAPE_TYPE,
					"CUSTOM");
			break;
		default:
			break;
		}
		switch (shapeSpaceType_) {
		case SIMILARITY:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_SHAPE_SPACE_TYPE,
					"SIMILARITY");
			break;
		case AFFINE:
			XMLUtil.setElementValue(node, ESnakeParameters.ID_SHAPE_SPACE_TYPE,
					"AFFINE");
			break;
		default:
			break;
		}
		XMLUtil.setElementDoubleValue(node, ESnakeParameters.ID_BETA, beta_);
		XMLUtil.setElementDoubleValue(node, ESnakeParameters.ID_MAX_LIFE,
				maxLife_);
		XMLUtil.setElementBooleanValue(node, ESnakeParameters.ID_IMMORTAL,
				immortal_);
	}

	// ============================================================================
	// GETTERS AND SETTERS

	public ESnakeTargetType getDetectType() {
		return detectType_;
	}

	// ----------------------------------------------------------------------------

	public void setDetectType(ESnakeTargetType detectType) {
		detectType_ = detectType;
	}

	// ----------------------------------------------------------------------------

	public int getM() {
		return M_;
	}

	// ----------------------------------------------------------------------------

	public void setM(int M) {
		M_ = M;
	}

	// ----------------------------------------------------------------------------

	public ESnakeEnergyType getEnergyType() {
		return energyType_;
	}

	// ----------------------------------------------------------------------------

	public void setEnergyType(ESnakeEnergyType energyType) {
		energyType_ = energyType;
	}

	// ----------------------------------------------------------------------------

	public double getAlpha() {
		return alpha_;
	}

	// ----------------------------------------------------------------------------

	public void setAlpha(double alpha) {
		alpha_ = alpha;
	}

	// ----------------------------------------------------------------------------

	public ESnakePriorShapeType getPriorShapeType() {
		return priorShapeType_;
	}

	// ----------------------------------------------------------------------------

	public void setPriorShapeType(ESnakePriorShapeType priorShapeType) {
		priorShapeType_ = priorShapeType;
	}

	// ----------------------------------------------------------------------------

	public ShapeSpaceType getShapeSpaceType() {
		return shapeSpaceType_;
	}

	// ----------------------------------------------------------------------------

	public void setShapeSpaceType(ShapeSpaceType shapeSpaceType) {
		shapeSpaceType_ = shapeSpaceType;
	}

	// ----------------------------------------------------------------------------

	public double getBeta() {
		return beta_;
	}

	// ----------------------------------------------------------------------------

	public void setBeta(double beta) {
		beta_ = beta;
	}

	// ----------------------------------------------------------------------------

	public int getMaxLife() {
		return maxLife_;
	}

	// ----------------------------------------------------------------------------

	public void setMaxLife(int maxLife) {
		maxLife_ = maxLife;
	}

	// ----------------------------------------------------------------------------

	public boolean isImmortal() {
		return immortal_;
	}

	// ----------------------------------------------------------------------------

	public void setImmortal(boolean immortal) {
		immortal_ = immortal;
	}
}
