/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 ******************************************************************************/
package plugins.big.shapedesigner.keeper;

import icy.roi.ROIEvent;
import icy.roi.ROIListener;
import icy.sequence.Sequence;

import java.util.ArrayList;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.shapedesigner.ShapeDesigner;
import plugins.big.shapedesigner.roi.SplineCurve2DROI;
import plugins.big.shapedesigner.roi.SplineCurveEditMode;
import plugins.big.shapedesigner.splinecurve.SplineCurve;
import plugins.big.shapedesigner.splinecurve.SplineCurveParameters;

/**
 * Class that takes care of the synchronization of the spline curve display.
 * 
 * @version May 3, 2014
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Masih Nilchian (masih_n85@yahoo.com)
 */
public class SplineCurveKeeper implements ROIListener {

	/** Pointer to the main class. */
	private ShapeDesigner mainPlugin_ = null;

	// ----------------------------------------------------------------------------
	// IMAGE-RELATIED FIELDS

	/** Image sequence where the spline curve lives. */
	private Sequence sequence_ = null;
	/**
	 * Flag that determines if the current <code>SplineCurveKeeper</code> is
	 * attached to a particular sequence of not.
	 */
	private boolean isAttachedToSequence_ = false;

	// ----------------------------------------------------------------------------
	// SPLINE CURVE-RELATIED FIELDS

	/** Unique identifier of the <code>SplineCurveKeeper</code>. */
	private final String ID = null;
	/** The spline curve associated to this <code>SplineCurveKeeper</code>. */
	public SplineCurve splineCurve_ = null;
	/** The ROI managing the spline curve-defining control points. */
	private SplineCurve2DROI roiNodes_ = null;

	// ----------------------------------------------------------------------------
	// LOCKS

	/**
	 * Lock used to ensure that the spline curve is modified as an atomic
	 * action.
	 */
	private final Lock splineCurveLock_ = new ReentrantLock();
	/** Lock used to ensure that the ROI is modified as an atomic action. */
	private final Lock roiLock_ = new ReentrantLock();

	// ----------------------------------------------------------------------------
	// OTHER

	/** Array with the running threads related to the update of the interface. */
	private final ArrayList<Thread> runningThreadList_ = new ArrayList<Thread>();

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public SplineCurveKeeper(Sequence sequence, SplineCurve splineCurve,
			ShapeDesigner mainPlugin) {
		if (sequence == null) {
			System.err.println("sequence is null");
			return;
		}
		if (splineCurve == null) {
			System.err.println("splineCurve is null");
			return;
		}
		sequence_ = sequence;
		isAttachedToSequence_ = true;
		mainPlugin_ = mainPlugin;

		splineCurveLock_.lock();
		splineCurve_ = splineCurve;
		splineCurveLock_.unlock();

		roiLock_.lock();
		roiNodes_ = new SplineCurve2DROI(splineCurve_, this);
		roiLock_.unlock();

		sequence.addROI(roiNodes_);
		roiNodes_.addListener(this);
		sequence_.roiChanged();
	}

	// ----------------------------------------------------------------------------

	public void activateSplineCurve() {
		mainPlugin_.activateSplineCurve(this);
	}

	// ----------------------------------------------------------------------------

	public void deactivateSplineCurve() {
		mainPlugin_.deactivateSplineCurve(this);
	}

	// ----------------------------------------------------------------------------

	public String getID() {
		return ID;
	}

	// ----------------------------------------------------------------------------

	public Snake2DNode[] getNodesCopy() {
		Snake2DNode[] splineCurveNodes = splineCurve_.getNodes();
		Snake2DNode[] splineCurveNodesCopy = new Snake2DNode[splineCurveNodes.length];
		for (int i = 0; i < splineCurveNodes.length; i++) {
			splineCurveNodesCopy[i] = new Snake2DNode(splineCurveNodes[i].x,
					splineCurveNodes[i].y, splineCurveNodes[i].isFrozen(),
					splineCurveNodes[i].isHidden());
		}
		return splineCurveNodesCopy;
	}

	// ----------------------------------------------------------------------------

	public SplineCurveParameters getSplineCurveParameters() {
		return splineCurve_.getSplineCurveParameters();
	}

	// ----------------------------------------------------------------------------

	public void removeFromSequence() {
		if (isAttachedToSequence_) {
			isAttachedToSequence_ = false;
			sequence_.removeROI(roiNodes_);
		}
	}

	// ----------------------------------------------------------------------------

	@Override
	public void roiChanged(ROIEvent event) {
		refreshSplineCurveFromViewer();
	}

	// ----------------------------------------------------------------------------

	public void setID(String id) {
		roiLock_.lock();
		try {
			roiNodes_.setName(ID);
		} finally {
			roiLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	public void setSelected(boolean selected) {
		roiNodes_.setEditable(selected);
	}

	// ----------------------------------------------------------------------------

	public void setSplineCurveEditMode(SplineCurveEditMode editingMode) {
		roiLock_.lock();
		try {
			roiNodes_.setEditMode(editingMode);
		} finally {
			roiLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	/** Updates the parameters of the spline curve. */
	public void setSplineCurveParameters(
			final SplineCurveParameters splineCurveParameters) {
		Thread thr = new Thread() {
			@Override
			public void run() {
				if (splineCurveLock_.tryLock()) {
					try {
						splineCurve_
								.setSplineCurveParameters(splineCurveParameters);
					} finally {
						splineCurveLock_.unlock();
					}
				}
				if (roiLock_.tryLock()) {
					try {
						roiNodes_.refreshROI();
					} finally {
						roiLock_.unlock();
					}
				}
			}
		};
		runningThreadList_.add(thr);
		thr.start();
	}

	// ----------------------------------------------------------------------------

	public void setSymmetric(boolean symmetric) {
		roiLock_.lock();
		try {
			roiNodes_.setSymmetric(symmetric);
		} finally {
			roiLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	/** Applies a shift on the position of the spline curve. */
	public void shiftSplineCurve(final int dx, final int dy) {
		Thread thr = new Thread() {
			@Override
			public void run() {
				if (splineCurveLock_.tryLock()) {
					try {
						Snake2DNode[] nodes = splineCurve_.getNodes();
						for (int i = 0; i < nodes.length; i++) {
							nodes[i].x += dx;
							nodes[i].y += dy;
						}
						splineCurve_.setNodes(nodes);
					} finally {
						splineCurveLock_.unlock();
					}
				}
				if (roiLock_.tryLock()) {
					try {
						roiNodes_.refreshROI();
					} finally {
						roiLock_.unlock();
					}
				}
			}
		};
		runningThreadList_.add(thr);
		thr.start();
	}

	// ============================================================================
	// PRIVATE METHODS

	private void refreshSplineCurveFromViewer() {
		Thread thr = new Thread() {
			@Override
			public void run() {
				if (splineCurveLock_.tryLock()) {
					try {
						if (roiLock_.tryLock()) {
							try {
								Snake2DNode[] nodes = new Snake2DNode[splineCurve_
										.getNumNodes()];
								for (int i = 0; i < nodes.length; i++) {
									nodes[i] = new Snake2DNode(roiNodes_
											.getControlPoint(i).getX(),
											roiNodes_.getControlPoint(i).getY());
								}
								// recompute outline
								splineCurve_.setNodes(nodes);
								roiNodes_.refreshScales();
								runningThreadList_.remove(this);
							} finally {
								roiLock_.unlock();
							}
						}
					} finally {
						splineCurveLock_.unlock();
					}
				}
			}
		};
		runningThreadList_.add(thr);
		thr.start();
	}
}
