/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 *     Daniel Schmitter (daniel.schmitter@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.process.process1D;

/**
 * Implements several basis B-spline basis functions.
 * 
 * @version February 7, 2015
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Daniel Schmitter (daniel.schmitter@epfl.ch)
 * @author Anais Badoual (anais.badoual@gmail.com)
 */
public class BSplineBasis {

	// ----------------------------------------------------------------------------
	// BASIS TYPES

	/** Type of B-spline basis functions. */
	public enum BSplineBasisType {
		ESPLINE3, ESPLINE4, LINEARBSPLINE, QUADRATICBSPLINE, CUBICBSPLINE, MSPLINE, ASPLINE, CERCLE
	}

	// ----------------------------------------------------------------------------
	// BASIS FUNCTIONS' SUPPORT

	/** Length of the support of the linear B-spline. */
	public static int LINEARBSPLINESUPPORT = 2;
	/** Length of the support of the quadratic B-spline. */
	public static int QUADRATICBSPLINESUPPORT = 3;
	/** Length of the support of the cubic B-spline. */
	public static int CUBICBSPLINESUPPORT = 4;

	/**
	 * Length of the support of the exponential B-spline basis function with
	 * three roots.
	 */
	public static int ESPLINE3SUPPORT = 3;
	/**
	 * Length of the support of the exponential B-spline basis function with
	 * four roots.
	 */
	public static int ESPLINE4SUPPORT = 4;
	/**
	 * Length of the support of the M-spline.
	 */
	public static int MSPLINESUPPORT = 4;
	
	/**
	 * Length of the support of the A-spline.
	 */
	public static int ASPLINESUPPORT = 4;

	// ----------------------------------------------------------------------------
	// OTHER

	/** Pi. */
	private static double PI = Math.PI;

	// ============================================================================
	// PUBLIC METHODS

	// ----------------------------------------------------------------------------
	// B-SPLINE FUNCTIONS

	/** Causal constant B-spline. */
	public static double ConstantBSpline(double t) {
		double SplineValue = 0.0;
		t -= 0.5;
		if ((t > -0.5) && (t < -0.5)) {
			SplineValue = 1;
		} else if (t == 0.5 || t == -0.5) {
			SplineValue = 0.5;
		}
		return SplineValue;
	}

	// ----------------------------------------------------------------------------

	/** Causal linear B-spline. */
	public static double LinearBSpline(double t) {
		double SplineValue = 0.0;
		t -= 1.0;
		if ((t >= -1.0) && (t <= -0.0)) {
			SplineValue = t + 1;
		} else if ((t > -0.0) & (t <= 1.0)) {
			SplineValue = -t + 1;
		}
		return SplineValue;
	}

	// ----------------------------------------------------------------------------

	/** Causal quadratic B-spline. */
	public static double QuadraticSpline(double t) {
		double SplineValue = 0.0;
		t -= 1.5;
		if ((t >= -1.5) && (t <= -0.5)) {
			SplineValue = 0.5 * t * t + 1.5 * t + 1.125;
		} else if ((t > -0.5) & (t <= 0.5)) {
			SplineValue = -t * t + 0.75;
		} else if ((t > 0.5) & (t <= 1.5)) {
			SplineValue = 0.5 * t * t - 1.5 * t + 1.125;
		}
		return SplineValue;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Refinement mask of causal B-spline (Quadratic and cubic) for a m scaling
	 * parameters: m = dilation factor, N= order of the basis function
	 */
	public static double[] MaskPolynomialSpline(int m, int N) {

		double[] h;
		int size = N * (m - 1) + 1;
		h = new double[size];

		for (int i = 0; i < size; i++) {
			h[i] = 0;
		}

		int[] index;
		index = new int[m];
		for (int i = 0; i < m; i++) {
			index[i] = 0;
		}

		multinomial(N, m, index, 0, m, N, h);

		return h;
	}
	
	// ----------------------------------------------------------------------------

	/**
	 * Refinement mask of Espline3 for a m scaling
	 * parameters: m = dilation factor, N= order of the basis function
	 */
	public static double[] MaskESpline(int m, int N, double alphaSub) {
		double den=0;
		den=Math.pow(2, N-1);
		
		double[] h;
		int size = N * (m - 1) + 1;
		h = new double[size];

		for (int i = 0; i < size; i++) {
			h[i] = 0;
		}

		h[0]=1/den;
		h[1]=(2*(double)Math.cos(alphaSub)+1)/den;
		h[2]=h[1];
		h[3]=1/den;

		return h;
	}

	// ----------------------------------------------------------------------------

	/** Causal cubic B-spline. */
	public static double CubicBSpline(double t) {
		double SplineValue = 0.0;
		t -= 2.0;
		if ((t >= -2.0) && (t <= -1.0)) {
			SplineValue = 1.0 / 6.0 * t * t * t + t * t + 2.0 * t + 4.0 / 3.0;
		} else if ((t > -1.0) & (t <= 0.0)) {
			SplineValue = -0.5 * t * t * t - t * t + 2.0 / 3.0;
		} else if ((t > 0.0) & (t <= 1.0)) {
			SplineValue = 0.5 * t * t * t - t * t + 2.0 / 3.0;
		} else if ((t > 1.0) & (t <= 2.0)) {
			SplineValue = -1.0 / 6.0 * t * t * t + t * t - 2.0 * t + 4.0 / 3.0;
		}
		return SplineValue;
	}

	// ----------------------------------------------------------------------------

	/** Causal B-spline of order 5. */
	public static double Order5BSpline(double t) {
		double SplineValue = 0.0;

		if ((t > 0) && (t <= 1.0)) {
			SplineValue = 1.0 / 24.0 * t * t * t * t;
		} else if ((t > 1.0) & (t <= 2.0)) {
			SplineValue = -5.0 / 24.0 + 5.0 / 6.0 * t - 5.0 / 4.0 * t * t + 5.0
					/ 6.0 * t * t * t - 1.0 / 6.0 * t * t * t * t;
		} else if ((t > 2.0) & (t <= 3.0)) {
			SplineValue = 155.0 / 24.0 - 25.0 / 2.0 * t + 35.0 / 4.0 * t * t
					- 5.0 / 2.0 * t * t * t + 1.0 / 4.0 * t * t * t * t;
		} else if ((t > 3.0) & (t <= 4.0)) {
			SplineValue = -655.0 / 24.0 + 65.0 / 2.0 * t - 55.0 / 4.0 * t * t
					+ 5.0 / 2.0 * t * t * t - 1.0 / 6.0 * t * t * t * t;
		} else if ((t > 4.0) & (t < 5.0)) {
			SplineValue = 625.0 / 24.0 - 125.0 / 6.0 * t + 25.0 / 4.0 * t * t
					- 5.0 / 6.0 * t * t * t + 1.0 / 24.0 * t * t * t * t;
		}
		return SplineValue;
	}

	// ----------------------------------------------------------------------------

	/** Causal B-spline of order 6. */
	public static double Order6BSpline(double t) {
		double SplineValue = 0.0;

		if ((t > 0) && (t <= 1.0)) {
			SplineValue = 1.0 / 120.0 * t * t * t * t * t;
		} else if ((t > 1.0) & (t <= 2.0)) {
			SplineValue = 1.0 / 20.0 - 1.0 / 4.0 * t + 1.0 / 2.0 * t * t - 1.0
					/ 2.0 * t * t * t + 1.0 / 4.0 * t * t * t * t - 1.0 / 24.0
					* t * t * t * t * t;
		} else if ((t > 2.0) & (t <= 3.0)) {
			SplineValue = -79.0 / 20.0 + 39.0 / 4.0 * t - 19.0 / 2.0 * t * t
					+ 9.0 / 2.0 * t * t * t - t * t * t * t + 1.0 / 12.0 * t
					* t * t * t * t;
		} else if ((t > 3.0) & (t <= 4.0)) {
			SplineValue = 731.0 / 20.0 - 231.0 / 4.0 * t + 71.0 / 2.0 * t * t
					- 21.0 / 2.0 * t * t * t + 3.0 / 2.0 * t * t * t * t - 1.0
					/ 12.0 * t * t * t * t * t;
		} else if ((t > 4.0) & (t <= 5.0)) {
			SplineValue = -1829.0 / 20.0 + 409.0 / 4.0 * t - 89.0 / 2.0 * t * t
					+ 19.0 / 2.0 * t * t * t - t * t * t * t + 1.0 / 24.0 * t
					* t * t * t * t;
		} else if ((t > 5.0) & (t < 6.0)) {
			SplineValue = 324.0 / 5.0 - 54.0 * t + 18.0 * t * t - 3.0 * t * t
					* t + 1.0 / 4.0 * t * t * t * t - 1.0 / 120.0 * t * t * t
					* t * t;
		}
		return SplineValue;
	}

	// ----------------------------------------------------------------------------

	/** Causal B-spline of order 7. */
	public static double Order7BSpline(double t) {
		double SplineValue = 0.0;

		if ((t > 0) && (t <= 1.0)) {
			SplineValue = 1.0 / 720.0 * t * t * t * t * t * t;
		} else if ((t > 1.0) & (t <= 2.0)) {
			SplineValue = -7.0 / 720.0 + 7.0 / 120.0 * t - 7.0 / 48.0 * t * t
					+ 7.0 / 36.0 * t * t * t - 7.0 / 48.0 * t * t * t * t + 7.0
					/ 120.0 * t * t * t * t * t - 1.0 / 120.0 * t * t * t * t
					* t * t;
		} else if ((t > 2.0) & (t <= 3.0)) {
			SplineValue = 1337.0 / 720.0 - 133.0 / 24.0 * t + 329.0 / 48.0 * t
					* t - 161.0 / 36.0 * t * t * t + 77.0 / 48.0 * t * t * t
					* t - 7.0 / 24.0 * t * t * t * t * t + 1.0 / 48.0 * t * t
					* t * t * t * t;
		} else if ((t > 3.0) & (t <= 4.0)) {
			SplineValue = -12089.0 / 360.0 + 196.0 / 3.0 * t - 1253.0 / 24.0
					* t * t + 196.0 / 9.0 * t * t * t - 119.0 / 24.0 * t * t
					* t * t + 7.0 / 12.0 * t * t * t * t * t - 1.0 / 36.0 * t
					* t * t * t * t * t;
		} else if ((t > 4.0) & (t <= 5.0)) {
			SplineValue = 59591.0 / 360.0 - 700.0 / 3.0 * t + 3227.0 / 24.0 * t
					* t - 364.0 / 9.0 * t * t * t + 161.0 / 24.0 * t * t * t
					* t - 7.0 / 12.0 * t * t * t * t * t + 1.0 / 48.0 * t * t
					* t * t * t * t;
		} else if ((t > 5.0) & (t <= 6.0)) {
			SplineValue = -208943.0 / 720.0 + 7525.0 / 24.0 * t - 6671.0 / 48.0
					* t * t + 1169.0 / 36.0 * t * t * t - 203.0 / 48.0 * t * t
					* t * t + 7.0 / 24.0 * t * t * t * t * t - 1.0 / 120.0 * t
					* t * t * t * t * t;
		} else if ((t > 6.0) & (t < 7.0)) {
			SplineValue = 117649.0 / 720.0 - 16807.0 / 120.0 * t + 2401.0
					/ 48.0 * t * t - 343.0 / 36.0 * t * t * t + 49.0 / 48.0 * t
					* t * t * t - 7.0 / 120.0 * t * t * t * t * t + 1.0 / 720.0
					* t * t * t * t * t * t;
		}

		return SplineValue;
	}

	// ----------------------------------------------------------------------------

	/** Causal exponential B-Spline with parameters (0,-j alpha, j alpha). */
	public static double ESpline3(double t, double alpha) {
		double alphaHalf = alpha / 2.0;
		double ESplineValue = 0.0;
		double eta = 2 * (1 - Math.cos(alpha)) / (alpha * alpha);
		if ((t >= 0) & (t <= 1)) {
			ESplineValue = (2 * (1 - Math.cos(alphaHalf * t)
					* Math.cos(alphaHalf * t)));
		} else if ((t > 1) & (t <= 2)) {
			ESplineValue = (Math.cos(alpha * (t - 2))
					+ Math.cos(alpha * (t - 1)) - 2 * Math.cos(alpha));
		} else if ((t > 2) & (t <= 3)) {
			ESplineValue = (1 - Math.cos(alpha * (t - 3)));
		}
		if(alpha==0)alpha=0.000001;
		
		ESplineValue = ESplineValue / (alpha * alpha * eta);
		return ESplineValue;
	}
	// ----------------------------------------------------------------------------

	/** Causal exponential B-Spline with parameters (0,-j alpha, j alpha) for a alpha which is non constant. */
	public static double ESpline3Bis(double t, double alpha, int M) {
		double alphaHalf = alpha / 2.0;
		double ESplineValue = 0.0;
		if(alpha==0)alpha=0.000001;
		double lambda=1;
		double nb=(double) Math.PI/M;
		double nb2=Math.sin(nb)/nb;
		
		
		lambda=Math.pow(nb2, -2);
		
		if ((t > 0) & (t < 1)) {
			ESplineValue = 2*Math.sin(alphaHalf*t)*Math.sin(alphaHalf*t);
		} else if ((t >= 1) & (t <= 2)) {
			ESplineValue = (Math.cos(alpha * (t - 2))
					+ Math.cos(alpha * (t - 1)) - 2 * Math.cos(alpha));
		} else if ((t > 2) & (t < 3)) {
			ESplineValue = 2*Math.sin(alphaHalf*(t-3))*Math.sin(alphaHalf*(t-3));
		} 
		
		
		ESplineValue = lambda*ESplineValue / (alpha * alpha);
		return ESplineValue;
	}

	// ----------------------------------------------------------------------------

	/** Causal exponential B-Spline with parameters (0, 0,-j alpha, j alpha). */
	public static double ESpline4(double t, double alpha) {
		double ESplineValue = 0.0;
		double eta = 2 * (1 - Math.cos(alpha)) / (alpha * alpha);
		if ((t >= 0) & (t <= 1)) {
			ESplineValue = t - Math.sin(alpha * t) / alpha;
		} else if ((t > 1) & (t <= 2)) {
			ESplineValue = 2 - t + 2 * Math.sin(alpha * (t - 1)) / alpha
					+ Math.sin(alpha * (t - 2)) / alpha - 2 * Math.cos(alpha)
					* t + 2 * Math.cos(alpha);
		} else if ((t > 2) & (t <= 3)) {
			ESplineValue = t - 2 - 4 * Math.cos(alpha) - 2
					* Math.sin(alpha * (t - 3)) / alpha + 2 * Math.cos(alpha)
					* (t - 1) - Math.sin(alpha * (t - 2)) / alpha;
		} else if ((t > 3) & (t <= 4)) {
			ESplineValue = 4 - t + Math.sin(alpha * (t - 4)) / alpha;
		}
		ESplineValue = ESplineValue / (alpha * alpha * eta);
		return ESplineValue;
	}

	// ----------------------------------------------------------------------------
	/** Causal Keys interpolator. */
	public static double Keys(double t) {

		double SplineValue = 0.0;

		if ((t >= 0) && (t <= 1.0)) {
			SplineValue = 0.5 * (-1 + t) * t * t;
		} else if ((t > 1) && (t <= 2.0)) {
			SplineValue = -0.5 * (-1 + t) * (6 + t * (-10 + 3 * t));
		} else if ((t > 2.0) & (t <= 3.0)) {
			SplineValue = 0.5 * (-3 + t) * (14 + t * (-14 + 3 * t));
		} else if ((t > 3.0) & (t <= 4.0)) {
			SplineValue = -0.5 * (-4 + t) * (-4 + t) * (-3 + t);
		}
		return SplineValue;

	}

	// ----------------------------------------------------------------------------

	/** Causal M-Spline. */
	public static double MSpline(double t, double alpha) {
		double ESplineValue = 0.0;
		double eta = 2.0 * PI * (Math.cos(alpha) - 1.0)
				* (1.0 + Math.cos(alpha) - 2.0 * Math.sin(alpha) / alpha);

		double m = 2.0 * PI / alpha;

		if ((t >= 0) & (t <= 1)) {
			ESplineValue = (-2.0 * PI + m * Math.sin(2.0 * PI / m))
					* Math.pow(Math.sin(PI * t / m), 2)
					+ Math.pow(Math.sin(PI / m), 2)
					* (2.0 * PI * t - m * Math.sin(2.0 * PI * t / m));
		} else if ((t > 1) & (t <= 2)) {
			ESplineValue = -PI
					* ((-2.0 + t) * Math.cos(2.0 * PI / m) - (-1.0 + t)
							* Math.cos(4.0 * PI / m) + Math.cos(2.0 * PI
									* (-2 + t) / m))
									+ m
									* (Math.cos(3.0 * PI / m)
											* (-Math.sin(PI / m) + Math.sin(PI
													* (3.0 - 2.0 * t) / m)) + Math.sin(2.0 * PI
															* (-1 + t) / m));
		} else if ((t > 2) & (t <= 3)) {
			ESplineValue = (-2.0
					* PI
					* (-(-2 + t) * Math.cos(2.0 * PI / m) + (-3.0 + t)
							* Math.cos((4.0 * PI) / m) + Math.cos(2.0 * PI
									* (-2 + t) / m)) + m
									* (Math.sin(2.0 * PI / m) - Math.sin(4.0 * PI / m)
											+ Math.sin((2.0 * PI * (-4.0 + t)) / m) - 2.0
											* Math.sin(2.0 * PI * (-3 + t) / m) + Math.sin(2.0
													* PI * (-1.0 + t) / m))) / 2.0;
		} else if ((t > 3) & (t <= 4)) {
			ESplineValue = (-2.0 * PI + m * Math.sin(2.0 * PI / m))
					* Math.sin((PI * (-4.0 + t)) / m)
					* Math.sin((PI * (-4.0 + t)) / m)
					+ Math.sin(PI / m)
					* Math.sin(PI / m)
					* (-2.0 * PI * (-4.0 + t) + m
							* Math.sin((2.0 * PI * (-4.0 + t)) / m));
		}
		ESplineValue = ESplineValue / eta;
		return ESplineValue;
	}
	
	// ----------------------------------------------------------------------------

		/** Causal A-Spline. */
		public static double ASpline(double t, double alpha) {
			double ASplineValue = 0.0;
			double m = 2.0 * PI / alpha;
			
			double eta = 1.0/27*m*m;
			
			if ((t >= 3) & (t < 3.5)) {
				ASplineValue = (2 - 3 * Math.cos(PI * (7 - 2*t) / m)
						+ Math.cos(2*PI * (-4 + t) / m));
				
				ASplineValue *= eta;
				
			} else if ((t >= 3.5) & (t < 4)) {
				ASplineValue = Math.sin(PI * (-4 + t) / m) * Math.sin(PI * (-4 + t) / m);
				
				ASplineValue *= -eta;
			}
			
			else if ((t >= 1.5) & (t < 2.5)) {
				ASplineValue = (1 + 4*Math.cos(2*PI/m)
						- 3*Math.cos(PI * (3-2*t)/m)
						- 3*Math.cos(PI * (5-2*t)/m)
						+ Math.cos(2*PI*(-2+t)/m));
				
				ASplineValue *= -eta;
			}
			
			else if ((t >= 2.5) & (t < 3)) {
				ASplineValue = (2 + 2*Math.cos(2*PI/m)
						- 3*Math.cos(PI * (7 - 2*t)/m)
						- Math.cos(2*PI*(-2+t)/m));
				
				ASplineValue *= eta;
			}
			
			else if ((t >= 1) & (t < 1.5)) {
				ASplineValue = (2 + 2*Math.cos(2*PI/m)
						- Math.cos(2*PI * (-2+t)/m)
						- 3*Math.cos((PI - 2*PI*t)/m));
				
				ASplineValue *= eta;
			}
			
			else if ((t >= 0) & (t < 0.5)) {
				ASplineValue = Math.sin(PI*t/m)*Math.sin(PI*t/m);
				
				ASplineValue *= -eta;
			}
			
			else if ((t >= 0.5) & (t < 1)) {
				ASplineValue = (2 + Math.cos(2*PI*t/m)
						- 3*Math.cos((PI-2*PI*t)/m));
				
				ASplineValue *= eta;
			}
			
			return ASplineValue;
		}

	// ----------------------------------------------------------------------------
	// DERIVATIVES OF B-SPLINE FUNCTIONS
		
		// ----------------------------------------------------------------------------

		/** Derivative of causal A-Spline. */
		public static double ASplinePrime(double t, double alpha) {
			double ASplineValue = 0.0;
			double m = 2.0 * PI / alpha;
			
			double eta = 2.0/27*m*PI;
			
			if ((t >= 0) & (t < 0.5)) {
				ASplineValue = Math.sin(2*PI*t/m);
				
				ASplineValue *= -eta;
				
			} else if ((t >= 3.5) & (t < 4)) {
				ASplineValue = Math.sin(2*PI*(-4+t)/m);
				
				ASplineValue *= -eta;
			}
			
			else if (2*t == 3) {
				ASplineValue = (Math.sin(PI/m)-3*Math.sin(2*PI/m));
				
				ASplineValue *= -eta;
			}
			
			else if ((t >= 1.5) & (t < 2.5)) {
				ASplineValue = (3*(Math.sin(PI*(3-2*t)/m)
						+ Math.sin(PI*(5-2*t)/m))
						+ Math.sin(2*PI*(-2+t)/m));
				
				ASplineValue *= eta;
			}
			
			else if ((t >= 2.5) & (t < 3)) {
				ASplineValue = (-3*Math.sin(PI*(7-2*t)/m)
						+ Math.sin(2*PI*(-2+t)/m));
				
				ASplineValue *= eta;
			}
			
			else if ((t >= 0.5) & (t < 1)) {
				ASplineValue = (Math.sin(2*PI*t/m)
						+ 3*Math.sin((PI-2*PI*t)/m));
				
				ASplineValue *= -eta;
			}
			
			else if ((t >= 3) & (t < 3.5)) {
				ASplineValue = (3*Math.sin(PI*(7-2*t)/m)
						+ Math.sin(2*PI*(-4+t)/m));
				
				ASplineValue *= -eta;
			}
			
			else if ((t >= 1) & (t < 1.5)) {
				ASplineValue = (Math.sin(2*PI*(-2+t)/m)
						- 3*Math.sin((PI-2*PI*t)/m));
				
				ASplineValue *= eta;
			}
			
			return ASplineValue;
		}

				

	// ----------------------------------------------------------------------------
	/**
	 * Derivative of the causal M-Spline
	 */
	public static double MSpline_Prime(double t, double alpha) {
		double mSplinePrimeValue = 0.0;

		double m = 2.0 * PI / alpha;

		double den = m * (-PI + m * Math.tan(PI / m));

		if ((t >= 0) & (t < 1)) {
			mSplinePrimeValue = PI
					* csc(alpha)
					* csc(alpha)
					* Math.sin(PI * t / m)
					* (-2 * PI * Math.cos(PI * t / m) + m
							* (-Math.sin(PI * (-2 + t) / m) + Math.sin(PI * t
									/ m))) / den;

		} else if (t == 1) { // ok
			mSplinePrimeValue = alpha * csc(alpha) / 2;
		} else if ((t > 1) & (t <= 2)) {
			mSplinePrimeValue = PI
					* csc(alpha)
					* csc(alpha)
					* (m
							* (-Math.cos(alpha) + Math.cos(2 * alpha) - 2
									* Math.cos(3 * PI / m)
									* Math.cos(PI * (3 - 2 * t) / m) + 2 * Math
									.cos(2 * PI * (-1 + t) / m)) + 2 * PI
									* Math.sin(2 * PI * (-2 + t) / m)) / (2 * den);

		} else if ((t > 2) & (t < 3)) {
			mSplinePrimeValue = PI
					* csc(alpha)
					* csc(alpha)
					* (m
							* (Math.cos(alpha) - Math.cos(2 * alpha)
									+ Math.cos(2 * PI * (-4 + t) / m) - 2
									* Math.cos(2 * PI * (-3 + t) / m) + Math
									.cos(2 * PI * (-1 + t) / m)) + 2 * PI
									* Math.sin(2 * PI * (-2 + t) / m)) / (2 * den);
		} else if (t == 3) { // ok
			mSplinePrimeValue = -alpha * csc(alpha) / 2;
		} else if ((t > 3) & (t <= 4)) {
			mSplinePrimeValue = PI
					* csc(alpha)
					* csc(alpha)
					* Math.sin(PI * (-4 + t) / m)
					* (-2 * PI * Math.cos(PI * (-4 + t) / m) + m
							* (-Math.sin(PI * (-4 + t) / m) + Math.sin(PI
									* (-2 + t) / m))) / den;
		}

		return mSplinePrimeValue;
	}

	// ----------------------------------------------------------------------------
	/**
	 * Second derivative of the causal M-Spline
	 */
	public static double MSpline_PrimePrime(double t, double alpha) {
		double mSplinePrimePrimeValue = 0.0;

		double m = 2.0 * PI / alpha;

		double den = m * m * (-PI + m * Math.tan(PI / m));

		if ((t >= 0) & (t <= 1)) {
			mSplinePrimePrimeValue = PI
					* PI
					* csc(alpha)
					* csc(alpha)
					* (-2 * PI * Math.cos(2 * PI * t / m) + m
							* (-Math.sin(2 * PI * (-1 + t) / m) + Math.sin(2
									* PI * t / m))) / den;

		} else if ((t > 1) & (t < 2)) {
			mSplinePrimePrimeValue = 2
					* PI
					* PI
					* csc(alpha)
					* csc(alpha)
					* (PI * Math.cos(2 * PI * (-2 + t) / m) - m
							* (Math.cos(3 * PI / m)
									* Math.sin(PI * (3 - 2 * t) / m) + Math
									.sin(2 * PI * (-1 + t) / m))) / den;

		} else if (t == 2) {
			mSplinePrimePrimeValue = 2
					* PI
					* PI
					* (2 * PI - 3 * m * Math.sin(alpha) + m
							* Math.sin(2 * alpha))
							/ (m * m * (PI * (-1 + Math.cos(2 * alpha)) + 8 * m
									* Math.cos(PI / m) * Math.sin(PI / m)
									* Math.sin(PI / m) * Math.sin(PI / m)));
		}

		else if ((t > 2) & (t <= 3)) {
			mSplinePrimePrimeValue = PI
					* PI
					* csc(alpha)
					* csc(alpha)
					* (2 * PI * Math.cos(2 * PI * (-2 + t) / m) - m
							* (Math.sin(2 * PI * (-4 + t) / m) - 2
									* Math.sin(2 * PI * (-3 + t) / m) + Math
									.sin(2 * PI * (-1 + t) / m))) / den;
		} else if ((t > 3) & (t < 4)) {
			mSplinePrimePrimeValue = PI
					* PI
					* csc(alpha)
					* csc(alpha)
					* (-2 * PI * Math.cos(2 * PI * (-4 + t) / m) + m
							* (-Math.sin(2 * PI * (-4 + t) / m) + Math.sin(2
									* PI * (-3 + t) / m))) / den;

		}

		return mSplinePrimePrimeValue;
	}

	/**
	 * First Derivative of Keys
	 *
	 */
	public static double Keys_Prime(double t) {
		double keysPrimeValue = 0.0;
		if (t <= 0) {
			keysPrimeValue = 0;
		} else if (t > 0 && t < 1) {
			keysPrimeValue = ((t - 1) * t + (Math.pow(t, 2) / 2));
		} else if (t == 1) {
			keysPrimeValue = 0.5;
		} else if (t > 1 && t < 2) {
			keysPrimeValue = (-0.5) * (-1 + t) * (-10 + 6 * t) + (0.5)
					* (-6 + 10 * t - 3 * (Math.pow(t, 2)));
		} else if (t == 2) {
			keysPrimeValue = 0;
		} else if (t > 2 && t < 3) {
			keysPrimeValue = (0.5) * (-3 + t) * (-14 + 6 * t) + (0.5)
					* (14 - 14 * t + 3 * (Math.pow(t, 2)));
		} else if (t == 3) {
			keysPrimeValue = (-0.5);
		} else if (t > 3 && t < 4) {
			keysPrimeValue = (-0.5) * Math.pow((-4 + t), 2) - (-4 + t)
					* (-3 + t);
		} else {
			throw new IllegalArgumentException(
					"this value is not defined for Keys_Prime");
		}
		return keysPrimeValue;
	}

	/**
	 * Second derivative of Keys
	 * 
	 */
	public static double Keys_PrimePrime(double t) {
		double keys_PrimePrimeValue = 0.0;
		if (t <= 0) {
			keys_PrimePrimeValue = 0;
		} else if (t > 0 && t <= 1) {
			keys_PrimePrimeValue = (0.5) * (-2 + 6 * t);
		} else if (t > 1 && t < 2) {
			keys_PrimePrimeValue = (0.5) * (26 - 18 * t);
		} else if (t == 2) {
			keys_PrimePrimeValue = -5;
		} else if (t > 2 && t <= 3) {
			keys_PrimePrimeValue = (0.5) * (-46 + 18 * t);
		} else if (t > 3 && t <= 4) {
			keys_PrimePrimeValue = (0.5) * (22 - 6 * t);
		} else if (t > 4) {
			keys_PrimePrimeValue = 0;
		} else {
			throw new IllegalArgumentException(
					"this value is not defined for Keys_PrimePrime");
		}
		return keys_PrimePrimeValue;

	}

	// ----------------------------------------------------------------------------

	/** Cosecant. */
	public static double csc(double arg) {
		double result = Math.sin(arg);
		if (result == 0) {
			return Double.POSITIVE_INFINITY;
		} else {
			return 1 / result;
		}
	}

	// ----------------------------------------------------------------------------

	/** Secant. */
	public static double sec(double arg) {
		double result = Math.cos(arg);
		if (result == 0) {
			return Double.POSITIVE_INFINITY;
		} else {
			return 1 / result;
		}
	}

	// ----------------------------------------------------------------------------

	/** Cotangent. */
	public static double cot(double arg) {
		double result = Math.tan(arg);
		if (result == 0) {
			return Double.POSITIVE_INFINITY;
		} else {
			return 1 / result;
		}
	}

	// ----------------------------------------------------------------------------

	/** Derivative of the linear B-spline. */
	public static double LinearBSpline_Prime(double t) {
		double ESplinePrimeValue = 0.0;
		if ((t > -1) & (t <= 0)) {
			ESplinePrimeValue = 1;
		} else if ((t > 0) & (t < 1)) {
			ESplinePrimeValue = -1;
		}
		return ESplinePrimeValue;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Derivative of the causal exponential B-Spline with parameters (0,-j
	 * alpha, j alpha).
	 */
	public static double ESpline3_Prime(double t, double alpha) {
		double ESplinePrimeValue = 0.0;
		double eta = 2 * (1 - Math.cos(alpha)) / (alpha * alpha);
		if ((t >= 0) & (t <= 1)) {
			ESplinePrimeValue = Math.sin(alpha * t);
		} else if ((t > 1) & (t <= 2)) {
			ESplinePrimeValue = -(Math.sin(alpha * (t - 2)) + Math.sin(alpha
					* (t - 1)));
		} else if ((t > 2) & (t <= 3)) {
			ESplinePrimeValue = Math.sin(alpha * (t - 3));
		}
		ESplinePrimeValue = ESplinePrimeValue / (alpha * eta);
		return ESplinePrimeValue;
	}
	
	// ----------------------------------------------------------------------------

	/**
	 * Derivative of the causal exponential B-Spline with parameters (0,-j
	 * alpha, j alpha) for a alpha which is non constant.
	 */
	public static double ESpline3_PrimeBis(double t, double alpha, int M) {
		double ESplinePrimeValue = 0.0;
		double alphaHalf=alpha/2.0;
		
		if(alpha==0)alpha=0.000001;
		double lambda=1;
		double nb=(double) Math.PI/M;
		double nb2=Math.sin(nb)/nb;
		
		
		lambda=Math.pow(nb2, -2);
		
		
		if ((t >= 0) & (t <= 1)) {
			ESplinePrimeValue = Math.sin(alpha * t);
		} else if ((t > 1) & (t <= 2)) {
			ESplinePrimeValue = 2*Math.cos(alphaHalf)*Math.sin(alphaHalf*(3-2*t));
		} else if ((t > 2) & (t <= 3)) {
			ESplinePrimeValue = Math.sin(alpha * (t - 3));
		}
		ESplinePrimeValue =lambda*ESplinePrimeValue /alpha;
		return ESplinePrimeValue;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Second derivative of the causal exponential B-Spline with parameters
	 * (0,-j alpha, j alpha).
	 */
	public static double ESpline3_PrimePrime(double t, double alpha) {
		double eSplinePrimeValue = 0.0;
		double eta = 2.0 * (1.0 - Math.cos(alpha)) / (alpha * alpha);
		if (t >= 0.0 && t <= 1.0) {
			eSplinePrimeValue = Math.cos(alpha * t);
		} else if (t > 1.0 && t <= 2.0) {
			eSplinePrimeValue = -Math.cos(alpha * (t - 2.0))
					- Math.cos(alpha * (t - 1.0));
		} else if (t > 2.0 && t <= 3.0) {
			eSplinePrimeValue = Math.cos(alpha * (t - 3.0));
		}
		eSplinePrimeValue /= eta;
		return eSplinePrimeValue;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Sampled correlation of the exponential B-spline of order three with a
	 * filter.
	 */
	public static double correlationESpline(int l, double alpha) {
		double PIM_ = 0.5 * alpha;
		double M_ = PI / PIM_;

		double value = 0.0;
		switch (l) {
		case -2:
			value = (1.0 / 8.0)
			* (PI * Math.cos(PIM_) * Math.sin(PIM_) - M_ + M_
					* Math.cos(PIM_) * Math.cos(PIM_))
					/ (M_ * (1.0 - 2.0 * Math.cos(PIM_) * Math.cos(PIM_) + Math
							.cos(PIM_)
							* Math.cos(PIM_)
							* Math.cos(PIM_)
							* Math.cos(PIM_)));
			break;
		case -1:
			value = -0.25
			* (PI * Math.cos(PIM_) * Math.sin(PIM_) + M_ - 3.0 * M_
					* Math.cos(PIM_) * Math.cos(PIM_) + 2.0 * M_
					* Math.cos(PIM_) * Math.cos(PIM_) * Math.cos(PIM_)
					* Math.cos(PIM_))
					/ (M_ * (1.0 - 2.0 * Math.cos(PIM_) * Math.cos(PIM_) + Math
							.cos(PIM_)
							* Math.cos(PIM_)
							* Math.cos(PIM_)
							* Math.cos(PIM_)));
			break;
		case 1:
			value = 0.25
			* (PI * Math.cos(PIM_) * Math.sin(PIM_) + M_ - 3.0 * M_
					* Math.cos(PIM_) * Math.cos(PIM_) + 2.0 * M_
					* Math.cos(PIM_) * Math.cos(PIM_) * Math.cos(PIM_)
					* Math.cos(PIM_))
					/ (M_ * (1.0 - 2.0 * Math.cos(PIM_) * Math.cos(PIM_) + Math
							.cos(PIM_)
							* Math.cos(PIM_)
							* Math.cos(PIM_)
							* Math.cos(PIM_)));
			break;
		case 2:
			value = -(1.0 / 8.0)
			* (PI * Math.cos(PIM_) * Math.sin(PIM_) - M_ + M_
					* Math.cos(PIM_) * Math.cos(PIM_))
					/ (M_ * (1.0 - 2.0 * Math.cos(PIM_) * Math.cos(PIM_) + Math
							.cos(PIM_)
							* Math.cos(PIM_)
							* Math.cos(PIM_)
							* Math.cos(PIM_)));
			break;
		default:
			value = 0.0;
			break;
		}
		return value;
	}

	// ----------------------------------------------------------------------------
	/**
	 * Sampled correlation of the MSpline with MSplinePrime, i.e. to compute
	 * equation 4.21 in Ricard's thesis. p= k-l are the shifting parameters of
	 * the basis and its derivative, M is the number of control points
	 */
	public static double correlationMSplineMSplinePrime(int p, int M) {

		double value = 0.0;
		double arg = 2 * PI / M;
		double den = 2916 * (-27 + 6 * Math.sqrt(3) * PI - PI * PI);

		switch (p) {
		case -3:
			value = M
			* M
			* M
			* (M
					* (-27 + 18 * M * M + 2 * PI
							* (-6 * Math.sqrt(3) + 7 * PI))
							+ M
							* (27 - 18 * M * M - 42 * Math.sqrt(3) * PI + 40
									* PI * PI) * Math.cos(arg) + (9 * M * M
											* (3 * Math.sqrt(3) - 5 * PI) + PI
											* (27 + 8 * PI * (-3 * Math.sqrt(3) + 2 * PI)))
											* Math.sin(arg)) / den;

			break;
		case -2:
			value = M
			* M
			* M
			* M
			* (-3
					* (27 + 6 * M * M + 4 * PI
							* (-9 * Math.sqrt(3) + 8 * PI))
							+ 4
							* (27 + 21 * Math.sqrt(3) * PI - 26 * PI * PI)
							* Math.cos(arg)
							+ (-27 + 18 * M * M + 8
									* (3 * Math.sqrt(3) - 2 * PI) * PI)
									* Math.cos(2 * arg) - 54 * M
									* (Math.sqrt(3) - 2 * PI) * Math.sin(arg) + 9 * M
									* (-3 * Math.sqrt(3) + 4 * PI) * Math.sin(2 * arg))
									/ den;

			break;
		case -1:
			value = M
			* M
			* M
			* (-3
					* M
					* (27 + 6 * M * M + 36 * Math.sqrt(3) * PI - 46 * M
							* M)
							+ M
							* (135 + 54 * M * M + 2 * PI
									* (-69 * Math.sqrt(3) + 52 * PI))
									* Math.cos(arg)
									- 2
									* M
									* (27 + 18 * M * M + 2
											* (6 * Math.sqrt(3) - 7 * PI) * PI)
											* Math.cos(2 * arg)
											- 3
											* (-9 * M * M * (Math.sqrt(3) - 3 * PI) + PI
													* (27 + 8 * PI
															* (-3 * Math.sqrt(3) + 2 * PI)))
															* Math.sin(arg) + 18 * M * M
															* (3 * Math.sqrt(3) - 4 * PI) * Math.sin(2 * arg))
															/ den;

			break;
		case 1:
			value -= M
			* M
			* M
			* (-3
					* M
					* (27 + 6 * M * M + 36 * Math.sqrt(3) * PI - 46 * M
							* M)
							+ M
							* (135 + 54 * M * M + 2 * PI
									* (-69 * Math.sqrt(3) + 52 * PI))
									* Math.cos(arg)
									- 2
									* M
									* (27 + 18 * M * M + 2
											* (6 * Math.sqrt(3) - 7 * PI) * PI)
											* Math.cos(2 * arg)
											- 3
											* (-9 * M * M * (Math.sqrt(3) - 3 * PI) + PI
													* (27 + 8 * PI
															* (-3 * Math.sqrt(3) + 2 * PI)))
															* Math.sin(arg) + 18 * M * M
															* (3 * Math.sqrt(3) - 4 * PI) * Math.sin(2 * arg))
															/ den;

			break;

		case 2:
			value -= M
			* M
			* M
			* M
			* (-3
					* (27 + 6 * M * M + 4 * PI
							* (-9 * Math.sqrt(3) + 8 * PI))
							+ 4
							* (27 + 21 * Math.sqrt(3) * PI - 26 * PI * PI)
							* Math.cos(arg)
							+ (-27 + 18 * M * M + 8
									* (3 * Math.sqrt(3) - 2 * PI) * PI)
									* Math.cos(2 * arg) - 54 * M
									* (Math.sqrt(3) - 2 * PI) * Math.sin(arg) + 9 * M
									* (-3 * Math.sqrt(3) + 4 * PI) * Math.sin(2 * arg))
									/ den;

			break;

		case 3:
			value -= M
			* M
			* M
			* (M
					* (-27 + 18 * M * M + 2 * PI
							* (-6 * Math.sqrt(3) + 7 * PI))
							+ M
							* (27 - 18 * M * M - 42 * Math.sqrt(3) * PI + 40
									* PI * PI) * Math.cos(arg) + (9 * M * M
											* (3 * Math.sqrt(3) - 5 * PI) + PI
											* (27 + 8 * PI * (-3 * Math.sqrt(3) + 2 * PI)))
											* Math.sin(arg)) / den;

			break;

		default:
			value = 0.0;
			break;
		}
		return M * value;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The Green function corresponding to the convolution of two exponential
	 * splines with parameters alpha1 and alpha2 which have been scaled M1 and
	 * M2 times respectively.
	 */
	public static double convolutedGreenFunctionESpline(double l,
			double alpha1, double alpha2, double M1, double M2) {
		// double PIM_ = 0.5 * alpha;
		// double M_ = PI / PIM_;

		double value = 0.0;
		if (l > 0) {
			if ((Math.abs(alpha1 * M1 - alpha2 * M2)) < 1.0E-10) {
				value = (alpha2 * M2 * l * (2 + Math.cos(alpha2 * M2 * l)) - 3 * Math
						.sin(alpha2 * M2 * l))
						/ (2 * alpha1 * alpha1 * alpha2 * alpha2 * alpha2 * M2);
			} else {
				value = l
						/ (alpha1 * alpha1 * alpha2 * alpha2)
						+ (M2 * M2 * Math.sin(alpha1 * M1 * l))
						/ (alpha1 * alpha1 * alpha1 * M1 * (alpha1 * alpha1
								* M1 * M1 - alpha2 * alpha2 * M2 * M2))
								+ (M1 * M1 * Math.sin(alpha2 * M2 * l))
								/ (alpha2 * alpha2 * alpha2 * M2 * (alpha2 * alpha2
										* M2 * M2 - alpha1 * alpha1 * M1 * M1));
			}
		}
		return value;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Correlation function of two exponential B-splines of order three with
	 * parameters alpha1 and alpha2 which have been scaled M1 and M2 times
	 * respectively. \phi_{\alpha_1}(M1*t) \ast \phi_{\alpha_2}(M2*t)
	 */
	public static double correlationOfTwoESpline(double l, double alpha1,
			double alpha2, double M1, double M2) {
		// double PIM_ = 0.5 * alpha;
		// double M_ = PI / PIM_;
		double value = 0.0;

		value += convolutedGreenFunctionESpline(l - 0.0 / M1 - 0.0 / M2,
				alpha1, alpha2, M1, M2);

		value += (-1 - 2 * Math.cos(alpha2))
				* convolutedGreenFunctionESpline(l - 0.0 / M1 - 1.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (1 + 2 * Math.cos(alpha2))
				* convolutedGreenFunctionESpline(l - 0.0 / M1 - 2.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1)
				* convolutedGreenFunctionESpline(l - 0.0 / M1 - 3.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1 - 2 * Math.cos(alpha1))
				* convolutedGreenFunctionESpline(l - 1.0 / M1 - 0.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1 - 2 * Math.cos(alpha1))
				* (-1 - 2 * Math.cos(alpha2))
				* convolutedGreenFunctionESpline(l - 1.0 / M1 - 1.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1 - 2 * Math.cos(alpha1))
				* (1 + 2 * Math.cos(alpha2))
				* convolutedGreenFunctionESpline(l - 1.0 / M1 - 2.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1 - 2 * Math.cos(alpha1))
				* (-1)
				* convolutedGreenFunctionESpline(l - 1.0 / M1 - 3.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (1 + 2 * Math.cos(alpha1))
				* convolutedGreenFunctionESpline(l - 2.0 / M1 - 0.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (1 + 2 * Math.cos(alpha1))
				* (-1 - 2 * Math.cos(alpha2))
				* convolutedGreenFunctionESpline(l - 2.0 / M1 - 1.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (1 + 2 * Math.cos(alpha1))
				* (1 + 2 * Math.cos(alpha2))
				* convolutedGreenFunctionESpline(l - 2.0 / M1 - 2.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (1 + 2 * Math.cos(alpha1))
				* (-1)
				* convolutedGreenFunctionESpline(l - 2.0 / M1 - 3.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1)
				* convolutedGreenFunctionESpline(l - 3.0 / M1 - 0.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1)
				* (-1 - 2 * Math.cos(alpha2))
				* convolutedGreenFunctionESpline(l - 3.0 / M1 - 1.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1)
				* (1 + 2 * Math.cos(alpha2))
				* convolutedGreenFunctionESpline(l - 3.0 / M1 - 2.0 / M2,
						alpha1, alpha2, M1, M2);

		value += (-1)
				* (-1)
				* convolutedGreenFunctionESpline(l - 3.0 / M1 - 3.0 / M2,
						alpha1, alpha2, M1, M2);

		return ((alpha1 * alpha1 * alpha2 * alpha2) / (4 * (1 - Math
				.cos(alpha1)) * (1 - Math.cos(alpha2)))) * value;
	}

	// ----------------------------------------------------------------------------

	/** Sampled autocorrelation of the exponential B-spline of order three. */
	public static double autocorrelationESpline(int l, double alpha) {

		double value = 0.0;
		switch (l) {
		case -2:
			value = (2 * alpha + alpha * Math.cos(alpha) - 3 * Math.sin(alpha))
			/ (2 * alpha);
			break;
		case -1:
			value = (-alpha - 5 * alpha * Math.cos(alpha) + 3 * Math.sin(alpha) + 3
					* Math.cos(alpha) * Math.sin(alpha))
					/ alpha;
			break;
		case 0:
			value = (24 * alpha + 4 * alpha * Math.cos(alpha) + 8 * alpha
					* Math.cos(2 * alpha) - 12 * Math.sin(alpha) - 26
					* Math.cos(alpha) * Math.sin(alpha) + Math.sin(2 * alpha))
					/ (4 * alpha);
			break;
		case 1:
			value = (-alpha - 5 * alpha * Math.cos(alpha) + 3 * Math.sin(alpha) + 3
					* Math.cos(alpha) * Math.sin(alpha))
					/ alpha;
			break;
		case 2:
			value = (2 * alpha + alpha * Math.cos(alpha) - 3 * Math.sin(alpha))
			/ (2 * alpha);
			break;
		default:
			value = 0.0;
			break;
		}
		return value;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Sampled autocorrelation of the exponential B-spline of order three.
	 * Parameters are the number of control points M and the index of the
	 * circulant matrix l
	 */
	// --> the above implementation might differ up to a multiplication factor
	// TODO: check why?
	public static double autocorrelationESpline3(int l, int M) {

		double cscValue = csc(Math.PI / M);
		double value = 0.0;

		if (l == 0) {
			value = (Math.pow(cscValue, 4) * (12 * Math.PI + 2 * Math.PI
					* Math.cos(2 * Math.PI / M) + 4 * Math.PI
					* Math.cos(4 * Math.PI / M) - 3 * M
					* (Math.sin(2 * Math.PI / M) + Math.sin(4 * Math.PI / M))))
					/ (32 * M * Math.PI);
		}

		else if (l == 1 || l == -1) {
			double cotValue = cot(Math.PI / M);

			value = (3 * Math.pow(cotValue, 3))
					/ (8 * Math.PI)
					- ((1 + 5 * Math.cos(2 * Math.PI / M)) * Math.pow(cscValue,
							4)) / (16 * M);
		} else if (l == 2 || l == -2) {
			value = (Math.pow(cscValue, 4) * (2 * Math.PI
					* (2 + Math.cos(2 * Math.PI / M)) - 3 * M
					* Math.sin(2 * Math.PI / M)))
					/ (64 * M * Math.PI);
		}

		return value;
	}

	// ----------------------------------------------------------------------------

	/** This function compute the factorial function */
	public static int factorial(int n) {

		int factorial = 1;

		if (n == 0) {
			factorial = 1;
		} else {
			for (int i = 1; i < n + 1; i++) {
				factorial *= i;
			}
		}
		return factorial;
	}

	// ----------------------------------------------------------------------------

	/**
	 * This is a recursive function to compute the multinomial coefficient of
	 * (x0+x1+...+xm-1)^N The goal of this function is to find every
	 * {k0,...,km-1} as k0+...+km-1=N (cf multinomial theorem on Wikipedia for
	 * more understanding the formula) parameters : index= array of the ki,
	 * size= number f coefficient ki, coefMax= the maximum value that a ki can
	 * take, it= the iteration of the recursive function, m= dilation factor,
	 * N=order of the spline, h=mask
	 */
	public static int multinomial(int coefMax, int size, int[] index, int it,
			int m, int N, double[] h) {
		int newCoefMax;

		if (size == 1) {
			index[it] = coefMax;
			// for each combination we save the results and save
			// the coefficient in the mask h
			computeMultinomialCoef(index, m, N, h);
		}

		else {
			for (int k = 0; k < coefMax + 1; k++) {
				index[it] = k;
				newCoefMax = coefMax - k;

				multinomial(newCoefMax, size - 1, index, it + 1, m, N, h);
			}
		}

		return 0;
	}

	// ----------------------------------------------------------------------------

	/**
	 * This function compute the multinomial coefficient corresponding of the
	 * combination of the ki done by the function <code>multinomial()</code>.
	 */
	public static double[] computeMultinomialCoef(int[] index, int m, int N,
			double[] h) {
		int dem = 1, num = 0, degree = 0, l = 1;
		double coef = 0;

		num = factorial(N);
		for (int k = 1; k < N; k++) {
			l *= m;
		}

		for (int k = 0; k < m; k++) {
			dem *= factorial(index[k]);
			degree += k * index[k];
		}
		coef = num / dem;
		h[degree] += coef / l;

		return h;
	}

	// ----------------------------------------------------------------------------

	/** This function compute the B-Spline Filter B(z). */
	public static double[] computeBSplineFilter(int N) {
		double[] B;
		int Bsize;

		if (N == 0) {
			Bsize = 2;
		} else {
			Bsize = N + 1;
		}

		B = new double[Bsize];
		double a = 0;

		if (N == 0) {
			B[0] = 0.5;
			B[1] = 0.5;
		} else {
			for (int i = 0; i < Bsize; i++) {
				B[i] = 0;
			}

			for (int i = 0; i < N + 1; i++) {
				for (int k = 0; k < N + 2; k++) {

					a = i - k;
					if (a < 0) {
						a = 0;
					}

					B[i] += (factorial(N + 1) / (factorial(k) * factorial(N + 1
							- k)))
							* Math.pow(-1, k) * Math.pow(a, N);
				}
			}

			for (int i = 0; i < Bsize; i++) {
				B[i] /= factorial(N);
			}
		}

		return B;
	}

	// ----------------------------------------------------------------------------

	/** This function compute the derivative mask of a B-Spline */
	/* TODO : formula for E Spline */
	/* Parameter : N=order of the function to derive */
	public static double[] computePolynomialDerivativeMask(int N) {
		double[] Der;
		Der = new double[N + 1];
		for (int i = 0; i < N + 1; i++) {
			Der[i] = computePolynomialDerivative(i, N);
		}
		return Der;
	}
	
	// ----------------------------------------------------------------------------

	/** This function compute the derivative mask of a ESpline3 */
	/* Parameter : N=order of the function to derive */
	public static double[] computeESplineDerivativeMask(int N, double alphaSub, int M) {
		double[] Der;
		Der = new double[N + 1];
		for (int i = 0; i < N + 1; i++) {
			Der[i] = ESpline3_PrimeBis(i, alphaSub,M);
		}
		return Der;
	}

	// ----------------------------------------------------------------------------

	/** This function compute the derivative mask of a B-Spline */
	/* TODO : formula for E Spline */
	/*
	 * Parameter : index= the value where the derivative is evaluated, N=order
	 * of the function to derive
	 */
	public static double computePolynomialDerivative(double index, int N) {
		double value = 0;
		if (N == 4) {
			value = QuadraticSpline(index) - QuadraticSpline(index - 1);
		} else if (N == 3) {
			value = LinearBSpline(index) - LinearBSpline(index - 1);
		} else if (N == 2) {
			value = ConstantBSpline(index) - ConstantBSpline(index - 1);
		} else {
			System.out
			.println("The order of the basis function has to be between 2 and 4");
		}
		return value;
	}
}
