package plugins.vannary.morphomaths;

import icy.sequence.Sequence;
import icy.type.collection.array.ArrayUtil;

/**
 * Class containing distance transform algorithms in 2D/3D
 * 
 * @author Alexandre Dufour
 */
public class DistanceTransforms
{
    public enum Algorithm
    {
        /**
         * Chamfer 3x3 propagation algorithm (see papers from Borgefors)
         */
        Chamfer
    }

    /**
     * Computes the unsigned distance map to an object
     * 
     * @param input
     *        the input sequence
     * @param threshold
     *        The object is defined by values strictly greater than this value
     * @param invertMap
     *        set to true (resp. false) to compute the distance map inside (resp. outside) the
     *        object
     * @param type
     *        The algorithm to use
     */
    public void unsignedDistanceMap(Sequence input, double threshold, boolean invertMap, Algorithm type)
    {
        switch (type)
        {
            case Chamfer:
            {
                double[][] map = new double[input.getSizeZ()][input.getWidth() * input.getHeight()];

                int maxDistance = map.length * map[0].length;
                int foregroundInit = invertMap ? maxDistance : 0;
                int backgroundInit = invertMap ? 0 : maxDistance;

                for (int t = 0; t < input.getSizeT(); t++)
                {
                    for (int c = 0; c < input.getSizeC(); c++)
                    {
                        ArrayUtil.arrayToDoubleArray2D(input.getDataXYZ(t, c), map, false);

                        // Initialize the distance map according to the threshold
                        for (int z = 0; z < map.length; z++)
                        {
                            double[] slice = map[z];

                            for (int i = 0; i < slice.length; i++)
                                slice[i] = (slice[i] > threshold ? foregroundInit : backgroundInit);
                        }

                        if (map.length == 1)
                        {
                            updateUnsignedChamferDistance2D(input.getWidth(), input.getHeight(), map[0]);
                        }
                        else
                        {
                            updateUnsignedChamferDistance3D(input.getWidth(), input.getHeight(), map.length, map);
                        }

                        ArrayUtil.doubleArrayToArray2D(map, input.getDataXYZ(t, c));
                    }
                }
            }
                break;

            default:
                throw new IllegalArgumentException("Distance algorithm not supported : " + type);
        }
    }

    private final float a = 1;
    private final float b = 1.4142135623730950488016887242097f;
    private final float c = 1.7320508075688772935274463415059f;

    /**
     * Updates an already-initialized 2D unsigned distance map (background value should be infinity)
     */
    public void updateUnsignedChamferDistance2D(int width, int height, double[] map)
    {
        int i, j, index = 0;
        double aDist, bDist;

        // First pass

        for (j = 0; j < height; j++)
            for (i = 0; i < width; i++, index++)
            {
                // 2 neighbors at distance output = 1
                aDist = map[index] + a;
                // 2 neighbors at distance b = sqrt(2)
                bDist = map[index] + b;

                if (i < width - 1)
                    map[index + 1] = Math.min(map[index + 1], aDist);

                if (j < height - 1)
                {
                    map[index + width] = Math.min(map[index + width], aDist);
                    if (i < width - 1)
                        map[index + 1 + width] = Math.min(map[index + 1 + width], bDist);
                    if (i > 0)
                        map[index - 1 + width] = Math.min(map[index - 1 + width], bDist);
                }
            }

        index--;

        // Second pass

        for (j = height - 1; j >= 0; j--)
            for (i = width - 1; i >= 0; i--, index--)
            {
                // 2 neighbors at distance output = 1
                aDist = map[index] + a;
                // 2 neighbors at distance b = sqrt(2)
                bDist = map[index] + b;

                if (i > 0)
                    map[index - 1] = Math.min(map[index - 1], aDist);

                if (j > 0)
                {
                    map[index - width] = Math.min(map[index - width], aDist);
                    if (i > 0)
                        map[index - 1 - width] = Math.min(map[index - 1 - width], bDist);
                    if (i < width - 1)
                        map[index + 1 - width] = Math.min(map[index + 1 - width], bDist);
                }
            }
    }

    /**
     * Updates an already-initialized 3D unsigned distance map (background value should be infinity)
     */
    public void updateUnsignedChamferDistance3D(int width, int height, int depth, double[][] map)
    {
        int i, j, k, offset = 0;
        double aDist, bDist, cDist;

        // First pass

        for (k = 0; k < depth - 1; k++)
        {
            double[] thisSlice = map[k];
            double[] nextSlice = map[k + 1];

            offset = 0;

            for (j = 0; j < height; j++)
                for (i = 0; i < width; i++, offset++)
                {
                    aDist = thisSlice[offset] + a;
                    bDist = thisSlice[offset] + b;
                    cDist = thisSlice[offset] + c;

                    if (k < depth - 1)
                    {
                        if (nextSlice[offset] > aDist)
                            nextSlice[offset] = aDist;
                        // map[offset + stack] = Math.min(map[offset + stack], aDist);

                        if (j < height - 1)
                        {
                            if (nextSlice[offset + width] > bDist)
                                nextSlice[offset + width] = bDist;
                            // map[offset + width + stack] = Math.min(map[offset + width + stack],
                            // bDist);

                            if (i > 0 && nextSlice[offset + width - 1] > cDist)
                                nextSlice[offset + width - 1] = cDist;
                            // if (i > 0) map[offset - 1 + width + stack] = Math.min(map[offset - 1
                            // + width + stack], cDist);

                            if (i < width - 1 && nextSlice[offset + width + 1] > cDist)
                                nextSlice[offset + width + 1] = cDist;
                            // if (i < width - 1) map[offset + 1 + width + stack] =
                            // Math.min(map[offset + 1 + width + stack], cDist);
                        }

                        if (j > 0)
                        {
                            if (nextSlice[offset - width] > bDist)
                                nextSlice[offset - width] = bDist;
                            // map[offset - width + stack] = Math.min(map[offset - width + stack],
                            // bDist);

                            if (i > 0 && nextSlice[offset - width - 1] > cDist)
                                nextSlice[offset - width - 1] = cDist;
                            // if (i > 0) map[offset - 1 - width + stack] = Math.min(map[offset - 1
                            // - width + stack], cDist);

                            if (i < width - 1 && nextSlice[offset - width + 1] > cDist)
                                nextSlice[offset - width + 1] = cDist;
                            // if (i < width - 1) map[offset + 1 - width + stack] =
                            // Math.min(map[offset + 1 - width + stack], cDist);
                        }

                        if (i > 0 && nextSlice[offset - 1] > bDist)
                            nextSlice[offset - 1] = bDist;
                        // if (i > 0) map[offset - 1 + stack] = Math.min(map[offset - 1 + stack],
                        // bDist);

                        if (i < width - 1 && nextSlice[offset + 1] > bDist)
                            nextSlice[offset + 1] = bDist;
                        // if (i < width - 1) map[offset + 1 + stack] = Math.min(map[offset + 1 +
                        // stack], bDist);
                    }

                    if (j < height - 1)
                    {
                        if (thisSlice[offset + width] > aDist)
                            thisSlice[offset + width] = aDist;
                        // map[offset + width] = Math.min(map[offset + width], aDist);

                        if (i < width - 1 && thisSlice[offset + width + 1] > bDist)
                            thisSlice[offset + width + 1] = bDist;
                        // if (i < width - 1) map[offset + 1 + width] = Math.min(map[offset + 1 +
                        // width], bDist);

                        if (i > 0 && thisSlice[offset + width - 1] > bDist)
                            thisSlice[offset + width - 1] = bDist;
                        // if (i > 0) map[offset - 1 + width] = Math.min(map[offset - 1 + width],
                        // bDist);
                    }

                    if (i < width - 1 && thisSlice[offset + 1] > aDist)
                        thisSlice[offset + 1] = aDist;
                    // if (i < width - 1) map[offset + 1] = Math.min(map[offset + 1], aDist);
                }
        }

        // Second pass

        for (k = depth - 1; k > 0; k--)
        {
            double[] thisSlice = map[k];
            double[] prevSlice = map[k - 1];

            offset = thisSlice.length - 1;

            for (j = height - 1; j >= 0; j--)
                for (i = width - 1; i >= 0; i--, offset--)
                {
                    aDist = thisSlice[offset] + a;
                    bDist = thisSlice[offset] + b;
                    cDist = thisSlice[offset] + c;

                    if (k > 0)
                    {
                        if (prevSlice[offset] > aDist)
                            prevSlice[offset] = aDist;
                        // map[offset - stack] = Math.min(map[offset - stack], aDist);

                        if (j > 0)
                        {
                            if (prevSlice[offset - width] > bDist)
                                prevSlice[offset - width] = bDist;
                            // map[offset - width - stack] = Math.min(map[offset - width - stack],
                            // bDist);

                            if (i > 0 && prevSlice[offset - width - 1] > cDist)
                                prevSlice[offset - width - 1] = cDist;
                            // if (i > 0) map[offset - 1 - width - stack] = Math.min(map[offset - 1
                            // - width - stack], cDist);

                            if (i < width - 1 && prevSlice[offset - width + 1] > cDist)
                                prevSlice[offset - width + 1] = cDist;
                            // if (i < width - 1) map[offset + 1 - width - stack] =
                            // Math.min(map[offset + 1 - width - stack], cDist);
                        }

                        if (j < height - 1)
                        {
                            if (prevSlice[offset + width] > bDist)
                                prevSlice[offset + width] = bDist;
                            // map[offset + width - stack] = Math.min(map[offset + width - stack],
                            // bDist);

                            if (i > 0 && prevSlice[offset + width - 1] > cDist)
                                prevSlice[offset + width - 1] = cDist;
                            // if (i > 0) map[offset - 1 + width - stack] = Math.min(map[offset - 1
                            // + width - stack], cDist);

                            if (i < width - 1 && prevSlice[offset + width + 1] > cDist)
                                prevSlice[offset + width + 1] = cDist;
                            // if (i < width - 1) map[offset + 1 + width - stack] =
                            // Math.min(map[offset + 1 + width - stack], cDist);
                        }

                        if (i > 0 && prevSlice[offset - 1] > bDist)
                            prevSlice[offset - 1] = bDist;
                        // if (i > 0) map[offset - 1 - stack] = Math.min(map[offset - 1 - stack],
                        // bDist);

                        if (i < width - 1 && prevSlice[offset + 1] > bDist)
                            prevSlice[offset + 1] = bDist;
                        // if (i < width - 1) map[offset + 1 - stack] = Math.min(map[offset + 1 -
                        // stack], bDist);
                    }

                    if (j > 0)
                    {
                        if (thisSlice[offset - width] > aDist)
                            thisSlice[offset - width] = aDist;
                        // map[offset - width] = Math.min(map[offset - width], aDist);

                        if (i > 0 && thisSlice[offset - width - 1] > bDist)
                            thisSlice[offset - width - 1] = bDist;
                        // if (i > 0) map[offset - 1 - width] = Math.min(map[offset - 1 - width],
                        // bDist);

                        if (i < width - 1 && thisSlice[offset - width + 1] > bDist)
                            thisSlice[offset - width + 1] = bDist;
                        // if (i < width - 1) map[offset + 1 - width] = Math.min(map[offset + 1 -
                        // width], bDist);
                    }

                    if (i > 0 && thisSlice[offset - 1] > aDist)
                        thisSlice[offset - 1] = aDist;
                    // if (i > 0) map[offset - 1] = Math.min(map[offset - 1], aDist);
                }
        }
    }

}
