/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 ******************************************************************************/
package plugins.big.shapedesigner.keeper;

import java.util.LinkedList;

import plugins.big.shapedesigner.roi.SplineCurveEditMode;

/**
 * Class that encapsulates a list of SplineCurveKeepers.
 * 
 * @version May 3, 2014
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Emrah Bostan (emrah.bostan@gmail.com)
 */
public class KeepersList {

	/** List of all <code>SplineCurveKeeper</code>. */
	private LinkedList<SplineCurveKeeper> splineCurveKeepers_ = null;
	/** <code>SplineCurveKeeper</code> associated to the active spline curve. */
	private SplineCurveKeeper activeSplineCurveKeeper_ = null;

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public KeepersList() {
		splineCurveKeepers_ = new LinkedList<SplineCurveKeeper>();
		activeSplineCurveKeeper_ = null;
	}

	// ----------------------------------------------------------------------------

	/**
	 * If the <code>SplineCurveKeeper</code> passed as a parameter is in the
	 * list, it becomes activated and the method returns <code>true</code>. If
	 * the element is not in the list, the method returns <code>false</code>.
	 */
	public boolean activateSplineCurveKeeper(SplineCurveKeeper keeper) {
		if (splineCurveKeepers_.contains(keeper)) {
			activeSplineCurveKeeper_ = keeper;
			for (SplineCurveKeeper k : splineCurveKeepers_) {
				k.setSelected(k == activeSplineCurveKeeper_);
			}
			return true;
		} else {
			return false;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Activates the i-th <code>SplineCurveKeeper</code> form the internal list.
	 * If the list have less than i-th elements, the method returns
	 * <code>false</code>, otherwise it returns <code>true</code>.
	 */
	public boolean activateSplineCurveKeeper(int i) {
		if (i >= 0 && i < splineCurveKeepers_.size()) {
			return activateSplineCurveKeeper(splineCurveKeepers_.get(i));
		}
		return false;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Adds a <code>SplineCurveKeeper</code> at the end of the list, and sets it
	 * to selected.
	 */
	public void addAndActivateKeeper(SplineCurveKeeper keeper) {
		if (keeper != null) {
			splineCurveKeepers_.add(keeper);
			activateSplineCurveKeeper(keeper);
		}
	}

	// ----------------------------------------------------------------------------

	/** Deactivates the active <code>SplineCurveKeeper</code> if any. */
	public void deactivateActiveSplineCurveKeeper() {
		if (activeSplineCurveKeeper_ != null) {
			activeSplineCurveKeeper_.setSelected(false);
			activeSplineCurveKeeper_ = null;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * If the <code>SplineCurveKeeper</code> passed as a parameter is in the
	 * list, it becomes deactivated and the method returns <code>true</code>. If
	 * the element is not in the list, the method returns <code>false</code>.
	 */
	public boolean deactivateSplineCurveKeeper(SplineCurveKeeper keeper) {
		if (splineCurveKeepers_.contains(keeper)) {
			activeSplineCurveKeeper_ = null;
			keeper.setSelected(false);
			return true;
		} else {
			return false;
		}
	}

	// ----------------------------------------------------------------------------

	/** Returns the active <code>SplineCurveKeeper</code>. */
	public SplineCurveKeeper getActiveSplineCurveKeeper() {
		return activeSplineCurveKeeper_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the number of <code>SplineCurveKeeper</code> in the internal
	 * list.
	 */
	public int getNumKeepers() {
		return splineCurveKeepers_.size();
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns <code>true</code> if the <code>SplineCurveKeeper</code> passed as
	 * a parameters if the active one.
	 */
	public boolean isActiveSplineCurveKeeper(SplineCurveKeeper keeper) {
		return activeSplineCurveKeeper_ == keeper;
	}

	// ----------------------------------------------------------------------------

	/** Returns <code>true</code> is the list contains no elements. */
	public boolean isEmpty() {
		return splineCurveKeepers_.isEmpty();
	}

	// ----------------------------------------------------------------------------

	/**
	 * Removes the active <code>SplineCurveKeeper</code> from the list and sets
	 * to active the next one in the list. If the removed keeper is the last one
	 * in the list,
	 */
	public void removeActiveSplineCurveKeeper() {
		if (activeSplineCurveKeeper_ != null) {
			activeSplineCurveKeeper_.removeFromSequence();
			int index = splineCurveKeepers_.indexOf(activeSplineCurveKeeper_);
			splineCurveKeepers_.remove(index);
			if (!splineCurveKeepers_.isEmpty()) {
				activateSplineCurveKeeper(splineCurveKeepers_.get(Math.min(
						index, splineCurveKeepers_.size() - 1)));
			} else {
				activeSplineCurveKeeper_ = null;
			}
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Removes all of the elements from the list, and removes the
	 * <code>SplineCurveKeeper</code> from the associated images.
	 */
	public void removeAllSplineCurveKeepers() {
		for (SplineCurveKeeper keeper : splineCurveKeepers_) {
			if (keeper != null) {
				keeper.removeFromSequence();
			}
		}
		splineCurveKeepers_.clear();
		activeSplineCurveKeeper_ = null;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Sets the <code>SplineCurveEditMode</code> to all elements of
	 * <code>SplineCurveKeeper</code> contained in the list.
	 */
	public void setSplineCurveEditMode(SplineCurveEditMode editingMode) {
		for (SplineCurveKeeper keeper : splineCurveKeepers_) {
			if (keeper != null) {
				keeper.setSplineCurveEditMode(editingMode);
			}
		}
	}
}
