/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnake.io;

import icy.file.FileUtil;
import icy.gui.frame.progress.AnnounceFrame;
import icy.roi.ROI;
import icy.sequence.Sequence;
import icy.util.XMLUtil;

import java.io.File;
import java.util.ArrayList;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileNameExtensionFilter;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

import plugins.big.bigsnake.BIGSnake;
import plugins.big.bigsnake.core.ImageLUTContainer;
import plugins.big.bigsnake.keeper.SnakeKeeper;
import plugins.big.bigsnake.roi.ROI2DSnake;
import plugins.big.bigsnake.snake.ESnake;
import plugins.big.bigsnake.snake.ESnakeParameters;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Custom;

/**
 * Class that provides loading and exporting capabilities to the snake.
 * 
 * @version May 24, 2015
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Daniel Schmitter (daniel.schmitter@epfl.ch)
 */
public class IOXMLUtils {

	/**
	 * Label of the XML tag the metadata of the image from which the XML was
	 * generated.
	 */
	private final static String ROOT_META = "meta";
	/**
	 * Label of the XML tag containing the name of the image from which the XML
	 * was generated.
	 */
	private final static String ID_NAME = "name";
	/**
	 * Label of the XML tag containing the the information associated to the ROI
	 * of a collection of snakes.
	 */
	private final static String ROOT_ROIS = "rois";
	/**
	 * Label of the XML tag containing the the information associated to the ROI
	 * of a single snake.
	 */
	private final static String ID_ROI = "roi";

	/**
	 * File chooser used to select the source or destination file in the fily
	 * system.
	 */
	private JFileChooser fileChooser_ = null;

	/** Returned value from the <code>JFileChooser</code> */
	private int returnVal_;

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public IOXMLUtils() {
		String userDirLocation = System.getProperty("user.dir");
		File userDir = new File(userDirLocation);
		fileChooser_ = new JFileChooser(userDir);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Opens a dialog and saves all the snakes in the active
	 * <code>Sequence</code> in an XML file.
	 */
	public void saveSnakesToXML(ImageLUTContainer imageLUT) throws Exception {
		if (imageLUT == null) {
			throw new Exception("imageLUT is null in saveSnakesToXML.");
		}

		Sequence sequence = imageLUT.getOriginalSequence();
		if (sequence == null) {
			throw new Exception(
					"sequence is null within imageLUT in saveSnakesToXML.");
		}

		try {
			SwingUtilities.invokeAndWait(new Runnable() {
				@Override
				public void run() {
					returnVal_ = fileChooser_.showSaveDialog(new JFrame());
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
		if (returnVal_ == JFileChooser.APPROVE_OPTION) {
			String filename = fileChooser_.getSelectedFile().getAbsolutePath();
			String xmlFilename = filename + ".xml";
			Document document = XMLUtil.createDocument(true);
			XMLUtil.setElementValue(document.getDocumentElement(), ID_NAME,
					sequence.getName());
			addMetaDataToXML(document.getDocumentElement(), imageLUT);
			addROIsToXML(document.getDocumentElement(), sequence);
			XMLUtil.saveDocument(document, xmlFilename);
		}
	}

	// ----------------------------------------------------------------------------

	/** Opens a dialog to select an XML file, and loads all a prior shape. */
	public Custom loadPriorShapeFromXML() throws Exception {
		try {
			SwingUtilities.invokeAndWait(new Runnable() {
				@Override
				public void run() {
					fileChooser_.setFileFilter(new FileNameExtensionFilter(
							"XML Files", "xml"));
					returnVal_ = fileChooser_.showOpenDialog(new JFrame());
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
		if (returnVal_ == JFileChooser.APPROVE_OPTION) {
			String xmlFilename = fileChooser_.getSelectedFile()
					.getAbsolutePath();
			Document document = null;
			if (xmlFilename != null) {
				if (FileUtil.exists(xmlFilename)) {
					document = XMLUtil.loadDocument(xmlFilename, true);
				} else {
					new AnnounceFrame("Selected file does not exist.");
					throw new Exception(
							"xmlFilename is null in loadPriorShapeFromXML.");
				}
			} else {
				throw new Exception(
						"xmlFilename is null in loadPriorShapeFromXML.");
			}

			final Node nodeROIs = XMLUtil.getElement(
					document.getDocumentElement(), ROOT_ROIS);

			if (nodeROIs != null) {
				ArrayList<Snake2DNode[]> controlPointsList = loadSnakeNodesFromXML(nodeROIs);
				return new Custom(controlPointsList.get(0));
			}
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Opens a dialog to select an XML file, and loads all the snakes in the
	 * active <code>Sequence</code>.
	 */
	public ArrayList<SnakeKeeper> loadSnakesFromXML(ImageLUTContainer imageLUT,
			BIGSnake mainPlugin) throws Exception {
		if (imageLUT == null) {
			throw new Exception("imageLUT is null in loadSnakesFromXML.");
		}

		Sequence sequence = imageLUT.getOriginalSequence();

		if (sequence == null) {
			throw new Exception("sequence is null in loadSnakesFromXML.");
		}

		try {
			SwingUtilities.invokeAndWait(new Runnable() {
				@Override
				public void run() {
					fileChooser_.setFileFilter(new FileNameExtensionFilter(
							"XML Files", "xml"));
					returnVal_ = fileChooser_.showOpenDialog(new JFrame());
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
		if (returnVal_ == JFileChooser.APPROVE_OPTION) {
			String xmlFilename = fileChooser_.getSelectedFile()
					.getAbsolutePath();
			Document document = null;
			if (xmlFilename != null) {
				if (FileUtil.exists(xmlFilename)) {
					document = XMLUtil.loadDocument(xmlFilename, true);
				} else {
					new AnnounceFrame("Selected file does not exist.");
					throw new Exception(
							"xmlFilename is null in loadSnakesFromXML.");
				}
			} else {
				throw new Exception("xmlFilename is null in loadSnakesFromXML.");
			}

			loadMetaDataFromXML(document.getDocumentElement(), imageLUT);

			ArrayList<SnakeKeeper> loadedKeepers = new ArrayList<SnakeKeeper>();

			final Node nodeROIs = XMLUtil.getElement(
					document.getDocumentElement(), ROOT_ROIS);

			if (nodeROIs != null) {
				ArrayList<Snake2DNode[]> controlPointsList = loadSnakeNodesFromXML(nodeROIs);
				ArrayList<ESnakeParameters> snakeParametersList = loadSnakeParametersFromXML(nodeROIs);

				int detectedSnakes = controlPointsList.size();
				for (int i = 0; i < detectedSnakes; i++) {
					ESnake snake = new ESnake(imageLUT,
							snakeParametersList.get(i), null, null);
					snake.setNodes(controlPointsList.get(i));
					loadedKeepers.add(new SnakeKeeper(sequence, snake,
							mainPlugin));
				}
				return loadedKeepers;
			}
		}
		return null;
	}

	// ============================================================================
	// PRIVATE METHODS

	private void addMetaDataToXML(Node node, ImageLUTContainer imageLUT)
			throws Exception {
		if (node == null) {
			throw new Exception("XML node is null in addMetaDataToXML.");
		}
		if (imageLUT == null) {
			throw new Exception("imageLUT is null in addMetaDataToXML.");
		}

		Sequence sequence = imageLUT.getOriginalSequence();
		if (sequence == null) {
			throw new Exception(
					"sequence is null within imageLUT in addMetaDataToXML.");
		}

		final Node nodeMeta = XMLUtil.setElement(node, ROOT_META);
		if (nodeMeta != null) {
			XMLUtil.setElementDoubleValue(nodeMeta, Sequence.ID_PIXEL_SIZE_X,
					sequence.getPixelSizeX());
			XMLUtil.setElementDoubleValue(nodeMeta, Sequence.ID_PIXEL_SIZE_Y,
					sequence.getPixelSizeY());
			XMLUtil.setElementDoubleValue(nodeMeta, Sequence.ID_PIXEL_SIZE_Z,
					sequence.getPixelSizeZ());
			XMLUtil.setElementDoubleValue(nodeMeta, Sequence.ID_TIME_INTERVAL,
					sequence.getTimeInterval());
			XMLUtil.setElementDoubleValue(nodeMeta, ImageLUTContainer.ID_SIGMA,
					imageLUT.getSigma());
			XMLUtil.setElementDoubleValue(nodeMeta,
					ImageLUTContainer.ID_ACTIVECHANNEL,
					imageLUT.getChannelNumber());
			for (int c = 0; c < sequence.getSizeC(); c++) {
				XMLUtil.setElementValue(nodeMeta, Sequence.ID_CHANNEL_NAME + c,
						sequence.getChannelName(c));
			}
		}
	}

	// ----------------------------------------------------------------------------

	private void addROIsToXML(Node node, Sequence sequence) throws Exception {
		if (node == null) {
			throw new Exception("XML node is null in addROIsToXML.");
		}
		if (sequence == null) {
			throw new Exception("sequence is null in addROIsToXML.");
		}

		final Node nodeROIs = XMLUtil.setElement(node, ROOT_ROIS);
		if (nodeROIs != null) {
			XMLUtil.removeAllChildren(nodeROIs);
			for (ROI roi : sequence.getROIs()) {
				final Node nodeROI = XMLUtil.addElement(nodeROIs, ID_ROI);
				if (nodeROI != null) {
					if (!roi.saveToXML(nodeROI)) {
						XMLUtil.removeNode(nodeROIs, nodeROI);
					}
				}
			}
		}
	}

	// ----------------------------------------------------------------------------

	private void loadMetaDataFromXML(Node node, ImageLUTContainer imageLUT)
			throws Exception {

		if (node == null) {
			throw new Exception("XML node is null in loadMetaDataFromXML.");
		}

		if (imageLUT == null) {
			throw new Exception("imageLUT is null in loadMetaDataFromXML.");
		}

		Sequence sequence = imageLUT.getOriginalSequence();

		if (sequence == null) {
			throw new Exception("Sequence is null in loadMetaDataFromXML.");
		}

		final Node nodeMeta = XMLUtil.getElement(node, ROOT_META);
		if (nodeMeta != null) {
			imageLUT.setChannelNumber(XMLUtil.getElementIntValue(nodeMeta,
					ImageLUTContainer.ID_ACTIVECHANNEL, 0));
			imageLUT.setSigma(XMLUtil.getElementDoubleValue(nodeMeta,
					ImageLUTContainer.ID_SIGMA, 10d));
		}
		if (!imageLUT.isLUTUpToDate()) {
			imageLUT.buildLUTs();
		}
	}

	// ----------------------------------------------------------------------------

	private ArrayList<Snake2DNode[]> loadSnakeNodesFromXML(Node nodeROIs)
			throws Exception {

		if (nodeROIs == null) {
			throw new Exception("XML node is null in loadSnakeNodesFromXML.");
		}
		final ArrayList<Node> nodesROI = XMLUtil.getChildren(nodeROIs, ID_ROI);
		ArrayList<Snake2DNode[]> controlPointsList = new ArrayList<Snake2DNode[]>();
		if (nodesROI != null) {
			for (Node n : nodesROI) {
				final Node snakeParametersNode = XMLUtil.getElement(n,
						ROI2DSnake.ID_SNAKE_PARAMETERS);
				final Node controlPointsNode = XMLUtil.getElement(
						snakeParametersNode, ESnake.ID_CONTROL_POINTS);
				final ArrayList<Node> controlPointNodeList = XMLUtil
						.getChildren(controlPointsNode, ESnake.ID_CONTROL_POINT);
				int M = controlPointNodeList.size();
				Snake2DNode[] controlPoints = new Snake2DNode[M];
				for (int i = 0; i < M; i++) {
					controlPoints[i] = new Snake2DNode(0, 0);
					controlPoints[i].loadFromXML(controlPointNodeList.get(i));
				}
				controlPointsList.add(controlPoints);
			}
			return controlPointsList;
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	private ArrayList<ESnakeParameters> loadSnakeParametersFromXML(Node nodeROIs)
			throws Exception {

		if (nodeROIs == null) {
			throw new Exception(
					"XML node is null in loadSnakeParametersFromXML.");
		}
		final ArrayList<Node> nodesROI = XMLUtil.getChildren(nodeROIs, ID_ROI);
		ArrayList<ESnakeParameters> snakeParametersList = new ArrayList<ESnakeParameters>();
		if (nodesROI != null) {
			for (Node n : nodesROI) {
				final Node snakeParametersNode = XMLUtil.getElement(n,
						ROI2DSnake.ID_SNAKE_PARAMETERS);
				ESnakeParameters snakeParameters = new ESnakeParameters();
				snakeParameters.loadFromToXML(snakeParametersNode);
				snakeParametersList.add(snakeParameters);
			}
			return snakeParametersList;
		}
		return null;
	}
}
