package plugins.adufour.tutorial;

import icy.gui.frame.progress.AnnounceFrame;
import icy.main.Icy;
import icy.swimmingPool.SwimmingObject;
import plugins.adufour.ezplug.*;

/**
 * Tutorial on how to use the EzPlug library to write plugins fast and efficiently
 * 
 * @author Alexandre Dufour
 * 
 */
public class EzPlugTutorial extends EzPlug implements EzStoppable
{
	private enum SomeEnumeration
	{
		CHOICE_1, CHOICE_2, CHOICE_3, SOME_other_CHOICE,
	}
	
	// declare here the variables you wish to use
	// in this tutorial I will use one of each
	
	EzVarBoolean					varBoolean;
	EzVarDouble						varDouble;
	EzVarDoubleArray				varDoubleA;
	EzVarEnum<SomeEnumeration>		varEnum;
	EzVarFile						varFile;
	EzVarFileArray					varFileA;
	EzVarFolder						varFolder;
	EzVarInteger					varInt;
	EzVarIntegerArray				varIntA;
	EzVarSequence					varSequence;
	EzVarText						varText;
	EzVarSwimmingObject<Integer>	varSwimmingInteger;
	
	// some other data
	boolean							stopFlag;
	
	@Override
	protected void initialize()
	{
		// 1) variables must be initialized
		
		varBoolean = new EzVarBoolean("boolean var.", false);
		varDouble = new EzVarDouble("double with free input");
		varDoubleA = new EzVarDoubleArray("double array without input", new Double[][] { new Double[] { 2.2, 4.4, 6.6 }, new Double[] { 1.1, 3.3, 5.5 } }, false);
		varEnum = new EzVarEnum<SomeEnumeration>("enumeration", SomeEnumeration.values(), SomeEnumeration.SOME_other_CHOICE);
		varFile = new EzVarFile("some file", null);
		varFileA = new EzVarFileArray("some files", null);
		varFolder = new EzVarFolder("a folder", null);
		varInt = new EzVarInteger("bounded integer", 4, -34, 28, 2);
		varIntA = new EzVarIntegerArray("int array with input", new Integer[0][0], true);
		varSequence = new EzVarSequence("Input sequence");
		varText = new EzVarText("Some text", new String[] { "yes", "no", "maybe?" }, 2, true);
		varSwimmingInteger = new EzVarSwimmingObject<Integer>("Swimming object");
		
		// 2) and added to the interface in the desired order
		
		super.addEzComponent(varText);
		super.addEzComponent(varEnum);
		
		// let's group other variables per type
		
		EzGroup groupNumeric = new EzGroup("Numeric variables", varDouble, varInt, varDoubleA, varIntA);
		super.addEzComponent(groupNumeric);
		
		EzGroup groupFiles = new EzGroup("File choosers", varFile, varFileA, varFolder);
		super.addEzComponent(groupFiles);
		
		// let's add a description label
		EzLabel label = new EzLabel("Check above to show/hide a variable");
		EzGroup groupSequence = new EzGroup("Sequence group", varBoolean, label, varSequence);
		super.addEzComponent(groupSequence);
		
		// now let's add a trigger that shows varSequence when varBoolean is checked...
		varBoolean.addVisibilityTriggerTo(varSequence, true);
		// or when varEnum is on "CHOICE_2" or "CHOICE_3"
		varEnum.addVisibilityTriggerTo(varSequence, SomeEnumeration.CHOICE_2, SomeEnumeration.CHOICE_3);
		
		super.addEzComponent(varSwimmingInteger);
	}
	
	@Override
	protected void execute()
	{
		// main plugin code goes here, and runs in a separate thread
		
		System.out.println(varBoolean.name + " = " + varBoolean.getValue());
		System.out.println(varDouble.name + " = " + varDouble.getValue());
		System.out.println(varDoubleA.name + " = " + varDoubleA.getValue());
		System.out.println(varEnum.name + " = " + varEnum.getValue().name());
		System.out.println(varFile.name + " = " + varFile.getValue());
		System.out.println(varSequence.name + " = " + varSequence.getValue());
		new AnnounceFrame(varText.getValue());
		
		stopFlag = false;
		
		Icy.getMainInterface().getSwimmingPool().add(new SwimmingObject(varInt.getValue(), "value of " + varInt.name));
		
		super.getUI().setProgressBarMessage("Waiting...");
		
		int cpt = 0;
		while (!stopFlag)
		{
			cpt++;
			if (cpt % 10 == 0) super.getUI().setProgressBarValue((cpt % 5000000) / 5000000.0);
			Thread.yield();
		}
	}
	
	@Override
	public void clean()
	{
		// use this method to clean local variables or input streams (if any) to avoid memory leaks
		
	}
	
	@Override
	public void stopExecution()
	{
		// this method is from the EzStoppable interface
		// if this interface is implemented, a "stop" button is displayed
		// and this method is called when the user hits the "stop" button
		stopFlag = true;
	}
	
}
