/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.snake2D;

import icy.util.XMLUtil;

import java.awt.Color;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.util.ArrayList;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * This class is used to store the scales that are used to draw the skin of the
 * snake. It extends the capabilities of the class
 * <code>Path2D<code> by additional 
 * state variables.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 */
public class Snake2DScale extends Path2D.Double {

	/**
	 * Determines if a de-serialized file is compatible with this class.
	 * 
	 * Maintainers must change this value if and only if the new version of this
	 * class is not compatible with old versions. See Sun docs for <a
	 * href=http://java.sun.com/products/jdk/1.1/docs/guide
	 * /serialization/spec/version.doc.html> details. </a>
	 * 
	 * Not necessary to include in first version of the class, but included here
	 * as a reminder of its importance.
	 */
	private static final long serialVersionUID = 7880294057351993774L;

	/** This is the color with which this scale will be drawn. */
	private Color color_ = Color.RED;

	/**
	 * This flag is used to determine how to draw the outline of this scale. If
	 * it is set to <code>true</code>, then the first and the last point of the
	 * polygon are joined. Otherwise, if it is set to <code>false</code>, then
	 * the first and the last point of the polygon are not joined.
	 */
	private boolean closed_ = true;

	/** Width of the stroke. */
	private float strokeWidth_ = 1;

	/** XML tag for each point element. */
	private static final String ID_POINT = "point";
	/** XML tag for the x coordinate of each point. */
	private static final String ID_POINT_XPOS = "x_position";
	/** XML tag for the y coordinate of each point. */
	private static final String ID_POINT_YPOS = "y_position";
	/** XML tag for the color of the scale. */
	private static final String ID_COLOR = "color";
	/** XML tag to inform if the scale is closed or not. */
	private static final String ID_CLOSED = "closed";
	
	public float[] xpoints = null;
	public float[] ypoints = null;
	public int npoints;

	// ============================================================================
	// PUBLIC METHODS

	/**
	 * This constructor builds a scale with default values. Points can be added
	 * with the method <code>add()</code> of the class <code>Polygon</code>. The
	 * default color <code>color</code>is set to <code>Color.RED</code>. The
	 * scale is closed by default.
	 */
	public Snake2DScale() {
		super();
	}

	// ----------------------------------------------------------------------------

	/** This constructor builds a scale with the provided colors and flags. */
	public Snake2DScale(Color color, boolean closed) {
		super();
		color_ = color;
		closed_ = closed;
	}

	// ----------------------------------------------------------------------------

	/** This constructor builds a scale with the provided colors and flags. */
	public Snake2DScale(double[] xpoints, double[] ypoints, int npoints,
			Color color, boolean closed) {
		for (int i = 0; i < npoints; i++) {
			if (i == 0) {
				moveTo(xpoints[i], ypoints[i]);
			} else {
				lineTo(xpoints[i], ypoints[i]);
			}
		}
		if (closed) {
			lineTo(xpoints[0], ypoints[0]);
		}
		color_ = color;
		closed_ = closed;
		
		this.xpoints = doubleArrayToFloat(xpoints);
		this.ypoints = doubleArrayToFloat(ypoints);
		this.npoints = npoints;
	}
	
	/** This constructor builds a scale with the provided colors and flags. */
	public Snake2DScale(float[] xpoints, float[] ypoints, int npoints,
			Color color, boolean closed) {
		for (int i = 0; i < npoints; i++) {
			if (i == 0) {
				moveTo(xpoints[i], ypoints[i]);
			} else {
				lineTo(xpoints[i], ypoints[i]);
			}
		}
		if (closed) {
			lineTo(xpoints[0], ypoints[0]);
		}
		color_ = color;
		closed_ = closed;
		
		this.xpoints = xpoints.clone();
		this.ypoints = ypoints.clone();
		this.npoints = npoints;
	}
	
	private float[] doubleArrayToFloat(double[] arr){
		int n = arr.length;
		float[] arrFl = new float[n];
		
		for (int i=0; i<n; i++)
			arrFl[i]= (float)arr[i];
		
		return arrFl;
	}

	// ----------------------------------------------------------------------------

	/** This method returns text-based information about this object. */
	@Override
	public String toString() {
		return ("[" + super.toString() + ", color: " + color_ + ", closed: "
				+ closed_ + "]");
	}

	// ----------------------------------------------------------------------------

	/** Saves the scale in an XML node. */
	public void saveToXML(Node node) {
		XMLUtil.setElementIntValue(node, ID_COLOR, color_.getRGB());
		XMLUtil.setElementBooleanValue(node, ID_CLOSED, closed_);

		PathIterator pathIt = getPathIterator(null);
		double[] coords = new double[6];
		while (!pathIt.isDone()) {
			int type = pathIt.currentSegment(coords);
			if (type != PathIterator.SEG_CLOSE) {
				Element e = XMLUtil.addElement(node, ID_POINT);
				XMLUtil.setElementDoubleValue(e, ID_POINT_XPOS, coords[0]);
				XMLUtil.setElementDoubleValue(e, ID_POINT_YPOS, coords[1]);
			}
			pathIt.next();
		}
	}

	// ----------------------------------------------------------------------------

	/** Loads the scale from an XML node. */
	public void loadFromXML(Node node) {
		color_ = new Color(XMLUtil.getElementIntValue(node, ID_COLOR,
				color_.getRGB()));
		closed_ = XMLUtil.getElementBooleanValue(node, ID_CLOSED, closed_);

		ArrayList<Element> pointElements = XMLUtil.getElements(node, ID_POINT);
		for (Element e : pointElements) {
			double xpos = XMLUtil.getElementDoubleValue(e, ID_POINT_XPOS, 0);
			double ypos = XMLUtil.getElementDoubleValue(e, ID_POINT_YPOS, 0);
			if (this.getCurrentPoint() == null) {
				moveTo(xpos, ypos);
			} else {
				lineTo(xpos, ypos);
			}
		}
	}

	// ----------------------------------------------------------------------------

	/** Returns the associated color of the scale. */
	public Color getColor() {
		return color_;
	}
	
	public boolean isClosed(){ return closed_; }

	// ----------------------------------------------------------------------------

	/** Returns the associated stroke width of the scale. */
	public float getStrokeWidth() {
		return strokeWidth_;
	}

	// ----------------------------------------------------------------------------

	/** Returns the associated stroke width of the scale. */
	public void setStrokeWidth(float strokeWidth) {
		strokeWidth_ = strokeWidth;
	}

}
