package plugins.spop.advancefilters;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.type.collection.array.Array1DUtil;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.ezplug.EzVarText;
import plugins.adufour.filtering.Convolution1D;
import plugins.adufour.filtering.Kernels1D;

public class PeronaMalikFilter extends EzPlug implements Block
{
	/**
	 * PDE Non-linear Filters
	 * 
	 * @author Sorin POP
	 * 
	 * @see P. Perona, J. Malik Scale space and edge detection using anisotropic diffusion, IEEE Transactions on Pattern Analysis and Machine Intelligence, vol.12, no.7, pp.629-639, 1990.
	 * @see F. Catte, P.L Lions, J.M. Morel, T Coll   Image selective smoothing and edge detection by nonlinear diffusion I, SIAM Journal on Numerical Analysis, vol.29, no.1, pp. 182-193, 1992.
	 */

    EzVarSequence input = new EzVarSequence("Input");
    EzVarText type_2D = new EzVarText("Type", new String[] {"2D", "3D"},0, false);
   
    EzVarInteger iterations = new EzVarInteger("Iterations", 30, 1, 10000, 1);
    EzVarDouble step = new EzVarDouble("Discretization step", 0.15,0.01, 0.25, 0.01);
    //K threshold for the rational function g(s)=1/(1+s*s/K*K)
    EzVarDouble thr = new EzVarDouble("Threshold", 7.5, 1, 5000, 0.1);
    EzGroup groupcatte;
    EzVarBoolean catte = new EzVarBoolean("Gaussian regularization", true);
    EzVarDouble sigmaX = new EzVarDouble("Sigma", 1, 0.1, 10, 0.1);
    
    
    EzVarSequence output = new EzVarSequence("Output");
   
    long time = 0;

    @Override
    public void initialize()
    {
        // TODO Auto-generated method stub
        addEzComponent(input);
        addEzComponent(type_2D);
        addEzComponent(iterations);
        addEzComponent(step);
        addEzComponent(thr);
        addEzComponent(catte);
        groupcatte = new EzGroup("Catte version", sigmaX);
        addEzComponent(groupcatte);
        catte.addVisibilityTriggerTo(groupcatte, true);
        setTimeDisplay(true);

    }

    @Override
    public void execute()
    {
        
        double dt = step.getValue();
      
        int iter = iterations.getValue();
        double k = thr.getValue();
        double sX = sigmaX.getValue();
       
        Sequence sequence = input.getValue(true);
        Sequence sequence_out;
       
        if (k == 0.0)
            throw new IllegalArgumentException("Division by O. K = " + Double.toString(k));

	
        if (type_2D.getValue().equalsIgnoreCase("3D"))
        {
        	 if (catte.getValue())// 3D regularized version
          	   sequence_out=Catte_3D(sequence, iter, dt, k, sX);
             else // 3D classical version
          	   sequence_out=Perona_Malik_3D(sequence, iter, dt, k);
            
        }	       
        else
        {
        	
        	if (catte.getValue())// 2D regularized version
            	sequence_out=Catte(sequence, iter, dt, k, sX);
            else// 3D classical version
            	sequence_out=Perona_Malik(sequence, iter, dt, k); 
        }
        
        if(getUI()!=null)
			addSequence(sequence_out);
		output.setValue(sequence_out);
       

    }

    public Sequence Perona_Malik(Sequence sequence,  int iter, double dt, double k)
    {
        int temp = 0;
        double Dx1, Dx2, Dy1, Dy2;
        double current;
        //square of k -used in the diffusion function
        k = k * k;
        
        Sequence seq_out = new Sequence();       
        int dim_x = sequence.getSizeX();
        int dim_y = sequence.getSizeY();
        int dim_z = sequence.getSizeZ();            
        int dim_t = sequence.getSizeT();
        int dim_c = sequence.getSizeC();
        
               
        double[] tab_out = new double[dim_x*dim_y];
      
        for (int time_slice = 0; time_slice < dim_t; time_slice++)
        {
        	for (int z = 0; z < dim_z; z++)
        	{
        	
	    		IcyBufferedImage icy_img = new IcyBufferedImage(dim_x, dim_y, dim_c, sequence.getDataType());
	        	for (int c = 0; c < dim_c; c++)
	        	{
		            double[] tab = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice, z, c), false);
		           		           
		            System.arraycopy(tab, 0, tab_out, 0, dim_x*dim_y);
		           	         
		            for (int t = 0; t < iter; t++)
		            {
		                for (int x = 1; x < dim_x - 1; x++)
	                    for (int y = 1; y < dim_y - 1; y++)
	                    {
	                        temp = x + y * dim_x;
	                        current = tab[temp];
	                        // first order derivative (forward and backward approximations)
	                        Dx1 = tab[temp - 1] - current;
	                        Dx2 = tab[temp + 1] - current;
	                        Dy1 = tab[temp - dim_x] - current;
	                        Dy2 = tab[temp + dim_x] - current;
	
	                        tab_out[temp] = current+ dt * (DiffusionTools.PM_function(Dx1, k) * Dx1 + DiffusionTools.PM_function(Dx2, k) * Dx2 + DiffusionTools.PM_function(Dy1, k) * Dy1 + DiffusionTools.PM_function(Dy2, k) * Dy2);
	                    }
		                System.arraycopy(tab_out, 0, tab, 0, dim_x * dim_y);
		             
		                if(getUI()!=null)
		                	getUI().setProgressBarValue((double) (time_slice*dim_z*dim_c*iter+z*dim_c*iter+c*iter+t+1) / (double) (dim_t*dim_z*dim_c*iter));
		               
		               
		            }
		            Array1DUtil.doubleArrayToSafeArray(tab_out, icy_img.getDataXY(c), icy_img.isSignedDataType());		
		           
	        	}
	        	seq_out.setImage(time_slice, z, icy_img);
	        	
        	}
        	       	
        }
        seq_out.updateComponentsBounds();
        seq_out.setName("Diffusion Perona-Malik 2D");
        return seq_out;

    }

    public Sequence Catte(Sequence sequence,  int iter, double dt, double k, double sigma)
    {
    	int temp = 0;
        double Dx1, Dx2, Dy1, Dy2, Dx1g, Dx2g, Dy1g, Dy2g;
        double current, current_g;
       
        k = k * k;
        
        Sequence seq_out = new Sequence();       
        int dim_x = sequence.getSizeX();
        int dim_y = sequence.getSizeY();
        int dim_z = sequence.getSizeZ();            
        int dim_t = sequence.getSizeT();
        int dim_c = sequence.getSizeC();
             
        double[] tab_out = new double[dim_x*dim_y];
        
        double[] ker= Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigma).getData();
	
		double[][] tab_gauss = new double[1][dim_x*dim_y];
		
        for (int time_slice = 0; time_slice < dim_t; time_slice++)
        {
        	
        	for (int z = 0; z < dim_z; z++)
        	{
        	
	    		IcyBufferedImage icy_img = new IcyBufferedImage(dim_x, dim_y, dim_c, sequence.getDataType());
	        	for (int c = 0; c < dim_c; c++)
	        	{
		            double[] tab = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice, z, c), false);
		            System.arraycopy(tab, 0, tab_gauss[0], 0, dim_x*dim_y);
		            System.arraycopy(tab, 0, tab_out, 0, dim_x*dim_y);
		           	         
		            for (int t = 0; t < iter; t++)
		            {
		            	try {
							Convolution1D.convolve(tab_gauss, dim_x,dim_y,ker, ker,null);
		            	 } catch (Exception e) {
		         			// TODO Auto-generated catch block
		         			e.printStackTrace();
		         		} 
		                for (int x = 1; x < dim_x - 1; x++)
		                    for (int y = 1; y < dim_y - 1; y++)
		                    {
		                        temp = x + y * dim_x;
		                        current = tab[temp];
		                        Dx1 = tab[temp - 1] - current;
		                        Dx2 = tab[temp + 1] - current;
		                        Dy1 = tab[temp - dim_x] - current;
		                        Dy2 = tab[temp + dim_x] - current;
		                        
		                        current_g = tab_gauss[0][temp];
		                        Dx1g = tab_gauss[0][temp - 1] - current_g;
		                        Dx2g = tab_gauss[0][temp + 1] - current_g;
		                        Dy1g = tab_gauss[0][temp - dim_x] - current_g;
		                        Dy2g = tab_gauss[0][temp + dim_x] - current_g;
		
		                        tab_out[temp] = current+ dt * (DiffusionTools.PM_function(Dx1g, k) * Dx1 + DiffusionTools.PM_function(Dx2g, k) * Dx2 + DiffusionTools.PM_function(Dy1g, k) * Dy1 + DiffusionTools.PM_function(Dy2g, k) * Dy2);
		                    }
		                System.arraycopy(tab_out, 0, tab, 0, dim_x * dim_y);
		                System.arraycopy(tab_out, 0, tab_gauss[0], 0, dim_x * dim_y);
		                if(getUI()!=null)
		                	getUI().setProgressBarValue((double) (time_slice*dim_z*dim_c*iter+z*dim_c*iter+c*iter+t+1) / (double) (dim_t*dim_z*dim_c*iter));
			               
		               
		            }
		            Array1DUtil.doubleArrayToSafeArray(tab_out, icy_img.getDataXY(c), icy_img.isSignedDataType());
		            
	        	}
	        	seq_out.setImage(time_slice, z, icy_img);
	        	
        	}
        	
        }
        seq_out.updateComponentsBounds();
        seq_out.setName("Diffusion Catte 2D");
        return seq_out;

    }

    public Sequence Perona_Malik_3D(Sequence sequence, int iter, double dt, double k)
    {
    	int temp = 0;
        double Dx1, Dx2, Dy1, Dy2, Dz1, Dz2;
        double current;
     
        k = k * k;
        
        Sequence seq_out = new Sequence();       
        int dim_x = sequence.getSizeX();
        int dim_y = sequence.getSizeY();
        int dim_z = sequence.getSizeZ();
        int dim_t = sequence.getSizeT();
        int dim_c = sequence.getSizeC();
        
        double[][] tab     = new double[dim_z][];
        double[][] tab_out = new double[dim_z][dim_x*dim_y];
        
        
        for (int time_slice = 0; time_slice < dim_t; time_slice++)
        {
        		IcyBufferedImage[] icy_img = new IcyBufferedImage[dim_z];
        		for (int z = 0; z < dim_z; z++)
        			icy_img[z] = new IcyBufferedImage(dim_x, dim_y, dim_c, sequence.getDataType());
        		
	        	for (int c = 0; c < dim_c; c++)
	        	{
	        		for (int z = 0; z < dim_z; z++)
	        		{
	        			tab[z] = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice, z, c), false);
	        			System.arraycopy(tab[z], 0, tab_out[z], 0, dim_x*dim_y);
	        		}
		           	         
		            for (int t = 0; t < iter; t++)
		            {
		            	for (int z = 0; z < dim_z ; z++)
		            	{
		            		if (z == 0 || z == dim_z - 1)
		            		{
		            			for (int x = 1; x < dim_x - 1; x++)
			                    for (int y = 1; y < dim_y - 1; y++)
			                    {
			                        temp = x + y * dim_x;
			                        current = tab[z][temp];
			                        Dx1 = tab[z][temp - 1] - current;
			                        Dx2 = tab[z][temp + 1] - current;
			                        Dy1 = tab[z][temp - dim_x] - current;
			                        Dy2 = tab[z][temp + dim_x] - current;
			                       		
			                        tab_out[z][temp] = current+ dt * (DiffusionTools.PM_function(Dx1, k) * Dx1 + DiffusionTools.PM_function(Dx2, k) * Dx2 + DiffusionTools.PM_function(Dy1, k) * Dy1 + DiffusionTools.PM_function(Dy2, k) * Dy2);
			                    }
		            			
		            		}
		            		else
		            		{
				                for (int x = 1; x < dim_x - 1; x++)
			                    for (int y = 1; y < dim_y - 1; y++)
			                    {
			                        temp = x + y * dim_x;
			                        current = tab[z][temp];
			                        Dx1 = tab[z][temp - 1] - current;
			                        Dx2 = tab[z][temp + 1] - current;
			                        Dy1 = tab[z][temp - dim_x] - current;
			                        Dy2 = tab[z][temp + dim_x] - current;
			                        Dz1 = tab[z - 1][temp] - current;
			                        Dz2 = tab[z + 1][temp] - current;
			
			                        tab_out[z][temp] = current+ dt * (DiffusionTools.PM_function(Dx1, k) * Dx1 + DiffusionTools.PM_function(Dx2, k) * Dx2 + DiffusionTools.PM_function(Dy1, k) * Dy1 + DiffusionTools.PM_function(Dy2, k) * Dy2
			                        		+DiffusionTools.PM_function(Dz1, k) * Dz1 + DiffusionTools.PM_function(Dz2, k) * Dz2);
			                    }
		            		}
		            	}
		            	for (int z = 0; z < dim_z; z++)
		            		System.arraycopy(tab_out[z], 0, tab[z], 0, dim_x * dim_y);
		            	 if(getUI()!=null)
		            		 getUI().setProgressBarValue((double) (time_slice*dim_c*iter+c*iter+t+1) / (double) (dim_c*dim_t*iter));
		               
		            }
		            for (int z = 0; z < dim_z; z++)
		            	Array1DUtil.doubleArrayToSafeArray(tab_out[z], icy_img[z].getDataXY(c), icy_img[z].isSignedDataType());
		            System.gc();
	        	}
	        	for (int z = 0; z < dim_z; z++)
	        		seq_out.setImage(time_slice, z, icy_img[z]);
	        	
        	      	
        }
        seq_out.updateComponentsBounds();
        seq_out.setName("Diffusion Perona-Malik 3D");
        return seq_out;

    }

    public Sequence Catte_3D(Sequence sequence, int iter, double dt, double k, double sigma)
    {
    	int temp = 0;
        double Dx1, Dx2, Dy1, Dy2, Dz1, Dz2;
        double Dx1g, Dx2g, Dy1g, Dy2g, Dz1g, Dz2g;
        double current, current_g;
     
        k = k * k;
        
        Sequence seq_out = new Sequence();       
        int dim_x = sequence.getSizeX();
        int dim_y = sequence.getSizeY();
        int dim_z = sequence.getSizeZ();
        int dim_t = sequence.getSizeT();
        int dim_c = sequence.getSizeC();
        
        double[][] tab     = new double[dim_z][];
        double[][] tab_out = new double[dim_z][dim_x*dim_y];
        
        double[] ker= Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigma).getData();
	
		double[][] tab_gauss = new double[dim_z][dim_x*dim_y];
        
        for (int time_slice = 0; time_slice < dim_t; time_slice++)
        {
        		IcyBufferedImage[] icy_img = new IcyBufferedImage[dim_z];
        		for (int z = 0; z < dim_z; z++)
        			icy_img[z] = new IcyBufferedImage(dim_x, dim_y, dim_c, sequence.getDataType());
        		
	        	for (int c = 0; c < dim_c; c++)
	        	{
	        		for (int z = 0; z < dim_z; z++)
	        		{
	        			tab[z] = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice, z, c), false);
	        			System.arraycopy(tab[z], 0, tab_gauss[z], 0, dim_x*dim_y);
	        			System.arraycopy(tab[z], 0, tab_out[z], 0, dim_x*dim_y);
	        		}
		           	         
		            for (int t = 0; t < iter; t++)
		            {
		            	try{
		            	Convolution1D.convolve(tab_gauss, dim_x,dim_y,ker, ker,ker);
		            	 } catch (Exception e) {
		         			// TODO Auto-generated catch block
		         			e.printStackTrace();
		         		} 
		            	for (int z = 0; z < dim_z ; z++)
		            	{
		            		if (z == 0 || z == dim_z - 1)
		            		{
		            			for (int x = 1; x < dim_x - 1; x++)
			                    for (int y = 1; y < dim_y - 1; y++)
			                    {
			                        temp = x + y * dim_x;
			                        current = tab[z][temp];
			                        Dx1 = tab[z][temp - 1] - current;
			                        Dx2 = tab[z][temp + 1] - current;
			                        Dy1 = tab[z][temp - dim_x] - current;
			                        Dy2 = tab[z][temp + dim_x] - current;
			                        
			                        current_g = tab_gauss[z][temp];
			                        Dx1g = tab_gauss[z][temp - 1] - current_g;
			                        Dx2g = tab_gauss[z][temp + 1] - current_g;
			                        Dy1g = tab_gauss[z][temp - dim_x] - current_g;
			                        Dy2g = tab_gauss[z][temp + dim_x] - current_g;
			                       		
			                        tab_out[z][temp] = current+ dt * (DiffusionTools.PM_function(Dx1g, k) * Dx1 + DiffusionTools.PM_function(Dx2g, k) * Dx2 + DiffusionTools.PM_function(Dy1g, k) * Dy1 + DiffusionTools.PM_function(Dy2g, k) * Dy2);
			                    }		            			
		            		}
		            		else
		            		{
				                for (int x = 1; x < dim_x - 1; x++)
			                    for (int y = 1; y < dim_y - 1; y++)
			                    {
			                        temp = x + y * dim_x;
			                        current = tab[z][temp];
			                        Dx1 = tab[z][temp - 1] - current;
			                        Dx2 = tab[z][temp + 1] - current;
			                        Dy1 = tab[z][temp - dim_x] - current;
			                        Dy2 = tab[z][temp + dim_x] - current;
			                        Dz1 = tab[z - 1][temp] - current;
			                        Dz2 = tab[z + 1][temp] - current;
			                        
			                        current_g = tab_gauss[z][temp];
			                        Dx1g = tab_gauss[z][temp - 1] - current_g;
			                        Dx2g = tab_gauss[z][temp + 1] - current_g;
			                        Dy1g = tab_gauss[z][temp - dim_x] - current_g;
			                        Dy2g = tab_gauss[z][temp + dim_x] - current_g;
			                        Dz1g = tab_gauss[z - 1][temp] - current_g;
			                        Dz2g = tab_gauss[z + 1][temp] - current_g;
			
			                        tab_out[z][temp] = current+ dt * (DiffusionTools.PM_function(Dx1g, k) * Dx1 + DiffusionTools.PM_function(Dx2g, k) * Dx2 + DiffusionTools.PM_function(Dy1g, k) * Dy1 + DiffusionTools.PM_function(Dy2g, k) * Dy2
			                        		+DiffusionTools.PM_function(Dz1g, k) * Dz1 + DiffusionTools.PM_function(Dz2g, k) * Dz2);
			                    }
		            		}
		            	}
		            	for (int z = 0; z < dim_z; z++)
		            	{
		            		System.arraycopy(tab_out[z], 0, tab[z], 0, dim_x * dim_y);
		            		System.arraycopy(tab_out[z], 0, tab_gauss[z], 0, dim_x * dim_y);
		            	}
		            	 if(getUI()!=null)
		            		 getUI().setProgressBarValue((double) (time_slice*dim_c*iter+c*iter+t+1) / (double) (dim_c*dim_t*iter));
		               
		            }
		            for (int z = 0; z < dim_z; z++)
		            	Array1DUtil.doubleArrayToSafeArray(tab_out[z], icy_img[z].getDataXY(c), icy_img[z].isSignedDataType());	
		            System.gc();
	        	}
	        	for (int z = 0; z < dim_z; z++)
	        		seq_out.setImage(time_slice, z, icy_img[z]);
        }
        seq_out.updateComponentsBounds();
        seq_out.setName("Diffusion Catte 3D");
        return seq_out;

    }
   

    public void clean()
    {
    }

	@Override
	public void declareInput(VarList inputMap) {
		// TODO Auto-generated method stub
		
		inputMap.add(input.getVariable());				
		inputMap.add(type_2D.getVariable());			   
		inputMap.add(iterations.getVariable());
		inputMap.add(step.getVariable()); 
		inputMap.add(thr.getVariable()); 			    
	    inputMap.add(catte.getVariable());
	    inputMap.add(sigmaX.getVariable()); 
	}

	@Override
	public void declareOutput(VarList outputMap) {
		// TODO Auto-generated method stub
		outputMap.add(output.getVariable());
		
	}

}
