/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.gui.curve;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import javax.swing.JPanel;

/**
 * Class displaying a curve in a frame. Control points are represented by yellow
 * dot. The current selected control point is surrounded by a dark circle.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public class CurveCanvas extends JPanel {
	private static final long serialVersionUID = 8811809436053457926L;

	private static final int BORDER_OFFSET = 3;
	private static final Color BORDER_COLOR = new Color(100, 100, 100);

	private static final int CHECKER_SIZE = 20;
	private static final Color BACKGROUND_COLOR_1 = new Color(150, 150, 150);
	private static final Color BACKGROUND_COLOR_2 = new Color(200, 200, 200);

	private static final int CONTROL_RADIUS = 3;
	private static final Color CONTROL_COLOR = new Color(230, 170, 0);
	private static final Color CONTROL_BORDER_COLOR = new Color(100, 50, 0);

	private static final Color CURVE_COLOR = new Color(240, 240, 240, 200);

	private Curve curve_ = null;
	private boolean showControls_ = false;
	private int currentIndex_ = -1;

	/** Default constructor. */
	public CurveCanvas() {
	}

	/** Default constructor. */
	public CurveCanvas(final Curve curve) {
		curve_ = curve;
	}

	/** Returns the curve displayed by the canvas. */
	public Curve getCurve() {
		return curve_;
	}

	/** Sets the curve to display. */
	public void setCurve(final Curve curve) {
		curve_ = curve;
		repaint();
	}

	/** Returns the index selected control point. */
	public int getCurrentIndex() {
		return currentIndex_;
	}

	/** Selects a control point. */
	public void setCurrentIndex(int index) {
		currentIndex_ = index;
		repaint();
	}

	/** Indicates if the control points are shown. */
	public boolean showControls() {
		return showControls_;
	}

	/** Set if the control points have to be displayed. */
	public void setShowControls(boolean show) {
		showControls_ = show;
		repaint();
	}

	/**
	 * Return the index of the control point under the given point. If ther is
	 * no control point, returns -1.
	 */
	public int indexAt(int x, int y) {
		int index = 0;
		for (Point2D pt : curve_.getControlPoints()) {
			Point dPt = mapToCanvasCoordinate(pt.getX(), pt.getY());
			int dx = x - dPt.x;
			int dy = y - dPt.y;

			if (dx * dx + dy * dy <= (CONTROL_RADIUS + 2)
					* (CONTROL_RADIUS + 2)) {
				return index;
			}

			++index;
		}

		return -1;
	}

	/** Maps the given curve coordinates to canvas coordinates. */
	public Point mapToCanvasCoordinate(double x, double y) {
		Point res = new Point();

		Rectangle2D.Double frame = getDisplayRect();
		res.x = (int) (frame.x + (Math.max(Math.min(1.0, x), 0.0) * frame.width));
		res.y = (int) (frame.y + (Math.max(Math.min(1.0, 1 - y), 0.0) * frame.height));

		return res;
	}

	/** Maps the given canvas coordinates to curve coordinates. */
	public Point2D mapToCurveCoordinate(int x, int y) {
		Point2D.Double res = new Point2D.Double();

		Rectangle2D.Double frame = getDisplayRect();
		res.x = Math.min(Math.max((x - frame.x) / frame.width, 0), 1);
		res.y = 1 - Math.min(Math.max((y - frame.y) / frame.height, 0), 1);

		return res;
	}

	@Override
	public Dimension getPreferredSize() {
		return new Dimension(140, 140);
	}

	@Override
	protected void paintComponent(Graphics g) {
		Graphics2D g2 = (Graphics2D) g;
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		Rectangle2D.Double frame = getDisplayRect();

		// Draw background
		g2.setColor(BACKGROUND_COLOR_1);
		g2.fillRect(0, 0, getWidth(), getHeight());
		g2.setColor(BORDER_COLOR);
		g2.drawRect(0, 0, getWidth() - 1, getHeight() - 1);
		Shape clipArea = g2.getClip();
		g2.clip(frame);
		g2.setColor(BACKGROUND_COLOR_1);
		g2.fillRect(0, 0, getWidth(), getHeight());
		g2.setColor(BACKGROUND_COLOR_2);
		for (int i = 0; i < getWidth(); i += 2 * CHECKER_SIZE) {
			for (int j = 0; j < getHeight(); j += 2 * CHECKER_SIZE) {
				g2.fillRect(i, j, CHECKER_SIZE, CHECKER_SIZE);
				g2.fillRect(i + CHECKER_SIZE, j + CHECKER_SIZE, CHECKER_SIZE,
						CHECKER_SIZE);
			}
		}

		// Draw Curve
		if (curve_ != null) {
			int X[] = new int[(int) frame.width + 2];
			int Y[] = new int[X.length];
			X[0] = (int) (frame.x);
			Y[0] = (int) (frame.y + frame.height);
			X[X.length - 1] = (int) (frame.x + frame.width);
			Y[Y.length - 1] = (int) (frame.y + frame.height);

			for (int i = 0; i < X.length - 2; ++i) {
				double x = (i) / (frame.width - 1);
				double y = curve_.valueAt(x);

				Point pt = mapToCanvasCoordinate(x, y);

				X[i + 1] = pt.x;
				Y[i + 1] = pt.y;
			}

			g2.setColor(CURVE_COLOR);
			g2.fillPolygon(X, Y, X.length);

			g2.setClip(clipArea);
			g2.setColor(BORDER_COLOR);
			g2.draw(frame);

			// Draw control points
			if (showControls_) {
				int index = 0;
				for (Point2D pt : curve_.getControlPoints()) {
					paintControl(g2, pt, index == currentIndex_);
					++index;
				}
			}
		}
	}

	/** Returns the rectangle inside the frame of the canvas. */
	private Rectangle2D.Double getDisplayRect() {
		int x0 = CONTROL_RADIUS + BORDER_OFFSET - 1;
		int y0 = x0;
		int width = getWidth() - 2 * (CONTROL_RADIUS + BORDER_OFFSET);
		int height = getHeight() - 2 * (CONTROL_RADIUS + BORDER_OFFSET);
		return new Rectangle2D.Double(x0, y0, width, height);
	}

	/** Paints the given control point. */
	private void paintControl(Graphics2D g, final Point2D pt, boolean isSelected) {
		Point dPt = mapToCanvasCoordinate(pt.getX(), pt.getY());

		g.setColor(CONTROL_COLOR);
		g.fillOval(dPt.x - CONTROL_RADIUS, dPt.y - CONTROL_RADIUS,
				2 * CONTROL_RADIUS + 1, 2 * CONTROL_RADIUS + 1);
		g.setColor(CONTROL_BORDER_COLOR);
		g.drawOval(dPt.x - CONTROL_RADIUS, dPt.y - CONTROL_RADIUS,
				2 * CONTROL_RADIUS + 1, 2 * CONTROL_RADIUS + 1);

		if (isSelected) {
			g.setColor(new Color(100, 100, 100));
			g.drawOval(dPt.x - 20, dPt.y - 20, 41, 41);
		}
	}
}
