package plugins.vannary.morphomaths;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.type.collection.array.Array2DUtil;

/**
 * Pseudo-Euclidean distance transform based on a two-pass 3x3 chamfer distance
 * 
 * @author Alexandre Dufour
 * 
 */
public class Chamfer3
{
	/**
	 * direct neighbor distance
	 */
	public final double	a	= 1;
	
	/**
	 * 2D diagonal neighbor distance
	 */
	public final double	b	= Math.sqrt(2);
	
	/**
	 * 3D diagonal neighbor distance
	 */
	public final double	c	= Math.sqrt(3);
	
	public final double	d	= Math.sqrt(5);
	
	/**
	 * Computes the unsigned distance map to an object
	 * 
	 * @param input
	 *            the input sequence
	 * @param threshold
	 *            The object is defined by values strictly greater than this value
	 * @param interiorMap
	 *            set to true (resp. false) to compute the distance map inside (resp. outside) the
	 *            object
	 */
	public void unsignedChamferDistanceMap(Sequence input, double threshold, boolean interiorMap)
	{
		int width = input.getSizeX();
		int height = input.getSizeY();
		int depth = input.getSizeZ();
		int time = input.getSizeT();
		int channels = input.getSizeC();
		boolean signed = input.getDataType_().isSigned();
		
		double[][] map = new double[depth][width * height];
		
		int maxDistance = width * height * depth;
		int foregroundInit = interiorMap ? maxDistance : 0;
		int backgroundInit = interiorMap ? 0 : maxDistance;
		
		for (int t = 0; t < time; t++)
		{
			for (int c = 0; c < channels; c++)
			{
				Array2DUtil.arrayToDoubleArray(input.getDataXYZ(t, c), map, signed);
				
				// Initialize the distance map according to the threshold
				for (int z = 0; z < depth; z++)
				{
					double[] slice = map[z];
					
					for (int i = 0; i < slice.length; i++)
						slice[i] = (slice[i] > threshold ? foregroundInit : backgroundInit);
				}
				
				if (depth == 1)
				{
					updateUnsignedChamferDistance2D(width, height, map[0]);
				}
				else
				{
					updateUnsignedChamferDistance3D(width, height, depth, map);
				}
				
				Array2DUtil.doubleArrayToArray(map, input.getDataXYZ(t, c));
			}
		}
	}
	
	/**
	 * Updates an already-initialized 2D unsigned distance map (background value should be infinity)
	 */
	public void updateUnsignedChamferDistance2D(final int width, final int height, final double[] map)
	{
		int i, j, index = 0;
		double aDist, bDist;
		
		// First pass
		
		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++, index++)
			{
				aDist = map[index] + a;
				
				if (i < width - 1 && map[index + 1] > aDist) map[index + 1] = aDist;
				
				if (j < height - 1)
				{
					bDist = aDist - a + b;
					
					if (map[index + width] > aDist) map[index + width] = aDist;
					
					if (i < width - 1 && map[index + width + 1] > bDist) map[index + width + 1] = bDist;
					
					if (i > 0 && map[index + width - 1] > bDist) map[index + width - 1] = bDist;
				}
			}
		
		index--;
		
		// Second pass
		
		for (j = height - 1; j >= 0; j--)
			for (i = width - 1; i >= 0; i--, index--)
			{
				aDist = map[index] + a;
				
				if (i > 0 && map[index - 1] > aDist) map[index - 1] = aDist;
				
				if (j > 0)
				{
					bDist = aDist - a + b;
					
					if (map[index - width] > aDist) map[index - width] = aDist;
					
					if (i > 0 && map[index - width - 1] > bDist) map[index - width - 1] = bDist;
					
					if (i < width - 1 && map[index - width + 1] > bDist) map[index - width + 1] = bDist;
				}
			}
	}
	
	/**
	 * Updates an already-initialized 3D unsigned distance map (background value should be infinity)
	 */
	public void updateUnsignedChamferDistance3D(final int width, final int height, final int depth, final double[][] map)
	{
		int i, j, k, offset = 0;
		double aDist, bDist, cDist, val;
		
		// First pass
		
		for (k = 0; k < depth - 1; k++)
		{
			double[] thisSlice = map[k];
			double[] nextSlice = (k < depth - 1) ? map[k + 1] : null;
			
			offset = 0;
			
			for (j = 0; j < height; j++)
				for (i = 0; i < width; i++, offset++)
				{
					val = thisSlice[offset];
					aDist = val + a;
					bDist = val + b;
					cDist = val + c;
					
					if (k < depth - 1)
					{
						if (nextSlice[offset] > aDist) nextSlice[offset] = aDist;
						
						if (j < height - 1)
						{
							if (nextSlice[offset + width] > bDist) nextSlice[offset + width] = bDist;
							
							if (i > 0 && nextSlice[offset + width - 1] > cDist) nextSlice[offset + width - 1] = cDist;
							
							if (i < width - 1 && nextSlice[offset + width + 1] > cDist) nextSlice[offset + width + 1] = cDist;
						}
						
						if (j > 0)
						{
							if (nextSlice[offset - width] > bDist) nextSlice[offset - width] = bDist;
							
							if (i > 0 && nextSlice[offset - width - 1] > cDist) nextSlice[offset - width - 1] = cDist;
							
							if (i < width - 1 && nextSlice[offset - width + 1] > cDist) nextSlice[offset - width + 1] = cDist;
						}
						
						if (i > 0 && nextSlice[offset - 1] > bDist) nextSlice[offset - 1] = bDist;
						
						if (i < width - 1 && nextSlice[offset + 1] > bDist) nextSlice[offset + 1] = bDist;
					}
					
					if (j < height - 1)
					{
						if (thisSlice[offset + width] > aDist) thisSlice[offset + width] = aDist;
						
						if (i < width - 1 && thisSlice[offset + width + 1] > bDist) thisSlice[offset + width + 1] = bDist;
						
						if (i > 0 && thisSlice[offset + width - 1] > bDist) thisSlice[offset + width - 1] = bDist;
					}
					
					if (i < width - 1 && thisSlice[offset + 1] > aDist) thisSlice[offset + 1] = aDist;
				}
		}
		
		// Second pass
		
		for (k = depth - 1; k > 0; k--)
		{
			double[] thisSlice = map[k];
			double[] prevSlice = (k > 0) ? map[k - 1] : null;
			
			offset = thisSlice.length - 1;
			
			for (j = height - 1; j >= 0; j--)
				for (i = width - 1; i >= 0; i--, offset--)
				{
					val = thisSlice[offset];
					aDist = val + a;
					bDist = val + b;
					
					if (k > 0)
					{
						cDist = val + c;
						
						if (prevSlice[offset] > aDist) prevSlice[offset] = aDist;
						
						if (j > 0)
						{
							if (prevSlice[offset - width] > bDist) prevSlice[offset - width] = bDist;
							
							if (i > 0 && prevSlice[offset - width - 1] > cDist) prevSlice[offset - width - 1] = cDist;
							
							if (i < width - 1 && prevSlice[offset - width + 1] > cDist) prevSlice[offset - width + 1] = cDist;
						}
						
						if (j < height - 1)
						{
							if (prevSlice[offset + width] > bDist) prevSlice[offset + width] = bDist;
							
							if (i > 0 && prevSlice[offset + width - 1] > cDist) prevSlice[offset + width - 1] = cDist;
							
							if (i < width - 1 && prevSlice[offset + width + 1] > cDist) prevSlice[offset + width + 1] = cDist;
						}
						
						if (i > 0 && prevSlice[offset - 1] > bDist) prevSlice[offset - 1] = bDist;
						
						if (i < width - 1 && prevSlice[offset + 1] > bDist) prevSlice[offset + 1] = bDist;
					}
					
					if (j > 0)
					{
						if (thisSlice[offset - width] > aDist) thisSlice[offset - width] = aDist;
						
						if (i > 0 && thisSlice[offset - width - 1] > bDist) thisSlice[offset - width - 1] = bDist;
						
						if (i < width - 1 && thisSlice[offset - width + 1] > bDist) thisSlice[offset - width + 1] = bDist;
					}
					
					if (i > 0 && thisSlice[offset - 1] > aDist) thisSlice[offset - 1] = aDist;
				}
		}
	}
	
}
