/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.snake3D;

import icy.util.XMLUtil;

import java.awt.Color;
import java.awt.Polygon;
import java.awt.geom.Point2D;
import java.util.ArrayList;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * This class is used to store the scales that are used to draw the skin of the
 * snake.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 */
public class Snake3DScale {

	/** This is the color with which this scale will be drawn. */
	private Color color_ = Color.RED;

	/**
	 * This flag is used to determine how to draw the outline of this scale. If
	 * it is set to <code>true</code>, then the first and the last point of the
	 * polygon are joined. Otherwise, if it is set to <code>false</code>, then
	 * the first and the last point of the polygon are not joined.
	 */
	private boolean closed_ = true;

	/**
	 * Coordinates of the polylines. The first index corresponds to the point,
	 * and the second to the coordinate index (x, y or z).
	 */
	private double[][] points_ = null;

	/** XML tag for the number of points. */
	private static final String ID_NUM_POINTS = "num_points";

	/** XML tag for each point element. */
	private static final String ID_POINT = "point";

	/** XML tag for the x coordinate of each point. */
	private static final String ID_POINT_XPOS = "x_position";

	/** XML tag for the y coordinate of each point. */
	private static final String ID_POINT_YPOS = "y_position";

	/** XML tag for the z coordinate of each point. */
	private static final String ID_POINT_ZPOS = "z_position";

	/** XML tag for the color of the scale. */
	private static final String ID_COLOR = "color";

	/** XML tag to inform if the scale is closed or not. */
	private static final String ID_CLOSED = "closed";

	// ============================================================================
	// PUBLIC METHODS

	/**
	 * This constructor builds a scale of length <code>length</code>. The
	 * default color <code>color</code>is set to <code>Color.RED</code>. The
	 * scale is closed by default.
	 */
	public Snake3DScale(int nPoints) {
		points_ = new double[nPoints][3];
	}

	// ----------------------------------------------------------------------------

	/** This constructor builds a scale with the provided colors and flags. */
	public Snake3DScale(int nPoints, Color color, boolean closed) {
		points_ = new double[nPoints][3];
		color_ = color;
		closed_ = closed;
	}

	// ----------------------------------------------------------------------------

	/** This constructor builds a scale with the provided colors and flags. */
	public Snake3DScale(double[][] coordinates, Color color, boolean closed) {
		points_ = coordinates;
		color_ = color;
		closed_ = closed;
	}

	// ----------------------------------------------------------------------------

	/** This method returns text-based information about this object. */
	@Override
	public String toString() {
		if (points_ != null) {
			return "[number of scales: " + points_.length + ", color: "
					+ color_ + ", closed: " + closed_ + "]";
		} else {
			return "[number of scales: 0, color: " + color_ + ", closed: "
					+ closed_ + "]";
		}
	}

	// ----------------------------------------------------------------------------

	/** Saves the scale in an XML node. */
	public void saveToXML(Node node) throws Exception {
		XMLUtil.setElementIntValue(node, ID_COLOR, color_.getRGB());
		XMLUtil.setElementBooleanValue(node, ID_CLOSED, closed_);
		if (points_ != null) {
			XMLUtil.setElementIntValue(node, ID_NUM_POINTS, points_.length);
			for (double[] element : points_) {
				Element e = XMLUtil.addElement(node, ID_POINT);
				XMLUtil.setElementDoubleValue(e, ID_POINT_XPOS, element[0]);
				XMLUtil.setElementDoubleValue(e, ID_POINT_YPOS, element[1]);
				XMLUtil.setElementDoubleValue(e, ID_POINT_ZPOS, element[2]);
			}
		} else {
			throw new Exception("Scales are invalid.");
		}
	}

	// ----------------------------------------------------------------------------

	/** Loads the scale from an XML node. */
	public void loadFromXML(Node node) {
		color_ = new Color(XMLUtil.getElementIntValue(node, ID_COLOR,
				color_.getRGB()));
		closed_ = XMLUtil.getElementBooleanValue(node, ID_CLOSED, closed_);
		int numPoints = XMLUtil.getElementIntValue(node, ID_NUM_POINTS, 0);
		if (numPoints > 0) {
			ArrayList<Element> nodesPoint = XMLUtil.getElements(node, ID_POINT);
			if (nodesPoint.size() != numPoints) {
				System.err
						.print("Number of points does not match while loading Snake3DScale object from xml file.");
				return;
			}
			points_ = new double[numPoints][3];
			for (int i = 0; i < points_.length; i++) {
				points_[i] = new double[3];
				points_[i][0] = XMLUtil.getElementIntValue(nodesPoint.get(i),
						ID_POINT_XPOS, 0);
				points_[i][1] = XMLUtil.getElementIntValue(nodesPoint.get(i),
						ID_POINT_YPOS, 0);
				points_[i][2] = XMLUtil.getElementIntValue(nodesPoint.get(i),
						ID_POINT_ZPOS, 0);
			}
		}
	}

	// ----------------------------------------------------------------------------

	/** Returns the associated color of the scale. */
	public Color getColor() {
		return color_;
	}

	// ----------------------------------------------------------------------------

	/** Returns if the scale is closed or not. */
	public boolean isClosed() {
		return closed_;
	}

	// ----------------------------------------------------------------------------

	/** Returns the coordinates of the scales. */
	public double[][] getCoordinates() {
		return points_;
	}

	// ----------------------------------------------------------------------------

	public boolean containsInXYProjection(Point2D point) {
		Polygon polygon = new Polygon();
		for (double[] element : points_) {
			polygon.addPoint((int) Math.round(element[0]),
					(int) Math.round(element[1]));
		}
		return polygon.contains(point);
	}
}
