/*
 * Copyright 2010-2015 Institut Pasteur.
 * 
 * This file is part of Icy.
 * 
 * Icy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Icy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Icy. If not, see <http://www.gnu.org/licenses/>.
 */
package icy.gui.component;

import java.awt.Color;
import java.util.ArrayList;
import java.util.List;

/**
 * Component to display and modify a numeric (double) value
 * 
 * @author Yoann Le Montagner & Stephane
 */
public class NumberTextField extends IcyTextField
{
    /**
     * 
     */
    private static final long serialVersionUID = 3043750422009529874L;

    /**
     * Listener interface
     */
    public static interface ValueChangeListener
    {
        /**
         * Method triggered when the numeric value in the component changes
         */
        public void valueChanged(double newValue, boolean validate);
    }

    protected double value;
    protected boolean integer;
    protected List<ValueChangeListener> listeners;

    /**
     * Constructor
     */
    public NumberTextField(boolean integer)
    {
        super();

        value = 0;
        this.integer = integer;
        listeners = new ArrayList<ValueChangeListener>();
    }

    /**
     * Constructor
     */
    public NumberTextField()
    {
        this(false);
    }

    /**
     * Return true if the range use integer number
     */
    public boolean isInteger()
    {
        return integer;
    }

    /**
     * Return true if the range use integer number
     */
    public void setInteger(boolean integer)
    {
        this.integer = integer;

        // force value adjustment
        setNumericValue(value);
    }

    /**
     * Add a new listener
     */
    public void addValueListener(ValueChangeListener l)
    {
        listeners.add(l);
    }

    /**
     * Remove a listener
     */
    public void removeValueListener(ValueChangeListener l)
    {
        listeners.remove(l);
    }

    /**
     * Retrieve the value as a double
     */
    public double getNumericValue()
    {
        return value;
    }

    /**
     * Set the value
     */
    public void setNumericValue(double value)
    {
        if (integer)
            setText(Integer.toString((int) value));
        else
            setText(Double.toString(value));
    }

    protected void valueChanged(boolean validate)
    {
        fireValueChangedEvent(validate);
    }

    @Override
    protected void textChanged(boolean validate)
    {
        super.textChanged(validate);

        double oldValue = value;

        try
        {
            final String text = getText();
            value = text.isEmpty() ? 0.0 : Double.parseDouble(text);
            // force integer
            if (integer)
                value = (int) value;
            setForeground(Color.BLACK);
        }
        catch (NumberFormatException err)
        {
            setForeground(Color.RED);
        }

        if (validate)
            valueChanged(validate);
        else if (value != oldValue)
            valueChanged(false);
    }

    /**
     * Fire the value changed event
     */
    private void fireValueChangedEvent(boolean validate)
    {
        for (ValueChangeListener l : listeners)
            l.valueChanged(value, validate);
    }
}
