/*
 * Copyright 2010, 2011 Institut Pasteur.
 * Copyright 2012 Institut National de l'Audiovisuel.
 * 
 * This file is part of NHerveTools.
 * 
 * NHerveTools is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * NHerveTools is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with NHerveTools. If not, see <http://www.gnu.org/licenses/>.
 */
package plugins.nherve.toolbox.image.feature.signature;

import java.util.Iterator;

import plugins.nherve.toolbox.image.feature.SignatureDistance;

/**
 * The Class L1Distance.
 * 
 * @author Nicolas HERVE - nherve@ina.fr
 */
public class L1Distance extends SignatureDistance<VectorSignature> {

	/**
	 * Instantiates a new l1 distance.
	 */
	public L1Distance() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * plugins.nherve.toolbox.image.feature.SignatureDistance#computeDistance
	 * (plugins.nherve.toolbox.image.feature.Signature,
	 * plugins.nherve.toolbox.image.feature.Signature)
	 */
	@Override
	public double computeDistance(VectorSignature vs1, VectorSignature vs2) throws SignatureException {
		if ((vs1 == null) || (vs2 == null)) {
			throw new SignatureException("Null signature in L1Distance.compute()");
		}

		if (vs1.getSize() != vs2.getSize()) {
			throw new SignatureException("VectorSignature dimensions mismatch");
		}

		double sum = 0.0;

		if (vs1 instanceof SparseVectorSignature && vs2 instanceof SparseVectorSignature) {
			SparseVectorSignature s1 = (SparseVectorSignature) vs1;
			SparseVectorSignature s2 = (SparseVectorSignature) vs2;

			Iterator<Integer> idx1it = s1.iterator();
			Iterator<Integer> idx2it = s2.iterator();

			int idx1 = Integer.MAX_VALUE;
			int idx2 = Integer.MAX_VALUE;

			if (idx1it.hasNext()) {
				idx1 = idx1it.next();
			}

			if (idx2it.hasNext()) {
				idx2 = idx2it.next();
			}

			while ((idx1 < Integer.MAX_VALUE) || (idx2 < Integer.MAX_VALUE)) {
				if (idx1 == idx2) {
					sum += Math.abs(s1.get(idx1) - s2.get(idx2));
					if (idx1it.hasNext()) {
						idx1 = idx1it.next();
					} else {
						idx1 = Integer.MAX_VALUE;
					}
					if (idx2it.hasNext()) {
						idx2 = idx2it.next();
					} else {
						idx2 = Integer.MAX_VALUE;
					}
				} else if (idx1 > idx2) {
					sum += Math.abs(s2.get(idx2));
					if (idx2it.hasNext()) {
						idx2 = idx2it.next();
					} else {
						idx2 = Integer.MAX_VALUE;
					}
				} else {
					sum += Math.abs(s1.get(idx1));
					if (idx1it.hasNext()) {
						idx1 = idx1it.next();
					} else {
						idx1 = Integer.MAX_VALUE;
					}
				}
			}
		} else if (vs1 instanceof DenseVectorSignature && vs2 instanceof DenseVectorSignature) {
			double[] s1 = ((DenseVectorSignature)vs1).getData();
			double[] s2 = ((DenseVectorSignature)vs2).getData();
			for (int dim = 0; dim < s1.length; dim++) {
				sum += Math.abs(s1[dim] - s2[dim]);
			}
		} else {
			for (int dim = 0; dim < vs1.getSize(); dim++) {
				double e = vs1.get(dim) - vs2.get(dim);
				sum += Math.abs(e);
			}
		}

		return sum;
	}

}
