/*
 * Copyright 2010, 2011 Institut Pasteur.
 * 
 * This file is part of ICY.
 * 
 * ICY is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * ICY is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ICY. If not, see <http://www.gnu.org/licenses/>.
 */
package plugins.fab.trackmanager;

import icy.canvas.Canvas2D;
import icy.canvas.Canvas3D;
import icy.canvas.IcyCanvas;
import icy.painter.Painter;
import icy.roi.ROI2D;
import icy.sequence.Sequence;
import icy.swimmingPool.SwimmingObject;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.util.ArrayList;

import plugins.kernel.canvas.VtkCanvas;
import plugins.nchenouard.spot.Detection;
import vtk.vtkActor;
import vtk.vtkCardinalSpline;
import vtk.vtkCellArray;
import vtk.vtkGlyph3D;
import vtk.vtkMath;
import vtk.vtkPoints;
import vtk.vtkPolyData;
import vtk.vtkPolyDataMapper;
import vtk.vtkSphereSource;
import vtk.vtkTubeFilter;

/** 
 * 
 * Painter used to display tracks on sequence (as overlay)
 * 
 * @author Fabrice de Chaumont
 * 
 *  */
public class TrackManagerPainter implements Painter
{

    private TrackPool trackPool = null;

    public TrackManagerPainter(TrackPool trackPool)
    {
        this.trackPool = trackPool;
    }

    private boolean drawTracksOnSequence = true;
    private boolean subPixelicDisplay = true;

    public boolean isSubPixelicDisplay()
    {
        return subPixelicDisplay;
    }

    public void setSubPixelicDisplay(boolean subPixelicDisplay)
    {
        this.subPixelicDisplay = subPixelicDisplay;
    }

    public void keyPressed(KeyEvent e, Point2D imagePoint, IcyCanvas icyCanvas)
    {
    }

    public void mouseClick(MouseEvent e, Point2D p, IcyCanvas icyCanvas)
    {

    	float distanceThreshold = (float) ROI2D.canvasToImageLogDeltaX( icyCanvas, 3 );
        // Check if a detection is around.
        for (TrackGroup tg : trackPool.getTrackGroupList())
            for (TrackSegment ts : tg.getTrackSegmentList())
            {
                for (Detection d : ts.getDetectionList())
                    if (d.isEnabled())
                    {
                        double dist = new Point2D.Double(p.getX(), p.getY()).distance(new Point2D.Double(d.getX(), d
                                .getY()));

                        if (dist < distanceThreshold )
                        {
//                            if (!EventUtil.isShiftDown(e))
                            {                            
                                TrackSegment ts1 = trackPool.getTrackSegmentWithDetection(d);
                                if ( ts1 != null )
                                {
                                	ts1.setAllDetectionSelected(!ts1.isAllDetectionSelected());
                                	trackPool.getDisplaySequence().painterChanged( null );
                                	e.consume();
                                	fireTrackPainterEvent(); // FIXME : a remettre ?
                                }
                                break;
                            }
//                            else
//                            {
//                                d.setSelected(!d.isSelected());
//                                trackPool.getDisplaySequence().painterChanged( null );
//                                e.consume();
//                                fireTrackPainterEvent(); // FIXME : a remettre ?
//                                break;
//                            }
                        }
                    }
            }

    }

    public void mouseDrag(MouseEvent e, Point2D p, IcyCanvas icyCanvas)
    {
    }

    public void mouseMove(MouseEvent e, Point2D p, IcyCanvas icyCanvas)
    {
    }
    
    // should switch to an hashmap to manage several 3D viewers.    
    ArrayList<vtkActor> actorList = new ArrayList<vtkActor>();

    private void draw3D(Sequence sequence, VtkCanvas canvas3D)
    {

    	canvas3D.getRenderer().SetGlobalWarningDisplay(0);

    	// remove existing objects

    	for ( vtkActor actor : actorList )
    	{
    		canvas3D.getRenderer().RemoveActor( actor );    			
    	}

    	// display new tracks

    	for (SwimmingObject result : trackPool.resultList)
    	{
    		final TrackGroup tg = (TrackGroup) result.getObject();
    		{
    			for (TrackSegment ts : tg.getTrackSegmentList())
    				//    				TrackSegment ts = tg.getTrackSegmentList().get( 0 );
    			{
    				//if ( ts.getDetectionList().size() < 3 ) continue;
    				{
    					//System.out.println("should pass here uin 3D");
    					// This will be used later to get random numbers.
    					vtkMath math = new vtkMath();

    					// Total number of points.
    					//    						int numberOfInputPoints = 20;

    					// One spline for each direction.
    					vtkCardinalSpline aSplineX = new vtkCardinalSpline();
    					vtkCardinalSpline aSplineY = new vtkCardinalSpline();
    					vtkCardinalSpline aSplineZ = new vtkCardinalSpline();

    					vtkPoints inputPoints = new vtkPoints();
    					Color color = null;
    					boolean somethingToDisplay=false;
    					for (int i = 0; i < ts.getDetectionList().size() - 1; i++)
    					{
    						if (ts.getDetectionList().get(i).isEnabled()
    								&&
    								ts.getDetectionList().get(i + 1).isEnabled())
    						{
    							//g.setStroke(new BasicStroke(trackWidth * 3));
    							//g.setColor(Color.white);
    							if ( color==null)
    							{
    								color = ts.getDetectionList().get( i ).getColor();
    							}

    							double x = ts.getDetectionList().get(i).getX();
    							double y = ts.getDetectionList().get(i).getY();
    							double z = ts.getDetectionList().get(i).getZ() * canvas3D.getZScaling();

    							aSplineX.AddPoint(i, x );
    							aSplineY.AddPoint(i, y );
    							aSplineZ.AddPoint(i, z );
    							inputPoints.InsertPoint(i, x, y, z);
    							somethingToDisplay=true;

    						}
    					}

    					if ( ! somethingToDisplay ) continue; // nothing to display

    					// The following section will create glyphs for the pivot points
    					// in order to make the effect of the spline more clear.

    					// Create a polydata to be glyphed.
    					vtkPolyData inputData = new vtkPolyData();
    					inputData.SetPoints(inputPoints);

    					//    						System.out.println("nb point: " + inputPoints.GetNumberOfPoints() );

    					// Use sphere as glyph source.
    					vtkSphereSource balls = new vtkSphereSource();
    					balls.SetRadius(0.2);
    					balls.SetPhiResolution(5);
    					balls.SetThetaResolution(5);    						
    					vtkGlyph3D glyphPoints = new vtkGlyph3D();
    					
    					//glyphPoints.SetInputConnection( balls.GetOutputPort() );
    					/*
    					 * l faut en fait remplacer dans certains cas setInputData(xxx.getOutput()) par setInputConnection(xxx.getOutputPort())
    					 */
    					glyphPoints.SetInputData(inputData);    					
    					glyphPoints.SetSourceData(balls.GetOutput());
    					//glyphPoints.SetInputConnection( balls.GetOutputPort() );
    					

    					vtkPolyDataMapper glyphMapper = new vtkPolyDataMapper();
    					glyphMapper.SetInputData(glyphPoints.GetOutput());

    					vtkActor glyph = new vtkActor();
    					glyph.SetMapper(glyphMapper);

    					glyph.GetProperty().SetDiffuseColor( color.getRed()/255d, color.getGreen()/255d, color.getBlue()/255d );
    					glyph.GetProperty().SetSpecular(.3);
    					glyph.GetProperty().SetSpecularPower(30);

    					// Generate the polyline for the spline.
    					vtkPoints points = new vtkPoints();
    					vtkPolyData profileData = new vtkPolyData();

    					// Number of points on the spline
    					int nbSpots = inputPoints.GetNumberOfPoints() ;
    					int numberOfOutputPointsOverTheSpline = nbSpots * 10;

    					// Interpolate x, y and z by using the three spline filters and
    					// create new points

    					//    						System.out.println("number of point over spline: " + numberOfOutputPointsOverTheSpline );

    					for (int i = 0; i < numberOfOutputPointsOverTheSpline; i++)
    					{
    						double t = ( nbSpots - 1.0) / (numberOfOutputPointsOverTheSpline - 1.0) * i;
    						points.InsertPoint(i, aSplineX.Evaluate(t), aSplineY.Evaluate(t), aSplineZ.Evaluate(t));
    					}

    					// Create the polyline.
    					vtkCellArray lines = new vtkCellArray();
    					lines.InsertNextCell( numberOfOutputPointsOverTheSpline );
    					//lines.InsertNextCell( 123456 );
    					for (int i = 0; i < numberOfOutputPointsOverTheSpline ; i++)
    					{
    						lines.InsertCellPoint(i);
    					}

    					profileData.SetPoints(points);
    					profileData.SetLines(lines);

    					// Add thickness to the resulting line.
    					vtkTubeFilter profileTubes = new vtkTubeFilter();
    					profileTubes.SetNumberOfSides(8);
    					profileTubes.SetInputData(profileData);
    					// profileTubes.SetRadius(.01);
    					profileTubes.SetRadius(0.1);

    					vtkPolyDataMapper profileMapper = new vtkPolyDataMapper();
    					profileMapper.SetInputData(profileTubes.GetOutput());

    					vtkActor profile = new vtkActor();
    					profile.SetMapper(profileMapper);

    					// FIXME: support only 1 color for a full track.
    					profile.GetProperty().SetOpacity(1);
    					//    					profile.GetProperty().SetDiffuseColor(1, 0, 1);
    					profile.GetProperty().SetDiffuseColor(
    							color.darker().getRed()/255d,
    							color.darker().getGreen()/255d,
    							color.darker().getBlue()/255d
    							);

    					profile.GetProperty().SetSpecular(.3);
    					profile.GetProperty().SetSpecularPower(30);

    					// Add the actors
    					canvas3D.getRenderer().AddActor(glyph);
    					canvas3D.getRenderer().AddActor(profile);

    					actorList.add( glyph );
    					actorList.add( profile );
    				}


    			}
    		}
    	}

    }

    
    
    @Override
    public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas)
    {
    	
    	//System.out.println("TP");
    	
    	if ( trackPool != null )
    		if ( trackPool.getTrackManager() != null )
    			if ( canvas!=null )
    				if ( trackPool.getTrackManager().currentT != canvas.getPositionT() )
    				{    		
    					trackPool.getTrackManager().timeCursorChanged( canvas.getPositionT() );
    					sequence.painterChanged( this );

    					//    		System.out.println("Paint synchro: t " + canvas.getPositionT() );

    					//    		return;
    				}

    	//System.out.println("TP: now painting frame " + trackPool.getTrackManager().currentT );
    	
        if (isDrawTracksOnSequence() == false)
            return;
        
        if (canvas instanceof VtkCanvas )
        {
            draw3D(sequence, (VtkCanvas) canvas);
        }

        if (canvas instanceof Canvas2D)
        {
        	float trackWidth = (float) ROI2D.canvasToImageLogDeltaX(canvas, 2 );

        	g= (Graphics2D) g.create();
        	
            // emphasis all detections.
            g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

            g.setColor(Color.RED);
            
            // shift 0.5 to display track in the center of pixel
            g.translate( 0.5d , 0.5d );
            
            g.setStroke(new BasicStroke(trackWidth));
            
            {
                // display background of tracks if selected;

                for (SwimmingObject result : trackPool.resultList)
                {
                    final TrackGroup tg = (TrackGroup) result.getObject();
                    {
                        for (TrackSegment ts : tg.getTrackSegmentList())
                        {
                            for (int i = 0; i < ts.getDetectionList().size() - 1; i++)
                            {
                                if (ts.getDetectionList().get(i).isSelected())
                                    if (ts.getDetectionList().get(i + 1).isSelected())
                                    {
                                        g.setStroke(new BasicStroke(trackWidth * 3));
                                        g.setColor(Color.white);

                                        Line2D line = new Line2D.Double(ts.getDetectionList().get(i).getX(), ts
                                                .getDetectionList().get(i).getY(), ts.getDetectionList().get(i + 1)
                                                .getX(), ts.getDetectionList().get(i + 1).getY());
                                        g.draw(line);

                                    }
                            }
                        }
                    }
                }

                // display tracks.
                g.setStroke(new BasicStroke(trackWidth));

                for (SwimmingObject result : trackPool.resultList)
                {
                    final TrackGroup tg = (TrackGroup) result.getObject();

                    {

                        for (TrackSegment ts : tg.getTrackSegmentList())
                        {
                            for (int i = 0; i < ts.getDetectionList().size() - 1; i++)
                                if (ts.getDetectionList().get(i).isEnabled())
                                    if (ts.getDetectionList().get(i + 1).isEnabled())
                                    {
                                        g.setColor(ts.getDetectionList().get(i).getColor());

                                        Line2D line = new Line2D.Double(ts.getDetectionList().get(i).getX(), ts
                                                .getDetectionList().get(i).getY(), ts.getDetectionList().get(i + 1)
                                                .getX(), ts.getDetectionList().get(i + 1).getY());
                                        g.draw(line);
                                    }

                        }
                    }
                }
                // display link betweek tracks
                g.setColor(Color.green);

                for (SwimmingObject result : trackPool.resultList)
                {
                    final TrackGroup tg = (TrackGroup) result.getObject();

                    {

                        for (TrackSegment tsPrevious : tg.getTrackSegmentList())
                        {
                            if (tsPrevious.getLastDetection().isEnabled())
                                for (TrackSegment tsNext : tsPrevious.nextList)
                                {
                                	if (tsNext.getFirstDetection().isEnabled())
                                	{
                                		Line2D line = new Line2D.Double(tsPrevious.getLastDetection().getX(),
                                				tsPrevious.getLastDetection().getY(),
                                				tsNext.getFirstDetection().getX(), tsNext.getFirstDetection().getY());
                                		g.draw(line);
                                	}
                                }
                        }
                    }
                }

                Graphics2D g2= (Graphics2D) g.create();
                
                // display detections.
                for (SwimmingObject result : trackPool.resultList)
                {
                    final TrackGroup tg = (TrackGroup) result.getObject();

                    {
                        // for ( TrackGroup tg : trackPool.getTrackGroupList() )
                        for (TrackSegment ts : tg.getTrackSegmentList())
                        {

                        	for (Detection detection : ts.getDetectionList())
                                if (detection.isEnabled())
                                {
                                    g.setColor(Color.white);
                                    Ellipse2D ellipse = new Ellipse2D.Double(detection.getX() - trackWidth / 2f,
                                            detection.getY() - trackWidth / 2f, trackWidth, trackWidth);
                                    g.fill(ellipse);
                                    
                                	// call the detection painter.
                                                                        
                                	detection.paint( g2 , sequence, canvas );
                                	
                                }

                        }
                    }
                }
            }
        }
    }

    /** List of Listeners */
    ArrayList<TrackPainterListener> trackPainterListener = new ArrayList<TrackPainterListener>();

    /** add a listener */
    public void addTrackPainterListener(TrackPainterListener trackPainterListener)
    {
        this.trackPainterListener.add(trackPainterListener);
    }

    /** remove a listener */
    public void removeTrackPainterListener(TrackPainterListener trackPainterListener)
    {
        this.trackPainterListener.remove(trackPainterListener);
    }

    /** fire event with a given TrackPainterChangeEvent */
    private void fireTrackPainterEvent(TrackPainterChangeEvent e)
    {
        for (TrackPainterListener tpl : trackPainterListener)
        {
            tpl.trackPainterChanged(e);
        }
    }

    /** fire event with a non-specific TrackPainterChangeEvent */
    private void fireTrackPainterEvent()
    {
        for (TrackPainterListener tpl : trackPainterListener)
        {
            tpl.trackPainterChanged(new TrackPainterChangeEvent(this));
        }
    }

    public boolean isDrawTracksOnSequence()
    {
        return drawTracksOnSequence;
    }

    public void setDrawTracksOnSequence(boolean drawTracksOnSequence)
    {
        this.drawTracksOnSequence = drawTracksOnSequence;
    }

//	@Override
//    public final void sequenceChanged(SequenceEvent event)
//    {
//        switch (event.getSourceType())
//        {
//            case SEQUENCE_META:
//                String metadataName = (String) event.getSource();
//
//                if ( StringUtil.equals(metadataName, Sequence.ID_PIXEL_SIZE_Z ))
//                {
////                	adjustZScaling( getSequence().getPixelSizeZ() );              	
//                }
//                
//                break;
//        }
//    }
	
    @Override
    public void keyReleased(KeyEvent e, Point2D imagePoint, IcyCanvas canvas)
    {
    }

    @Override
    public void mousePressed(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
    {
    }

    @Override
    public void mouseReleased(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
    {
    }

//	@Override
//	public void sequenceClosed(Sequence sequence)
//	{	
//	}

}
