package plugins.adufour.trackprocessors.speed;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;

import org.math.plot.Plot2DPanel;
import org.math.plot.Plot3DPanel;
import org.math.plot.PlotPanel;

import icy.file.FileUtil;
import icy.gui.component.ExcelTable;
import icy.gui.component.NumberTextField;
import icy.gui.component.NumberTextField.ValueChangeListener;
import icy.gui.dialog.MessageDialog;
import icy.gui.dialog.SaveDialog;
import icy.gui.frame.progress.AnnounceFrame;
import icy.math.ArrayMath;
import icy.painter.Overlay;
import icy.preferences.XMLPreferences;
import icy.sequence.Sequence;
import icy.util.XLSUtil;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import plugins.fab.trackmanager.PluginTrackManagerProcessor;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.spot.Detection;

/**
 * TrackManager plug-in that compute motion and speed statistics from tracks and filter them based
 * on these statistics
 * 
 * @author Alexandre Dufour
 */
public class SpeedProfiler extends PluginTrackManagerProcessor
{
    private enum Comparator
    {
        GREATER_THAN(">"), LOWER_THAN("<");
        private String symbol;
        
        private Comparator(String symbol)
        {
            this.symbol = symbol;
        }
        
        boolean compare(double d1, double d2)
        {
            switch (this)
            {
                case GREATER_THAN:
                    return d1 > d2;
                case LOWER_THAN:
                    return d1 < d2;
                default:
                    throw new UnsupportedOperationException("Unknown operation: " + toString());
            }
        }
        
        @Override
        public String toString()
        {
            return symbol;
        }
    }
    
    private enum Criterion
    {
        CRITERION_T_START("start time"),
        CRITERION_CURRENT_T("current time"),
        CRITERION_T_END("end time"),
        CRITERION_DURATION("track duration"),
        CRITERION_TOT_DISP("total displacement"),
        CRITERION_NET_DISP("net displacement"),
        CRITERION_LINEARITY("linearity"),
        CRITERION_EXTENT("extent"),
        CRITERION_MIN_DISP("minimum disp/speed"),
        CRITERION_MAX_DISP("maximum disp/speed"),
        CRITERION_AVG_DISP("average disp/speed");
      
        String description;
        
        private Criterion(String description)
        {
            this.description = description;
        }
        
        @Override
        public String toString()
        {
            return description;
        }
    }
    
    private static XMLPreferences preferences = null;
    
    
    private static final int COL_TRACK_GP = 0;
    private static final int COL_TRACK_ID = 1;
    private static final int COL_TRACK_START = 2;
    private static final int COL_TRACK_END = 3;
    private static final int COL_DURATION = 4;
    private static final int COL_TOT_DISP = 5;
    private static final int COL_NET_DISP = 6;
    private static final int COL_LINEARITY = 7;
    private static final int COL_RADIUS = 8;
    private static final int COL_MIN_DISP = 9;
    private static final int COL_MAX_DISP = 10;
    private static final int COL_AVG_DISP = 11;
    
    private JPanel options = new JPanel();
    private JCheckBox filterTracks = new JCheckBox("Filter tracks...");
    private JCheckBox useRealUnits = new JCheckBox("Use real units");
    private JCheckBox showSpeed = new JCheckBox("Show speed instead of displacement");
    
    private JPanel filter = new JPanel();
    private final ArrayList<TrackSegment> discardedTracks = new ArrayList<TrackSegment>();
    private final JComboBox<Criterion> criterion = new JComboBox<Criterion>(Criterion.values());
    private final JComboBox<Comparator> test = new JComboBox<Comparator>(Comparator.values());
    private final NumberTextField number = new NumberTextField();
    private final JLabel dispUnit = new JLabel();
    
    private JTabbedPane statistics = new JTabbedPane();
    private WritableWorkbook book;
    private File xlsFile;
    private JPanel xTab = new JPanel();
    private Plot2DPanel xPlot = new Plot2DPanel();
    private JPanel yTab = new JPanel();
    private Plot2DPanel yPlot = new Plot2DPanel();
    private JPanel zTab = new JPanel();
    private Plot2DPanel zPlot = new Plot2DPanel();
    private JPanel globalTab = new JPanel();
    private PlotPanel globalPlot;
    private JButton export = new JButton("Export statistics...");
    
    private ActionListener updater = new ActionListener()
    {
        @Override
        public void actionPerformed(ActionEvent arg0)
        {
            trackPool.fireTrackEditorProcessorChange();
        }
    };
    
    public SpeedProfiler()
    {
        if (preferences == null) preferences = getPreferencesRoot();
        
        setName("Motion Profiler");
        
        // initialize tabs
        xTab.setLayout(new BoxLayout(xTab, BoxLayout.Y_AXIS));
        yTab.setLayout(new BoxLayout(yTab, BoxLayout.Y_AXIS));
        zTab.setLayout(new BoxLayout(zTab, BoxLayout.Y_AXIS));
        globalTab.setLayout(new BoxLayout(globalTab, BoxLayout.Y_AXIS));
        statistics.addTab("Global", globalTab);
        statistics.addTab("Along X", xTab);
        statistics.addTab("Along Y", yTab);
        statistics.addTab("Along Z", zTab);
        
        xPlot.setAxisLabels("Time", "Displacement along X");
        yPlot.setAxisLabels("Time", "Displacement along Y");
        zPlot.setAxisLabels("Time", "Displacement along Z");
        
        // set the tab pane's height
        Dimension dim = statistics.getPreferredSize();
        dim.height = 900;
        statistics.setPreferredSize(dim);
        
        // options panel
        useRealUnits.setSelected(false);
        useRealUnits.addActionListener(updater);
        
        showSpeed.setSelected(false);
        showSpeed.addActionListener(updater);
        
        options.setLayout(new BoxLayout(options, BoxLayout.X_AXIS));
        options.add(filterTracks);
        options.add(useRealUnits);
        options.add(showSpeed);
        options.add(Box.createHorizontalGlue());
        options.add(export);
        
        // assemble the main interface
        // panel.setLayout(new BorderLayout());
        // panel.add(options, BorderLayout.NORTH);
        // panel.add(statistics, BorderLayout.CENTER);
        
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        panel.add(options);
        panel.add(statistics);
        
        // make the button do something
        export.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                new Thread()
                {
                    public void run()
                    {
                        // Restore last used folder
                        String xlsFolder = preferences.get("xlsFolder", null);
                        
                        String path = SaveDialog.chooseFile("Export motion data", xlsFolder, "Speed", ".xls");
                        
                        if (path == null) return;
                        
                        // store the folder in the preferences
                        preferences.put("xlsFolder", FileUtil.getDirectory(path));
                        
                        AnnounceFrame message = new AnnounceFrame("Saving motion data...", 0);
                        try
                        {
                            export(path);
                        }
                        finally
                        {
                            message.close();
                        }
                    }
                }.start();
                
            }
        });
        
        // FILTERS
        
        filter.setLayout(new BoxLayout(filter, BoxLayout.X_AXIS));
        filter.add(new JLabel(" Keep tracks with "));
        criterion.setFocusable(false);
        criterion.addActionListener(updater);
        filter.add(criterion);
        test.setFocusable(false);
        test.addActionListener(updater);
        filter.add(test);
        filter.add(number);
        filter.add(dispUnit);
        number.setNumericValue(3);
        number.addValueListener(new ValueChangeListener()
        {
            @Override
            public void valueChanged(double newValue, boolean validate)
            {
                if (validate)
                {
                    trackPool.fireTrackEditorProcessorChange();
                }
            }
        });
        
        filterTracks.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                panel.setVisible(false);
                if (filterTracks.isSelected())
                {
                    panel.remove(statistics);
                    panel.add(filter);
                    export.setVisible(false);
                }
                else
                {
                    panel.remove(filter);
                    panel.add(statistics);
                    export.setVisible(true);
                }
                panel.setVisible(true);
                
                trackPool.fireTrackEditorProcessorChange();
            }
        });
    }
    
    public void Close()
    {
        
    }
    
    public void Compute()
    {
        if (!isEnabled()) return;
        
        Sequence sequence = trackPool.getDisplaySequence();
        
        if (sequence == null && useRealUnits.isSelected()) useRealUnits.setSelected(false);
        
        useRealUnits.setEnabled(sequence != null);
        
        // declare stuff
        
        double xScale = 1.0, yScale = 1.0, zScale = 1.0, tScale = 1.0;
        
        WritableSheet globalSheet = null, xSheet = null, ySheet = null, zSheet = null;
        
        if (sequence != null && useRealUnits.isSelected())
        {
            xScale = sequence.getPixelSizeX();
            yScale = sequence.getPixelSizeY();
            zScale = sequence.getPixelSizeZ();
            tScale = sequence.getTimeInterval();
        }
        
        boolean show3D = (sequence == null || sequence.getSizeZ() > 1);
        
        // remove / clean graphical components while we work on them
        
        panel.setVisible(false);
        
        globalTab.removeAll();
        xTab.removeAll();
        yTab.removeAll();
        zTab.removeAll();
        
        if (globalPlot != null) globalPlot.removeAllPlots();
        xPlot.removeAllPlots();
        yPlot.removeAllPlots();
        if (show3D) zPlot.removeAllPlotables();
        
        // Plot2DPanel angularPlot = new Plot2DPanel();
        
        if (!filterTracks.isSelected())
        {
            // be ready to push statistics to a spreadsheet...
            try
            {
                if (xlsFile == null) xlsFile = File.createTempFile("Motion profiler", "xls");
                
                // create the workbook
                book = XLSUtil.createWorkbook(xlsFile);
            }
            catch (IOException e)
            {
                System.err.println("[Motion Profiler] Unable to create a temporary file. Cannot compute statistics");
                return;
            }
            
            if (show3D)
            {
                if (globalPlot == null || globalPlot instanceof Plot2DPanel)
                {
                    globalPlot = new Plot3DPanel();
                    globalPlot.setAxisLabels("X", "Y", "Z");
                    statistics.addTab("Along Z", zTab);
                }
            }
            else
            {
                if (globalPlot == null || globalPlot instanceof Plot3DPanel)
                {
                    globalPlot = new Plot2DPanel();
                    globalPlot.setAxisLabels("X", "Y");
                    statistics.removeTabAt(3);
                }
            }
            
            // create the sheets
            globalSheet = XLSUtil.createNewPage(book, "Global");
            xSheet = XLSUtil.createNewPage(book, "X");
            ySheet = XLSUtil.createNewPage(book, "Y");
            zSheet = show3D ? XLSUtil.createNewPage(book, "Z") : null;
            
            // write headers
            ArrayList<WritableSheet> sheets = new ArrayList<WritableSheet>(4);
            sheets.add(globalSheet);
            sheets.add(xSheet);
            sheets.add(ySheet);
            if (show3D) sheets.add(zSheet);
            
            String timeUnit = useRealUnits.isSelected() ? "sec" : "frames";
            String spaceUnit = useRealUnits.isSelected() ? "\u03BCm" : "px";
            String speedUnit = showSpeed.isSelected() ? "speed (" + spaceUnit + "/" + timeUnit + ")" : "disp. (" + spaceUnit + ")";
            
            dispUnit.setText("(" + spaceUnit + ", " + timeUnit + ")");
            
            for (WritableSheet sheet : sheets)
            {
                XLSUtil.setCellString(sheet, COL_TRACK_GP, 0, "Group");
                XLSUtil.setCellString(sheet, COL_TRACK_ID, 0, "Track #");
                XLSUtil.setCellString(sheet, COL_TRACK_START, 0, "Start (" + timeUnit + ")");
                XLSUtil.setCellString(sheet, COL_TRACK_END, 0, "End (" + timeUnit + ")");
                XLSUtil.setCellString(sheet, COL_DURATION, 0, "Duration (" + timeUnit + ")");
                XLSUtil.setCellString(sheet, COL_TOT_DISP, 0, "Total disp. (" + spaceUnit + ")");
                XLSUtil.setCellString(sheet, COL_NET_DISP, 0, "Net disp. (" + spaceUnit + ")");
                XLSUtil.setCellString(sheet, COL_LINEARITY, 0, "Linearity" + " (%)");
                XLSUtil.setCellString(sheet, COL_RADIUS, 0, "Search radius" + " (" + spaceUnit + ")");
                XLSUtil.setCellString(sheet, COL_MIN_DISP, 0, "Min. " + speedUnit);
                XLSUtil.setCellString(sheet, COL_MAX_DISP, 0, "Max. " + speedUnit);
                XLSUtil.setCellString(sheet, COL_AVG_DISP, 0, "Avg. " + speedUnit);
            }
        }
        
        int row = 1;
        
        for (TrackSegment ts : this.trackPool.getTrackSegmentList())
        {
            // check if the track is already filtered out
            if (!ts.isAllDetectionEnabled())
            {
                // was it by me?
                if (discardedTracks.contains(ts))
                {
                    // reactivate it
                    discardedTracks.remove(ts);
                    ts.setAllDetectionEnabled(true);
                }
                else continue;
            }
            
            int nbDetections = ts.getDetectionList().size();
            
            if (nbDetections < 2) continue;
            
            // extract the raw track data
            
            double[] xData = new double[nbDetections];
            double[] yData = new double[nbDetections];
            double[] zData = show3D ? new double[nbDetections] : null;
            
            // displacement data (size: N-1)
            double[] disp = new double[nbDetections - 1];
            double[] xDisp = new double[nbDetections - 1];
            double[] yDisp = new double[nbDetections - 1];
            double[] zDisp = show3D ? new double[nbDetections - 1] : null;
            
            // store minimum and maximum position to calculate the track extent
            double searchRadius = 0, xSearchRadius = 0, ySearchRadius = 0, zSearchRadius = 0;
            
            // angular shifts (size: N-2)
            // double initialAngle = 0;
            // double[] angularShift = new double[nbDetections - 2];
            // double[] instantAngularShift = new double[nbDetections - 2];
            
            // accumulate data here
            for (int i = 0; i < nbDetections; i++)
            {
                Detection det = ts.getDetectionAt(i);
                
                double x = det.getX();
                // if (x < 0) x /= -sequence.getPixelSizeX();
                xData[i] = x * xScale;
                
                double y = det.getY();
                // if (y < 0) y /= -sequence.getPixelSizeY();
                yData[i] = y * yScale;
                
                if (show3D)
                {
                    double z = det.getZ();
                    // if (z < 0) z /= -sequence.getPixelSizeZ();
                    zData[i] = z * zScale;
                }
                
                if (i > 0)
                {
                    // retrieve displacement from previous position
                    
                    xDisp[i - 1] = xData[i] - xData[i - 1];
                    yDisp[i - 1] = yData[i] - yData[i - 1];
                    if (show3D)
                    {
                        zDisp[i - 1] = zData[i] - zData[i - 1];
                        disp[i - 1] = Math.sqrt(xDisp[i - 1] * xDisp[i - 1] + yDisp[i - 1] * yDisp[i - 1] + zDisp[i - 1] * zDisp[i - 1]);
                    }
                    else
                    {
                        disp[i - 1] = Math.sqrt(xDisp[i - 1] * xDisp[i - 1] + yDisp[i - 1] * yDisp[i - 1]);
                    }
                    
                    // // store the (positive) initial angle
                    // if (i == 1) initialAngle = Math.atan2(yDisp[0], xDisp[0]) * 180 / Math.PI;
                    //
                    // if (i >= 2)
                    // {
                    // int j = i - 2;
                    // // retrieve the angular shift
                    // double angle = Math.atan2(yDisp[j], xDisp[j]) * 180 / Math.PI;
                    // // subtract the initial angle
                    // // angle -= initialAngle;
                    // // store the raw angular shift
                    // angularShift[j] = angle;
                    // // store the cumulated angular shift
                    // instantAngularShift[j] = (j == 0 ? 0 : angle - angularShift[j - 1]);
                    // }
                }
            }
            
            // // angular moments
            // double m0 = 0, m1 = 0, m2 = 0, m3 = 0, m4 = 0;
            //
            // m0 = angularShift.length;
            //
            // m1 = ArrayMath.mean(angularShift);
            //
            // for (int a = 0; a < angularShift.length; a++)
            // {
            // double normalised = angularShift[a] - m1;
            // m2 += Math.pow(normalised, 2.0);
            // m3 += Math.pow(normalised, 3.0);
            // m4 += Math.pow(normalised, 4.0);
            // }
            // m2 /= m0;
            // m3 /= m0;
            // m4 /= m0;
            //
            // m3 *= Math.sqrt(m0 * (m0 - 1)) / (Math.pow(m2, 1.5) * (m0 - 2));
            // m4 = (m4 / (m2 * m2)) - 3;
            // m4 = ((m0 + 1) * m4) + 6;
            // m4 *= (m0 - 1) / ((m0 - 2) * (m0 - 3));
            // System.out.println(trackPool.getTrackGroupContainingSegment(ts).getDescription() +
            // "Angular moments (0 to 5)");
            // System.out.println(m0);
            // System.out.println(m1);
            // System.out.println(m2);
            // System.out.println(m3);
            // System.out.println(m4);
            // System.out.println("--");
            
            // angularPlot.addHistogramPlot("angular histogram", angularShift, 10);
            // angularPlot.addLinePlot("Angular shift", instantAngularShift);
            
            // double[] angularFFT = new double[angularShift.length];
            // System.arraycopy(angularShift, 0, angularFFT, 0, angularShift.length);
            // DoubleFFT_1D fft = new DoubleFFT_1D(angularFFT.length);
            // fft.realForward(angularFFT);
            
            // compute parameters here
            
            // track duration
            
            // track start/end
            double startTime = ts.getFirstDetection().getT() * tScale;
            double endTime = ts.getLastDetection().getT() * tScale;
            double duration = (nbDetections - 1) * tScale;
            
            // relative displacement (start to end)
            double xNetDisp = Math.abs(xData[xData.length - 1] - xData[0]);
            double yNetDisp = Math.abs(yData[yData.length - 1] - yData[0]);
            double zNetDisp = show3D ? Math.abs(zData[zData.length - 1] - zData[0]) : 0;
            double netDisp = Math.sqrt(xNetDisp * xNetDisp + yNetDisp * yNetDisp + zNetDisp * zNetDisp);
            
            // total (sum of) displacement(s)
            double xTotDisp = ArrayMath.sum(ArrayMath.abs(xDisp, false));
            double yTotDisp = ArrayMath.sum(ArrayMath.abs(yDisp, false));
            double zTotDisp = show3D ? ArrayMath.sum(ArrayMath.abs(zDisp, false)) : 0;
            double totDisp = ArrayMath.sum(ArrayMath.abs(disp, false));
            
            // linearity (ratio of total to net displacement)
            double xTortuosity = xNetDisp / xTotDisp;
            double yTortuosity = yNetDisp / yTotDisp;
            double zTortuosity = zNetDisp / zTotDisp;
            double tortuosity = netDisp / totDisp;
            
            // extent (largest distance between any 2 detections)
            for (int i = 0; i < nbDetections; i++)
                for (int j = i + 1; j < nbDetections; j++)
                {
                    double dx = xData[i] - xData[j];
                    double dy = yData[i] - yData[j];
                    double dz = show3D ? zData[i] - zData[j] : 0;
                    double dist = Math.sqrt(dx * dx + dy * dy + dz * dz);
                    
                    if (dist > searchRadius)
                    {
                        searchRadius = dist;
                        xSearchRadius = Math.abs(dx);
                        ySearchRadius = Math.abs(dy);
                        zSearchRadius = Math.abs(dz);
                    }
                }
            
            // before computing displacement or speed, make sure it's the right one
            if (showSpeed.isSelected())
            {
                ArrayMath.divide(disp, tScale, disp);
                ArrayMath.divide(xDisp, tScale, xDisp);
                ArrayMath.divide(yDisp, tScale, yDisp);
                if (show3D) ArrayMath.divide(zDisp, tScale, zDisp);
            }
            
            double xMinDisp = ArrayMath.min(xDisp);
            double yMinDisp = ArrayMath.min(yDisp);
            double zMinDisp = show3D ? ArrayMath.min(zDisp) : 0;
            double minDisp = ArrayMath.min(disp);
            
            double xMaxDisp = ArrayMath.max(xDisp);
            double yMaxDisp = ArrayMath.max(yDisp);
            double zMaxDisp = show3D ? ArrayMath.max(zDisp) : 0;
            double maxDisp = ArrayMath.max(disp);
            
            double xAvgDisp = ArrayMath.mean(xDisp);
            double yAvgDisp = ArrayMath.mean(yDisp);
            double zAvgDisp = show3D ? ArrayMath.mean(zDisp) : 0;
            double avgDisp = ArrayMath.mean(disp);
            
            // center all tracks from plotting purposes
            ArrayMath.subtract(xData, xData[0], xData);
            ArrayMath.subtract(yData, yData[0], yData);
            if (show3D) ArrayMath.subtract(zData, zData[0], zData);
            
            // time to filter
            if (filterTracks.isSelected())
            {
                double testValue = number.getNumericValue();
                Comparator comparator = (Comparator) test.getSelectedItem();
                
                switch ((Criterion) criterion.getSelectedItem())
                {
                    case CRITERION_T_START:
                        ts.setAllDetectionEnabled(comparator.compare(ts.getFirstDetection().getT(), testValue));
                    break;
                    case CRITERION_CURRENT_T:
                        for (Detection det : ts.getDetectionList())
                            det.setEnabled(comparator.compare(det.getT(), testValue));
                    break;
                    case CRITERION_T_END:
                        ts.setAllDetectionEnabled(comparator.compare(ts.getLastDetection().getT(), testValue));
                    break;
                    case CRITERION_DURATION:
                        ts.setAllDetectionEnabled(comparator.compare(nbDetections - 1, testValue));
                    break;
                    case CRITERION_LINEARITY:
                        ts.setAllDetectionEnabled(comparator.compare(tortuosity, testValue));
                    break;
                    case CRITERION_EXTENT:
                        ts.setAllDetectionEnabled(comparator.compare(searchRadius, testValue));
                    break;
                    case CRITERION_TOT_DISP:
                        ts.setAllDetectionEnabled(comparator.compare(totDisp, testValue));
                    break;
                    case CRITERION_NET_DISP:
                        ts.setAllDetectionEnabled(comparator.compare(netDisp, testValue));
                    break;
                    case CRITERION_MIN_DISP:
                        ts.setAllDetectionEnabled(comparator.compare(minDisp, testValue));
                    break;
                    case CRITERION_MAX_DISP:
                        ts.setAllDetectionEnabled(comparator.compare(maxDisp, testValue));
                    break;
                    case CRITERION_AVG_DISP:
                        ts.setAllDetectionEnabled(comparator.compare(avgDisp, testValue));
                    break;
                    default:
                }
            }
            
            if (!ts.isAllDetectionEnabled())
            {
                discardedTracks.add(ts);
                
                // don't export anything and move to the next track
                continue;
            }
            
            if (filterTracks.isSelected()) continue;
            
            Color color = ts.getFirstDetection().getColor();
            
            TrackGroup group = trackPool.getTrackGroupContainingSegment(ts);
            String groupName = group.getDescription();
            
            // track index
            int trackIndex = group.getTrackSegmentList().indexOf(ts);
            
            // track name
            String trackName = group + " - Track " + trackIndex;
            
            xPlot.addLinePlot(trackName, color, xDisp);
            yPlot.addLinePlot(trackName, color, yDisp);
            if (show3D) zPlot.addLinePlot(trackName, color, zDisp);
            
            if (show3D)
            {
                ((Plot3DPanel) globalPlot).addLinePlot(trackName, color, xData, yData, zData);
            }
            else
            {
                ((Plot2DPanel) globalPlot).addLinePlot("Track " + trackIndex, color, xData, yData);
            }
            
            // store in the spreadsheet
            
            XLSUtil.setCellString(globalSheet, COL_TRACK_GP, row, groupName);
            XLSUtil.setCellString(xSheet, COL_TRACK_GP, row, groupName);
            XLSUtil.setCellString(ySheet, COL_TRACK_GP, row, groupName);
            if (show3D) XLSUtil.setCellString(zSheet, COL_TRACK_GP, row, groupName);
            
            XLSUtil.setCellNumber(globalSheet, COL_TRACK_ID, row, trackIndex);
            XLSUtil.setCellNumber(xSheet, COL_TRACK_ID, row, trackIndex);
            XLSUtil.setCellNumber(ySheet, COL_TRACK_ID, row, trackIndex);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_TRACK_ID, row, trackIndex);
            
            XLSUtil.setCellNumber(globalSheet, COL_TRACK_START, row, startTime);
            XLSUtil.setCellNumber(xSheet, COL_TRACK_START, row, startTime);
            XLSUtil.setCellNumber(ySheet, COL_TRACK_START, row, startTime);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_TRACK_START, row, startTime);
            
            XLSUtil.setCellNumber(globalSheet, COL_TRACK_END, row, endTime);
            XLSUtil.setCellNumber(xSheet, COL_TRACK_END, row, endTime);
            XLSUtil.setCellNumber(ySheet, COL_TRACK_END, row, endTime);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_TRACK_END, row, endTime);
            
            XLSUtil.setCellNumber(globalSheet, COL_DURATION, row, duration);
            XLSUtil.setCellNumber(xSheet, COL_DURATION, row, duration);
            XLSUtil.setCellNumber(ySheet, COL_DURATION, row, duration);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_DURATION, row, duration);
            
            XLSUtil.setCellNumber(globalSheet, COL_TOT_DISP, row, totDisp);
            XLSUtil.setCellNumber(xSheet, COL_TOT_DISP, row, xTotDisp);
            XLSUtil.setCellNumber(ySheet, COL_TOT_DISP, row, yTotDisp);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_TOT_DISP, row, zTotDisp);
            
            XLSUtil.setCellNumber(globalSheet, COL_NET_DISP, row, netDisp);
            XLSUtil.setCellNumber(xSheet, COL_NET_DISP, row, xNetDisp);
            XLSUtil.setCellNumber(ySheet, COL_NET_DISP, row, yNetDisp);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_NET_DISP, row, zNetDisp);
            
            XLSUtil.setCellNumber(globalSheet, COL_LINEARITY, row, tortuosity);
            XLSUtil.setCellNumber(xSheet, COL_LINEARITY, row, xTortuosity);
            XLSUtil.setCellNumber(ySheet, COL_LINEARITY, row, yTortuosity);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_LINEARITY, row, zTortuosity);
            
            XLSUtil.setCellNumber(globalSheet, COL_RADIUS, row, searchRadius);
            XLSUtil.setCellNumber(xSheet, COL_RADIUS, row, xSearchRadius);
            XLSUtil.setCellNumber(ySheet, COL_RADIUS, row, ySearchRadius);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_RADIUS, row, zSearchRadius);
            
            XLSUtil.setCellNumber(globalSheet, COL_MIN_DISP, row, minDisp);
            XLSUtil.setCellNumber(xSheet, COL_MIN_DISP, row, xMinDisp);
            XLSUtil.setCellNumber(ySheet, COL_MIN_DISP, row, yMinDisp);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_MIN_DISP, row, zMinDisp);
            
            XLSUtil.setCellNumber(globalSheet, COL_MAX_DISP, row, maxDisp);
            XLSUtil.setCellNumber(xSheet, COL_MAX_DISP, row, xMaxDisp);
            XLSUtil.setCellNumber(ySheet, COL_MAX_DISP, row, yMaxDisp);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_MAX_DISP, row, zMaxDisp);
            
            XLSUtil.setCellNumber(globalSheet, COL_AVG_DISP, row, avgDisp);
            XLSUtil.setCellNumber(xSheet, COL_AVG_DISP, row, xAvgDisp);
            XLSUtil.setCellNumber(ySheet, COL_AVG_DISP, row, yAvgDisp);
            if (show3D) XLSUtil.setCellNumber(zSheet, COL_AVG_DISP, row, zAvgDisp);
            
            row++;
        }
        
        if (!filterTracks.isSelected())
        {
            // X display
            ExcelTable xTable = new ExcelTable(xSheet);
            xTable.setPreferredSize(new Dimension(0, 200));
            xTab.add(xTable);
            xPlot.setPreferredSize(new Dimension(0, 300));
            xTab.add(xPlot);
            
            // Y display
            ExcelTable yTable = new ExcelTable(ySheet);
            yTable.setPreferredSize(new Dimension(0, 200));
            yTab.add(yTable);
            yPlot.setPreferredSize(new Dimension(0, 300));
            yTab.add(yPlot);
            
            if (show3D)
            {
                // Z display
                ExcelTable zTable = new ExcelTable(zSheet);
                zTable.setPreferredSize(new Dimension(0, 200));
                zTab.add(zTable);
                zPlot.setPreferredSize(new Dimension(0, 300));
                zTab.add(zPlot);
            }
            
            // global display
            ExcelTable globalTable = new ExcelTable(globalSheet);
            globalTable.setPreferredSize(new Dimension(0, 200));
            globalTab.add(globalTable);
            globalPlot.setPreferredSize(new Dimension(0, 600));
            globalTab.add(globalPlot);
            // globalTab.add(angularPlot);
        }
        
        panel.setVisible(true);
        
        if (sequence != null) sequence.overlayChanged((Overlay) null);
    }
    
    public void displaySequenceChanged()
    {
        Compute();
    }
    
    private void export(String path)
    {
        try
        {
            book.setOutputFile(new File(path));
            XLSUtil.saveAndClose(book);
        }
        catch (Exception e)
        {
            MessageDialog.showDialog("Cannot export file", MessageDialog.ERROR_MESSAGE);
        }
    }
}
