package plugins.ylemontag.matlabfunctioncaller;

import icy.sequence.Sequence;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Variant several types of objects, all of them being bind-able to a Matlab
 * function argument
 */
public class Variant
{
	/**
	 * Allowed types of object
	 */
	public static enum Type
	{
		INTEGER ("Integer"      ),
		INTEGERS("Integer array"),
		DECIMAL ("Decimal"      ),
		DECIMALS("Decimal array"),
		TEXT    ("Text"         ),
		SEQUENCE("Sequence"     );
		
		private String _name;
		
		private Type(String name)
		{
			_name = name;
		}
		
		@Override
		public String toString()
		{
			return _name;
		}
	}
	
	private Type   _type ;
	private Object _value;
	
	/**
	 * Wrap an integer
	 */
	public Variant(int value)
	{
		_type  = Type.INTEGER;
		_value = value;
	}
	
	/**
	 * Wrap an integer array
	 * @throws IllegalArgumentException If value==null
	 */
	public Variant(int[] value)
	{
		ensureNonNull(value);
		_type  = Type.INTEGERS;
		_value = value;
	}
	
	/**
	 * Wrap a decimal (i.e. double)
	 */
	public Variant(double value)
	{
		_type  = Type.DECIMAL;
		_value = value;
	}
	
	/**
	 * Wrap a decimal array
	 * @throws IllegalArgumentException If value==null
	 */
	public Variant(double[] value)
	{
		ensureNonNull(value);
		_type  = Type.DECIMALS;
		_value = value;
	}
	
	/**
	 * Wrap a string
	 * @throws IllegalArgumentException If value==null
	 */
	public Variant(String value)
	{
		ensureNonNull(value);
		_type  = Type.TEXT;
		_value = value;
	}
	
	/**
	 * Wrap a sequence
	 * @throws IllegalArgumentException If value==null
	 */
	public Variant(Sequence value)
	{
		ensureNonNull(value);
		_type  = Type.SEQUENCE;
		_value = value;
	}
	
	/**
	 * Underlying type
	 */
	public Type getType()
	{
		return _type;
	}
	
	/**
	 * Return the value casted as an integer
	 */
	public int getAsInteger()
	{
		return (Integer)_value;
	}
	
	/**
	 * Return the value casted as an integer array
	 */
	public int[] getAsIntegers()
	{
		return (int[])_value;
	}
	
	/**
	 * Return the value casted as a decimal (i.e. double)
	 */
	public double getAsDecimal()
	{
		return (Double)_value;
	}
	
	/**
	 * Return the value casted as a double array
	 */
	public double[] getAsDecimals()
	{
		return (double[])_value;
	}
	
	/**
	 * Return the value casted as a string
	 */
	public String getAsText()
	{
		return (String)_value;
	}
	
	/**
	 * Return the value casted as a sequence
	 */
	public Sequence getAsSequence()
	{
		return (Sequence)_value;
	}
	
	@Override
	public String toString()
	{
		switch(_type)
		{
			case INTEGER:
			case DECIMAL:
				return _value.toString();
			
			case TEXT:
				return getAsText();
			
			case SEQUENCE:
				return getAsSequence().getName();
				
			case INTEGERS:
				return formatIntegerArray(getAsIntegers());
				
			case DECIMALS:
				return formatDecimalArray(getAsDecimals());
			
			default:
				throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Implementation of the toString() function for integer arrays
	 */
	private static String formatIntegerArray(int[] value)
	{
		String retVal = "[";
		boolean first = true;
		for(int v : value) {
			if(first) {
				first = false;
			}
			else {
				retVal += " ";
			}
			retVal += v;
		}
		retVal += "]";
		return retVal;
	}
	
	/**
	 * Implementation of the toString() function for decimal arrays
	 */
	private static String formatDecimalArray(double[] value)
	{
		String retVal = "[";
		boolean first = true;
		for(double v : value) {
			if(first) {
				first = false;
			}
			else {
				retVal += " ";
			}
			retVal += v;
		}
		retVal += "]";
		return retVal;
	}
	
	/**
	 * Throw an exception if the given object is null
	 */
	private static void ensureNonNull(Object obj)
	{
		if(obj==null) {
			throw new IllegalArgumentException(
				"Wrapping a null object into a Variant object is forbidden"
			);
		}
	}
}
