package plugins.ylemontag.matlabfunctioncaller;

import icy.sequence.Sequence;

import java.io.File;
import java.io.IOException;

import plugins.ylemontag.matlabio.MatlabExporter;
import plugins.ylemontag.matlabio.lib.MLArrays;
import plugins.ylemontag.matlabio.lib.MatFileWriter;


/**
 * 
 * @author Yoann Le Montagner
 * 
 * Set of input arguments for a Matlab function
 */
public class ArgumentSetInput extends ArgumentSet
{
	/**
	 * Constructor
	 */
	public ArgumentSetInput(String[] names)
	{
		super(names);
	}
	
	/**
	 * Bind an argument
	 */
	public void bind(String name, Variant value)
	{
		if(!_data.containsKey(name)) {
			throw new IllegalArgumentException("Invalid argument name: " + name);
		}
		_data.put(name, value);
	}
	
	/**
	 * Save the argument values to a Matlab .mat file
	 */
	public void save(String fileName) throws IOException
	{
		save(new File(fileName));
	}
	
	/**
	 * Save the argument values to a Matlab .mat file
	 */
	public void save(File file) throws IOException
	{
		MatFileWriter writer = new MatFileWriter(file, false);
		for(String name : getArgumentNames()) {
			saveArgument(name, writer);
		}
	}
	
	/**
	 * Save the given argument to the file
	 */
	private void saveArgument(String name, MatFileWriter writer) throws IOException
	{
		// Retrieve the variant object binded to the argument 'name'
		Variant value = getArgument(name);
		if(value==null) {
			return;
		}
		
		// Different behaviors depending on the actual type of the variant object
		switch(value.getType())
		{
			case INTEGER : writer.putData(new MLArrays.Int32 (name, value.getAsInteger ())); break;
			case INTEGERS: writer.putData(new MLArrays.Int32 (name, value.getAsIntegers())); break;
			case DECIMAL : writer.putData(new MLArrays.Double(name, value.getAsDecimal ())); break;
			case DECIMALS: writer.putData(new MLArrays.Double(name, value.getAsDecimals())); break;
			
			case TEXT:
				writer.putData(new MLArrays.Char(name, value.getAsText()));
				break;
			
			case SEQUENCE:
				saveSequenceArgument(name, writer, value.getAsSequence());
				break;
				
			default:
				throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Bind a sequence to an argument
	 */
	private void saveSequenceArgument(String name, MatFileWriter writer, Sequence value)
		throws IOException
	{
		MatlabExporter exporter = new MatlabExporter(writer);
		exporter.putData(value, name, getDimensionMapping(), getComplexModeForInput());
		exporter.export();
	}
}
