package plugins.ylemontag.matlabfunctioncaller;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.LinkedList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Describe the prototype of a Matlab function (name, input/output arguments, etc)
 */
public class MatlabFunctionHeader
{
	private File     _folder;
	private String   _name  ;
	private String[] _in    ;
	private String[] _out   ;
	
	/**
	 * Constructor
	 */
	public MatlabFunctionHeader(File folder, String name, String[] in, String[] out)
	{
		_folder = folder;
		_name   = name  ;
		_in     = in    ;
		_out    = out   ;
	}
	
	/**
	 * Folder containing this function
	 */
	public File getFolder()
	{
		return _folder;
	}
	
	/**
	 * Name of the function
	 */
	public String getName()
	{
		return _name;
	}
	
	/**
	 * Return the name of the input arguments
	 */
	public String[] getIn()
	{
		return _in;
	}
	
	/**
	 * Return the name of the output arguments
	 */
	public String[] getOut()
	{
		return _out;
	}
	
	@Override
	public String toString()
	{
		return "[" + implode(_out) + "] = " + _name + " (" + implode(_in) + ")";
	}
	
	/**
	 * Auxilliary function for toString
	 */
	private static String implode(String[] names)
	{
		String retVal = "";
		for(int k=0; k<names.length; ++k) {
			if(k>0) {
				retVal += ", ";
			}
			retVal += names[k];
		}
		return retVal;
	}
	
	/**
	 * Exception thrown by the parsing functions
	 */
	public static class ParsingException extends Exception
	{
		private static final long serialVersionUID = 1L;
		
		public ParsingException()
		{
			super();
		}
		
		public ParsingException(String message)
		{
			super(message);
		}
	}
	
	/**
	 * Tokens used for the parsing of the function header
	 */
	private static class Token
	{
		/**
		 * Types of tokens
		 */
		public static enum Type
		{
			FUNCTION("FUN", "function"),
			OPEN_P  ("("  , "\\("),
			CLOSE_P (")"  , "\\)"),
			OPEN_B  ("["  , "\\["),
			CLOSE_B ("]"  , "\\]"),
			COMMA   (","  , ","  ),
			EQUAL   ("="  , "="  ),
			ID      ("ID" , "([A-Za-z][A-Za-z_0-9]*)");
			
			private String  _name   ;
			private Pattern _pattern;
			
			private Type(String name, String regex)
			{
				_name    = name;
				_pattern = Pattern.compile("^[ \\t]*" + regex);
			}
			
			public String getName()
			{
				return _name;
			}
			
			public Pattern getPattern()
			{
				return _pattern;
			}
		}
		
		private Type   _type ;
		private Object _value;
		
		/**
		 * Constructor for tokens with no value
		 */
		private Token(Type type)
		{
			_type  = type;
			_value = null;
		}
		
		/**
		 * Constructor
		 */
		private Token(Type type, Object value)
		{
			_type  = type ;
			_value = value;
		}
		
		/**
		 * Type of the token
		 */
		public Type getType()
		{
			return _type;
		}
		
		/**
		 * Value associated to the token (may be null if no associated value)
		 */
		public Object getValue()
		{
			return _value;
		}
		
		@Override
		public String toString()
		{
			String retVal = _type.getName();
			if(_value!=null) {
				retVal += "(" + _value + ")";
			}
			return retVal;
		}
		
		/**
		 * Parse one line of text
		 */
		public static LinkedList<Token> parse(String text) throws ParsingException
		{
			LinkedList<Token> retVal = new LinkedList<Token>();
			int pos    = 0;
			int length = text.length();
			while(pos<length)
			{
				// Try to detect the type of the next token
				Type    type = null;
				Matcher m    = null;
				for(Type t : Type.values()) {
					m = t.getPattern().matcher(text.subSequence(pos, length));
					if(m.find()) {
						type = t;
						break;
					}
				}
				
				// If no token type matches, try to match either a commentary or the end
				// of the stream
				if(type==null) {
					if(!Pattern.matches("[ \\t]*(?:%.*)?", text.subSequence(pos, length))) {
						throw new ParsingException("Invalid token stream");
					}
					pos = length; // break
				}
				
				// Otherwise, decode the token, and increment the index
				else {
					pos += m.group(0).length();
					retVal.add(decodeMatchedToken(type, m));
				}
			}
			return retVal;
		}
		
		/**
		 * Decode the matched token
		 */
		private static Token decodeMatchedToken(Type type, Matcher m) throws ParsingException
		{
			// Decode the value of the ID
			if(type==Type.ID) {
				if(m.groupCount()!=1) {
					throw new ParsingException("ID token badly formatted");
				}
				return new Token(type, m.group(1));
			}
			
			// Otherwise, return a simple token
			else {
				return new Token(type);
			}
		}
	}
	
	/**
	 * Parse a Matlab .m file and return a function header
	 */
	public static MatlabFunctionHeader parse(String fileName) throws ParsingException, IOException
	{
		return parse(new File(fileName));
	}
	
	/**
	 * Parse a Matlab .m file and return a function header
	 */
	public static MatlabFunctionHeader parse(File file) throws ParsingException, IOException
	{
		BufferedReader reader = new BufferedReader(new FileReader(file));
		try {
			String currentLine = reader.readLine();
			while(currentLine!=null)
			{
				// Parse the current line of text
				LinkedList<Token> tokens = Token.parse(currentLine);
				
				// If no token is found, go to the next line
				if(tokens.isEmpty()) {
					currentLine = reader.readLine();
					continue;
				}
				
				// Otherwise, decode the list of tokens as a Matlab function header, as
				// this header is always supposed to be defined on the first non-empty
				// line in the text
				return decodeFunctionHeader(file.getParentFile(), tokens);
			}
			throw new ParsingException("No Matlab function header found");
		}
		finally {
			reader.close();
		}
	}
	
	/**
	 * Try to interpret a list tokens as a Matlab function header
	 */
	private static MatlabFunctionHeader decodeFunctionHeader(File folder, LinkedList<Token> tokens)
		throws ParsingException
	{
		Token token = null;
		
		// The first token must be the keyword 'function'
		token = popToken(tokens);
		if(token.getType()!=Token.Type.FUNCTION) {
			throw new ParsingException("Expected: 'function' keyword");
		}
		
		// Next comes the list of output arguments
		String[] outArguments = decodeOuputArguments(tokens);
		
		// Now, extract the name of the function
		token = popToken(tokens);
		if(token.getType()!=Token.Type.ID) {
			throw new ParsingException("Expected: name of the function");
		}
		String functionName = (String)token.getValue();
		
		// Next, extract the list of input arguments
		String[] inArguments = decodeInputArguments(tokens);
		
		// Finally, ensure that all the tokens have been consume, and return the result
		if(!tokens.isEmpty()) {
			throw new ParsingException("Unexpected tokens at the end of the function header");
		}
		return new MatlabFunctionHeader(folder, functionName, inArguments, outArguments);
	}
	
	/**
	 * Decode the output arguments, and consume the '=' sign if any
	 */
	private static String[] decodeOuputArguments(LinkedList<Token> tokens) throws ParsingException
	{
		// There exists three syntaxes for output arguments:
		//  1) function [out1, out2, ... , outN] = nameFun ... (a list with only out1 is also allowed)
		//  2) function out = nameFun ...
		//  3) function nameFun ... (no output argument in this case)
		Token token = popToken(tokens);
		
		 // Case 1)
		if(token.getType()==Token.Type.OPEN_B) {
			String[] retVal = decodeArgumentList(tokens, Token.Type.CLOSE_B);
			Token equalToken = popToken(tokens);
			if(equalToken.getType()!=Token.Type.EQUAL) {
				throw new ParsingException("Expected: equal sign");
			}
			return retVal;
		}
		
		// Case 2) or 3)
		else if(token.getType()==Token.Type.ID) {
			String id = (String)token.getValue();
			
			// Special sub-case of case 3) => neither input nor output arguments
			if(tokens.isEmpty()) {
				tokens.addFirst(token); // Rebuild the list of tokens
				return new String[0];
			}
			Token equalToken = popToken(tokens);
			
			// Case 2)
			if(equalToken.getType()==Token.Type.EQUAL) { 
				return new String[] { id };
			}
			
			// Case 3)
			else {
				tokens.addFirst(equalToken); tokens.addFirst(token); // Rebuild the list of tokens
				return new String[0];
			}
		}
		
		// Syntax error
		else {
			throw new ParsingException("Expected: list of output arguments");
		}
	}
	
	/**
	 * Decode the input arguments
	 */
	private static String[] decodeInputArguments(LinkedList<Token> tokens) throws ParsingException
	{
		// Special syntax if no input argument
		if(tokens.isEmpty()) {
			return new String[0];
		}
		
		// Only one syntax here for input arguments:
		//  (out1, out2, ... , outN)
		Token token = popToken(tokens);
		if(token.getType()!=Token.Type.OPEN_P) {
			throw new ParsingException("Expected: list of input arguments");
		}
		return decodeArgumentList(tokens, Token.Type.CLOSE_P);
	}
	
	/**
	 * Decode a list of arguments separated with commas
	 */
	private static String[] decodeArgumentList(LinkedList<Token> tokens, Token.Type expectedTerminator)
		throws ParsingException
	{
		LinkedList<String> arguments = new LinkedList<String>();
		boolean endOfList = false;
		while(!endOfList) {
			Token token = popToken(tokens);
			switch(token.getType())
			{
				// Argument identifier
				case ID:
					arguments.add((String)token.getValue());
					break;
				
				// Separator
				case COMMA:
					break;
				
				// End of list
				default:
					if(token.getType()==expectedTerminator) {
						endOfList = true;
					}
					else {
						throw new ParsingException("Unexpected token: " + token);
					}
					break;
			}
		}
		
		// Convert the list of string into an array
		String[] retVal = new String[arguments.size()];
		int k = 0;
		for(String a : arguments) {
			retVal[k] = a;
			++k;
		}
		return retVal;
	}
	
	/**
	 * Ensure that the given list of tokens is not empty, and pop the first token
	 */
	private static Token popToken(LinkedList<Token> tokens) throws ParsingException
	{
		if(tokens.isEmpty()) {
			throw new ParsingException("Empty list of tokens");
		}
		return tokens.pop();
	}
}
