package plugins.ylemontag.matlabfunctioncaller;

import icy.sequence.Sequence;
import icy.type.TypeUtil;
import icy.type.collection.array.Array1DUtil;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import plugins.ylemontag.matlabio.MatlabImporter;
import plugins.ylemontag.matlabio.lib.MLArray;
import plugins.ylemontag.matlabio.lib.MLIOException;
import plugins.ylemontag.matlabio.lib.MLMeta;
import plugins.ylemontag.matlabio.lib.MLType;
import plugins.ylemontag.matlabio.lib.MatFileReader;


/**
 * 
 * @author Yoann Le Montagner
 * 
 * Set of output arguments for a Matlab function
 */
public class ArgumentSetOutput extends ArgumentSet
{
	private Object _mutex;
	private Map<String, Variant.Type> _preferredTypes;
	private MatFileReader _reader;
	
	/**
	 * Constructor
	 */
	public ArgumentSetOutput(String[] names)
	{
		super(names);
		_mutex = new Object();
		_preferredTypes = new HashMap<String, Variant.Type>();
	}
	
	/**
	 * Load the argument values from a Matlab .mat file
	 */
	public void load(String fileName) throws IOException
	{
		load(new File(fileName));
	}
	
	/**
	 * Load the argument values from a Matlab .mat file
	 */
	public void load(File file) throws IOException
	{
		synchronized (_mutex)
		{
			_preferredTypes.clear();
			_reader = new MatFileReader(file);
			for(String name : getArgumentNames()) {
				loadArgument(name);
			}
		}
	}
	
	/**
	 * Load the given argument from the file
	 */
	private void loadArgument(String name) throws IOException
	{
		// Reset the variable
		_data.put(name, null);
		
		// Make sure that a variable with the suitable name exists in the Matlab file
		if(_reader==null || !_reader.getKeys().contains(name)) {
			return;
		}
		
		// Compute how the Matlab variable should be interpreted
		Variant.Type targetType = _preferredTypes.get(name);
		if(targetType==null) {
			targetType = computeDefaultType(_reader.getMeta(name));
			if(targetType==null) {
				return;
			}
		}
		
		// Read and interpret the data
		Variant value = null;
		switch(targetType) {
			case INTEGER : value = interpretAsInteger (_reader.getData(name)); break;
			case INTEGERS: value = interpretAsIntegers(_reader.getData(name)); break;
			case DECIMAL : value = interpretAsDecimal (_reader.getData(name)); break;
			case DECIMALS: value = interpretAsDecimals(_reader.getData(name)); break;
			case TEXT    : value = interpretAsText    (_reader.getData(name)); break;
			case SEQUENCE: value = interpretAsSequence(name); break;
			default:
				throw new RuntimeException("Unreachable code point");
		}
		_data.put(name, value);
	}
	
	/**
	 * Recast the given argument into the given type
	 */
	public void recast(String name, Variant.Type targetType) throws IOException
	{
		synchronized (_mutex)
		{
			Variant previousValue = _data.get(name);
			if(previousValue==null) {
				throw new IllegalArgumentException("Invalid variable name: " + name);
			}
			
			// Nothing to do if the variable has already the right type
			if(previousValue.getType()==targetType) {
				return;
			}
			
			// Check whether the given variable can be recasted in the given target type
			if(!isRecastable(name, targetType)) {
				throw new IllegalArgumentException("Cannot recast the variable " + name + " as " + targetType);
			}
			
			// Reload the data with the right type
			_preferredTypes.put(name, targetType);
			loadArgument(name);
		}
	}
	
	/**
	 * Return the list of types in which the given variable can be recast
	 */
	public Variant.Type[] getRecastableTypes(String name)
	{
		synchronized (_mutex)
		{
			LinkedList<Variant.Type> buffer = new LinkedList<Variant.Type>();
			for(Variant.Type t : Variant.Type.values()) {
				if(isRecastable(name, t)) {
					buffer.add(t);
				}
			}
			Variant.Type[] retVal = new Variant.Type[buffer.size()];
			for(int k=0; k<retVal.length; ++k) {
				retVal[k] = buffer.pop();
			}
			return retVal;
		}
	}
	
	/**
	 * Check whether the given result can be casted in the given type
	 */
	public boolean isRecastable(String name, Variant.Type targetType)
	{
		synchronized (_mutex)
		{
			if(_reader==null) {
				throw new IllegalStateException("No Matlab .mat result file set");
			}
			MLMeta meta = _reader.getMeta(name);
			if(meta==null) {
				throw new IllegalArgumentException("Invalid variable name: " + name);
			}
			return isRecastable(meta, targetType);
		}
	}
	
	/**
	 * Check whether a Matlab raw result can be recast in the given type or not,
	 * depending on its meta-data
	 */
	private static boolean isRecastable(MLMeta meta, Variant.Type targetType)
	{
		
		boolean nativelyImportable = (meta.getType().getIsNumeric() && !meta.getIsComplex());
		switch(targetType)
		{
			case INTEGER : return nativelyImportable && meta.getSize()==1 && convertibleToInteger(meta.getType());
			case INTEGERS: return nativelyImportable && convertibleToInteger(meta.getType());
			case DECIMAL : return nativelyImportable && meta.getSize()==1;
			case DECIMALS: return nativelyImportable;
			case TEXT    : return meta.getType()==MLType.CHAR;
			case SEQUENCE: return MatlabImporter.isImportableAsSequence(meta);
			default:
				throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Return the default type that can be guessed from the meta-data associated
	 * to the variable
	 */
	private static Variant.Type computeDefaultType(MLMeta meta)
	{
		MLType type = meta.getType();
		int[]  dims = meta.getDimensions();
		boolean nativelyImportable = (type.getIsNumeric() && !meta.getIsComplex());
		
		// All char-valued variables are interpreted as string
		if(type==MLType.CHAR) {
			return Variant.Type.TEXT;
		}
		
		// Try to import non-trivial Matlab multi-dimension arrays as sequences
		// (only for variable that otherwise would not be natively importable)
		else if(MatlabImporter.isImportableAsSequence(meta) && (!nativelyImportable
			|| (dims.length>=2 && dims[0]>=2 && dims[1]>=2)))
		{
			return Variant.Type.SEQUENCE;
		}
		
		// Otherwise, numeric-valued variables are imported as scalar or arrays
		else if(nativelyImportable) {
			if(meta.getSize()==1) {
				return convertibleToInteger(type) ? Variant.Type.INTEGER : Variant.Type.DECIMAL;
			}
			else {
				return convertibleToInteger(type) ? Variant.Type.INTEGERS : Variant.Type.DECIMALS;
			}
		}
		
		// Cannot import this variable
		else {
			return null;
		}
	}
	
	/**
	 * Check whether the given Matlab type can be converted to an integer
	 */
	private static boolean convertibleToInteger(MLType type)
	{
		switch(type)
		{
			case INT8  :
			case INT16 :
			case UINT8 :
			case UINT16:
			case INT32 :
				return true;
			default:
				return false;
		}
	}
	
	/**
	 * Load an argument as an integer
	 */
	private static Variant interpretAsInteger(MLArray rawData) throws MLIOException
	{
		int buffer = 0;
		switch(rawData.getType())
		{
			case INT8  : buffer = rawData.getAsInt8 (); break;
			case INT16 : buffer = rawData.getAsInt16(); break;
			case UINT8 : buffer = TypeUtil.unsign(rawData.getAsUInt8 ()); break;
			case UINT16: buffer = TypeUtil.unsign(rawData.getAsUInt16()); break;
			default    : buffer = rawData.getAsInt32(); break;
		}
		return new Variant(buffer);
	}
	
	/**
	 * Load an argument as an integer array
	 */
	private static Variant interpretAsIntegers(MLArray rawData) throws MLIOException
	{
		int[] buffer = null;
		switch(rawData.getType())
		{
			case INT8  : buffer = Array1DUtil.byteArrayToIntArray (rawData.getAsInt8Array (), true); break;
			case INT16 : buffer = Array1DUtil.shortArrayToIntArray(rawData.getAsInt16Array(), true); break;
			case UINT8 : buffer = Array1DUtil.byteArrayToIntArray (rawData.getAsUInt8Array (), false); break;
			case UINT16: buffer = Array1DUtil.shortArrayToIntArray(rawData.getAsUInt16Array(), false); break;
			default    : buffer = rawData.getAsInt32Array(); break;
		}
		return new Variant(buffer);
	}
	
	/**
	 * Load an argument as a decimal
	 */
	private static Variant interpretAsDecimal(MLArray rawData) throws MLIOException
	{
		double buffer = 0;
		switch(rawData.getType())
		{
			case INT8  : buffer = rawData.getAsInt8 (); break;
			case INT16 : buffer = rawData.getAsInt16(); break;
			case INT32 : buffer = rawData.getAsInt32(); break;
			case INT64 : buffer = rawData.getAsInt64(); break;
			case UINT8 : buffer = TypeUtil.unsign(rawData.getAsUInt8 ()); break;
			case UINT16: buffer = TypeUtil.unsign(rawData.getAsUInt16()); break;
			case UINT32: buffer = TypeUtil.unsign(rawData.getAsUInt32()); break;
			case UINT64: buffer = TypeUtil.unsign(rawData.getAsUInt64()); break;
			case SINGLE: buffer = rawData.getAsSingle(); break;
			default    : buffer = rawData.getAsDouble(); break;
		}
		return new Variant(buffer);
	}
	
	/**
	 * Load an argument as a decimal array
	 */
	private static Variant interpretAsDecimals(MLArray rawData) throws MLIOException
	{
		double[] buffer = null;
		switch(rawData.getType())
		{
			case INT8  : buffer = Array1DUtil.byteArrayToDoubleArray (rawData.getAsInt8Array  (), true ); break;
			case INT16 : buffer = Array1DUtil.shortArrayToDoubleArray(rawData.getAsInt16Array (), true ); break;
			case INT32 : buffer = Array1DUtil.intArrayToDoubleArray  (rawData.getAsInt32Array (), true ); break;
			case INT64 : buffer = Array1DUtil.longArrayToDoubleArray (rawData.getAsInt64Array (), true ); break;
			case UINT8 : buffer = Array1DUtil.byteArrayToDoubleArray (rawData.getAsUInt8Array (), false); break;
			case UINT16: buffer = Array1DUtil.shortArrayToDoubleArray(rawData.getAsUInt16Array(), false); break;
			case UINT32: buffer = Array1DUtil.intArrayToDoubleArray  (rawData.getAsUInt32Array(), false); break;
			case UINT64: buffer = Array1DUtil.longArrayToDoubleArray (rawData.getAsUInt64Array(), false); break;
			case SINGLE: buffer = Array1DUtil.floatArrayToDoubleArray(rawData.getAsSingleArray()); break;
			default    : buffer = rawData.getAsDoubleArray(); break;
		}
		return new Variant(buffer);
	}
	
	/**
	 * Load an argument as a text
	 */
	private static Variant interpretAsText(MLArray rawData) throws MLIOException
	{
		return new Variant(rawData.getAsString());
	}
	
	/**
	 * Load an argument as a sequence
	 */
	private Variant interpretAsSequence(String name) throws IOException
	{
		MatlabImporter importer = new MatlabImporter(_reader);
		Sequence buffer = importer.getSequence(name, getDimensionMapping(), getComplexModeForOutput());
		return new Variant(buffer);
	}
}
