package plugins.adufour.ctc;

import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Document;

import icy.file.FileUtil;
import icy.file.Loader;
import icy.file.Saver;
import icy.image.IcyBufferedImage;
import icy.main.Icy;
import icy.network.NetworkUtil;
import icy.plugin.abstract_.PluginActionable;
import icy.roi.ROI;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.type.DataType;
import icy.util.StringUtil;
import icy.util.XMLUtil;
import plugins.adufour.activecontours.ActiveContour;
import plugins.adufour.activecontours.ActiveContours;
import plugins.adufour.blocks.lang.WorkFlow;
import plugins.adufour.blocks.util.BlocksML;
import plugins.adufour.connectedcomponents.ConnectedComponent;
import plugins.adufour.filtering.ConvolutionException;
import plugins.adufour.filtering.Median;
import plugins.adufour.hierarchicalkmeans.HierarchicalKMeans;
import plugins.adufour.protocols.Protocols;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackSegment;

public class CellTrackingChallenge2015 extends PluginActionable
{
    @Override
    public void run()
    {
        if (!Icy.getMainInterface().isHeadLess())
        {
            // Can't run the command-line from here
            // => load up the protocol instead...
            
            try
            {
                String filePath = Protocols.downloadedProtocolFolder + File.separator + getClass().getSimpleName() + ".protocol";
                FileOutputStream stream = new FileOutputStream(filePath);
                stream.write(NetworkUtil.download("http://icy.bioimageanalysis.org/repository/getProtocolFile?protocolId=55&version=1", null, false));
                stream.close();
                Protocols.loadWorkFlow(new File(filePath));
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
            
            // ...and exit nicely
            return;
        }
        
        // command line
        String[] args = Icy.getCommandLinePluginArgs();
        
        String inputFolder = args[0] + File.separator + args[1];
        String outputFolder = inputFolder + "_RES";
        String protocolFile = args[0].substring(args[0].lastIndexOf(File.separator) + 1) + "-" + args[1] + ".protocol";
        
        Document xml = XMLUtil.loadDocument(protocolFile);
        
        // Discard invalid files
        if (xml == null)
        {
            System.err.println(protocolFile + " is not a XML file or is invalid");
            System.exit(-1);
        }
        
        System.out.println("Loading workflow...");
        WorkFlow workFlow = new WorkFlow();
        BlocksML.getInstance().loadWorkFlow(xml, workFlow);
        
        // the first block should be the CTCInput
        CTCInput input = (CTCInput) workFlow.getBlock(0).getBlock();
        input.dataFolder.setValue(new File(inputFolder));
        input.resultFolder.setValue(new File(outputFolder));
        
        System.out.println("Running workflow...");
        workFlow.run();
    }
    
    @SuppressWarnings({ "unused", "deprecation" })
    private void runISBI2014()
    {
        long tic = System.nanoTime();
        
        System.out.println("ISBI 2014 Tracking Challenge");
        
        String[] args = Icy.getCommandLinePluginArgs();
        
        System.out.println("Loading data set from " + args[0]);
        String folderIN = args[0];
        
        // Sequence seqIN = Loader.loadSequence(folderIN, 0, false);
        
        String[] files = FileUtil.getFiles(folderIN, new FileFilter()
        {
            @Override
            public boolean accept(File arg0)
            {
                return arg0.getPath().toLowerCase().endsWith("tif");
            }
        }, false, false, false);
        
        // 1) Load each file into a sequence
        // 2) Convert each sequence to a stack (if necessary)
        // 3) Concatenate all into one sequence
        
        Sequence[] frames = new Sequence[files.length];
        for (int i = 0; i < files.length; i++)
        {
            Sequence frame = Loader.loadSequence(files[i], 0, false);
            if (frame.getSizeT() > 0) SequenceUtil.convertToStack(frame);
            frames[i] = frame;
        }
        Sequence seqIN = SequenceUtil.concatT(frames);
        
        boolean is3D = false;
        
        if (folderIN.contains("H157"))
        {
            if (folderIN.endsWith("01"))
            {
                int slices = 35;
                is3D = true;
                SequenceUtil.adjustZT(seqIN, slices, seqIN.getSizeT() / slices, false);
            }
            else if (folderIN.endsWith("02"))
            {
                int slices = 80;
                is3D = true;
                SequenceUtil.adjustZT(seqIN, slices, seqIN.getSizeT() / slices, false);
            }
            else
            {
                System.out.println("Error: unknown dataset: " + folderIN);
            }
        }
        else if (folderIN.contains("N3DH"))
            
            System.out.println("Filtering input data...");
        
        if (is3D)
        {
            // 3D
            // don't filter for now, looks good enough
            // seqIN = new Median().filterSquare(seqIN, 1, 1);
        }
        else
        {
            // 2D
            seqIN = new Median().filterSquare(seqIN, 2, 2);
        }
        
        if (!Icy.getMainInterface().isHeadLess()) addSequence(seqIN);
        
        System.out.println("Loading tracker...");
        ActiveContours tracker = new ActiveContours();
        tracker.initialize();
        System.out.println("Parameter file : " + args[2]);
        File f = new File(args[2]);
        tracker.loadParameters(f);
        
        ArrayList<ROI> rois = new ArrayList<ROI>();
        
        try
        {
            Sequence t0 = SequenceUtil.extractFrame(seqIN, 0);
            
            System.out.println("Pre-detecting objects...");
            
            int minSize = is3D ? 10000 : 100;
            int maxSize = is3D ? 1000000 : 3000;
            int nbClasses = is3D ? 3 : 10;
            
            Map<Integer, List<ConnectedComponent>> map = HierarchicalKMeans.hierarchicalKMeans(t0, 4, nbClasses, minSize, maxSize, (Sequence) null);
            // there is only one time point.
            rois.ensureCapacity(map.get(0).size());
            
            // convert connected components to ROI
            for (ConnectedComponent cc : map.get(0))
                rois.add(cc.toROI());
        }
        catch (ConvolutionException e1)
        {
            System.out.println("Error filtering the data. Exiting...");
            System.exit(1);
        }
        
        System.out.println("Tracking " + rois.size() + " ROI");
        
        tracker.input.setValue(seqIN);
        tracker.roiInput.setValue(rois.toArray(new ROI[0]));
        
        tracker.execute();
        
        // 3) results
        
        TrackGroup tg = tracker.getTrackGroup();
        
        String exportFolder = args[1] + File.separator;
        
        System.out.println("Export folder: " + exportFolder);
        
        System.out.println("Exporting tracks...");
        
        Sequence binSeq = new Sequence("Labeled output");
        
        for (int t = 0; t < seqIN.getSizeT(); t++)
        {
            binSeq.setImage(t, 0, new IcyBufferedImage(seqIN.getSizeX(), seqIN.getSizeY(), 1, DataType.USHORT));
        }
        
        try
        {
            File resFile = FileUtil.createFile(exportFolder + "res_track.txt");
            FileWriter w = new FileWriter(resFile);
            
            ArrayList<TrackSegment> tracks = tg.getTrackSegmentList();
            
            for (TrackSegment segment : tracks)
            {
                // id must be 1-based (0 is not used)
                int id = tracks.indexOf(segment) + 1;
                
                if (segment.getDetectionList().size() == 0) continue;
                
                // first frame of the segment
                int start = segment.getFirstDetection().getT();
                
                // last frame of the segment
                int end = segment.getLastDetection().getT();
                
                // export the labeled object
                for (int t = start; t <= end; t++)
                {
                    ActiveContour contour = (ActiveContour) segment.getDetectionAtTime(t);
                    contour.toSequence(binSeq, id);
                }
                
                // get the (unique) parent id
                
                // @Fabrice: why no getter for this field??
                Field field = TrackSegment.class.getDeclaredField("previousList");
                field.setAccessible(true);
                @SuppressWarnings("unchecked")
                ArrayList<TrackSegment> parents = (ArrayList<TrackSegment>) field.get(segment);
                // list should only have zero or one...
                int parent = parents.size() == 0 ? 0 : tracks.indexOf(parents.get(0)) + 1;
                
                w.write(id + " " + start + " " + end + " " + parent + "\n");
            }
            
            System.out.println("Exporting labels...");
            for (int t = 0; t < binSeq.getSizeT(); t++)
            {
                File imageFile = FileUtil.createFile(exportFolder + "mask" + StringUtil.toString(t, 3) + ".tif");
                Saver.save(SequenceUtil.extractFrame(binSeq, t), imageFile, false, false);
            }
            
            w.flush();
            w.close();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
        catch (SecurityException e)
        {
            e.printStackTrace();
        }
        catch (NoSuchFieldException e)
        {
            e.printStackTrace();
        }
        catch (IllegalArgumentException e)
        {
            e.printStackTrace();
        }
        catch (IllegalAccessException e)
        {
            e.printStackTrace();
        }
        
        long tac = System.nanoTime();
        long time = (tac - tic) / 1000000;
        
        System.out.println("Full process executed in " + time + "ms");
        
        System.out.println("Au revoir.");
        System.exit(0);
    }
}
