/**
 * 
 */
package plugins.adufour.trackprocessors.speed;

import icy.math.ArrayMath;
import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginBundled;
import icy.sequence.Sequence;
import icy.util.XLSUtil;

import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.math.plot.Plot2DPanel;
import org.math.plot.Plot3DPanel;
import org.math.plot.PlotPanel;

import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import jxl.write.WriteException;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.lang.Var;
import plugins.adufour.vars.lang.VarBoolean;
import plugins.adufour.vars.lang.VarMutable;
import plugins.adufour.vars.lang.VarSequence;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.spot.Detection;

/**
 * @author Stephane
 */
public class MotionProfiler extends Plugin implements Block, PluginBundled
{
    private static final int COL_TRACK_GP = 0;
    private static final int COL_TRACK_ID = 1;
    private static final int COL_TRACK_START = 2;
    private static final int COL_TRACK_END = 3;
    private static final int COL_DURATION = 4;
    private static final int COL_TOT_DISP = 5;
    private static final int COL_NET_DISP = 6;
    private static final int COL_LINEARITY = 7;
    private static final int COL_RADIUS = 8;
    private static final int COL_MIN_DISP = 9;
    private static final int COL_MAX_DISP = 10;
    private static final int COL_AVG_DISP = 11;

    // VAR
    public final Var<TrackGroup> tracks;
    public final VarSequence sequence;
    public final VarBoolean displaySpeed;
    public final VarMutable outputFile;

    public MotionProfiler()
    {
        super();

        tracks = new Var<>("Track group", new TrackGroup(null));
        sequence = new VarSequence("Sequence", null);
        displaySpeed = new VarBoolean("Show speed instead of disp.", false);
        outputFile = new VarMutable("Output file", null)
        {
            @Override
            public boolean isAssignableFrom(@SuppressWarnings("rawtypes") Var source)
            {
                return (String.class == source.getType()) || (File.class == source.getType());
            }
        };
    }

    /**
     * Export motion data in XLS format from the given list of {@link TrackSegment}.
     * 
     * @param tracks
     *        tracks we want to export data fort
     * @param sequence
     *        sequence used to retrieve real units from (using pixel size information from metadata).<br>
     *        Keep this value to <code>null</code> if you want use raw pixel / frame units instead.
     * @param displaySpeed
     *        display speed information instead of displacement
     * @param outputPath
     *        XLS output path (should contains <i>.xls</i> extension)
     * @param xPlotPanel
     *        optional X plot panel where we will draw the X motion
     * @param yPlotPanel
     *        optional Y plot panel where we will draw the Y motion
     * @param zPlotPanel
     *        optional Z plot panel where we will draw the Z motion
     * @param globalPlotPanel
     *        optional global plot panel where we will draw all the tracks motion
     */
    public static void exportTrackMotionToXLS(List<TrackSegment> tracks, Sequence sequence, boolean displaySpeed,
            String outputPath, Plot2DPanel xPlotPanel, Plot2DPanel yPlotPanel, Plot2DPanel zPlotPanel,
            PlotPanel globalPlotPanel) throws IOException, WriteException
    {
        final double xScale;
        final double yScale;
        final double zScale;
        final double tScale;

        if (sequence != null)
        {
            xScale = sequence.getPixelSizeX();
            yScale = sequence.getPixelSizeY();
            zScale = sequence.getPixelSizeZ();
            tScale = sequence.getTimeInterval();
        }
        else
        {
            xScale = 1d;
            yScale = 1d;
            zScale = 1d;
            tScale = 1d;
        }

        // if we have plot wanted, take care of cleaning them first
        if (xPlotPanel != null)
            xPlotPanel.removeAllPlots();
        if (yPlotPanel != null)
            yPlotPanel.removeAllPlots();
        if (zPlotPanel != null)
            zPlotPanel.removeAllPlots();
        if (globalPlotPanel != null)
            globalPlotPanel.removeAllPlots();

        // create the workbook
        final WritableWorkbook book = XLSUtil.createWorkbook(outputPath);

        // create the sheets
        final WritableSheet globalSheet = XLSUtil.createNewPage(book, "Global");
        final WritableSheet xSheet = XLSUtil.createNewPage(book, "X");
        final WritableSheet ySheet = XLSUtil.createNewPage(book, "Y");
        final WritableSheet zSheet = XLSUtil.createNewPage(book, "Z");

        // write headers
        final List<WritableSheet> sheets = new ArrayList<>(4);

        sheets.add(globalSheet);
        sheets.add(xSheet);
        sheets.add(ySheet);
        sheets.add(zSheet);

        final String timeUnit = (sequence != null) ? "sec" : "frames";
        final String spaceUnit = (sequence != null) ? "\u03BCm" : "px";
        final String speedUnit = displaySpeed ? "speed (" + spaceUnit + "/" + timeUnit + ")"
                : "disp. (" + spaceUnit + ")";

        // write headers
        for (WritableSheet sheet : sheets)
        {
            XLSUtil.setCellString(sheet, COL_TRACK_GP, 0, "Group");
            XLSUtil.setCellString(sheet, COL_TRACK_ID, 0, "Track #");
            XLSUtil.setCellString(sheet, COL_TRACK_START, 0, "Start (" + timeUnit + ")");
            XLSUtil.setCellString(sheet, COL_TRACK_END, 0, "End (" + timeUnit + ")");
            XLSUtil.setCellString(sheet, COL_DURATION, 0, "Duration (" + timeUnit + ")");
            XLSUtil.setCellString(sheet, COL_TOT_DISP, 0, "Total disp. (" + spaceUnit + ")");
            XLSUtil.setCellString(sheet, COL_NET_DISP, 0, "Net disp. (" + spaceUnit + ")");
            XLSUtil.setCellString(sheet, COL_LINEARITY, 0, "Linearity" + " (%)");
            XLSUtil.setCellString(sheet, COL_RADIUS, 0, "Search radius" + " (" + spaceUnit + ")");
            XLSUtil.setCellString(sheet, COL_MIN_DISP, 0, "Min. " + speedUnit);
            XLSUtil.setCellString(sheet, COL_MAX_DISP, 0, "Max. " + speedUnit);
            XLSUtil.setCellString(sheet, COL_AVG_DISP, 0, "Avg. " + speedUnit);
        }

        int row = 1;
        for (TrackSegment ts : tracks)
        {
            // check if the track is filtered out
            if (!ts.isAllDetectionEnabled())
                continue;

            final int nbDetections = ts.getDetectionList().size();

            // we need at least 2 detections to do a track
            if (nbDetections < 2)
                continue;

            // extract the raw track data
            final double[] xData = new double[nbDetections];
            final double[] yData = new double[nbDetections];
            final double[] zData = new double[nbDetections];

            // displacement data (size: N-1)
            final double[] disp = new double[nbDetections - 1];
            final double[] xDisp = new double[nbDetections - 1];
            final double[] yDisp = new double[nbDetections - 1];
            final double[] zDisp = new double[nbDetections - 1];

            // store minimum and maximum position to calculate the track extent
            double searchRadius = 0, xSearchRadius = 0, ySearchRadius = 0, zSearchRadius = 0;

            // accumulate data here
            for (int i = 0; i < nbDetections; i++)
            {
                final Detection det = ts.getDetectionAt(i);

                xData[i] = det.getX() * xScale;
                yData[i] = det.getY() * yScale;
                zData[i] = det.getZ() * zScale;

                if (i > 0)
                {
                    // retrieve displacement from previous position
                    xDisp[i - 1] = xData[i] - xData[i - 1];
                    yDisp[i - 1] = yData[i] - yData[i - 1];
                    zDisp[i - 1] = zData[i] - zData[i - 1];
                    disp[i - 1] = Math.sqrt(
                            xDisp[i - 1] * xDisp[i - 1] + yDisp[i - 1] * yDisp[i - 1] + zDisp[i - 1] * zDisp[i - 1]);
                }
            }

            // compute parameters here

            // track duration

            // track start/end
            double startTime = ts.getFirstDetection().getT() * tScale;
            double endTime = ts.getLastDetection().getT() * tScale;
            double duration = (nbDetections - 1) * tScale;

            // relative displacement (start to end)
            double xNetDisp = Math.abs(xData[xData.length - 1] - xData[0]);
            double yNetDisp = Math.abs(yData[yData.length - 1] - yData[0]);
            double zNetDisp = Math.abs(zData[zData.length - 1] - zData[0]);
            double netDisp = Math.sqrt(xNetDisp * xNetDisp + yNetDisp * yNetDisp + zNetDisp * zNetDisp);

            // total (sum of) displacement(s)
            double xTotDisp = ArrayMath.sum(ArrayMath.abs(xDisp, false));
            double yTotDisp = ArrayMath.sum(ArrayMath.abs(yDisp, false));
            double zTotDisp = ArrayMath.sum(ArrayMath.abs(zDisp, false));
            double totDisp = ArrayMath.sum(ArrayMath.abs(disp, false));

            // linearity (ratio of total to net displacement)
            double xTortuosity = (xTotDisp == 0d) ? 0 : xNetDisp / xTotDisp;
            double yTortuosity = (yTotDisp == 0d) ? 0 : yNetDisp / yTotDisp;
            double zTortuosity = (zTotDisp == 0d) ? 0 : zNetDisp / zTotDisp;
            double tortuosity = (totDisp == 0d) ? 0 : netDisp / totDisp;

            // extent (largest distance between any 2 detections)
            for (int i = 0; i < nbDetections; i++)
            {
                for (int j = i + 1; j < nbDetections; j++)
                {
                    double dx = xData[i] - xData[j];
                    double dy = yData[i] - yData[j];
                    double dz = zData[i] - zData[j];
                    double dist = Math.sqrt(dx * dx + dy * dy + dz * dz);

                    if (dist > searchRadius)
                    {
                        searchRadius = dist;
                        xSearchRadius = Math.abs(dx);
                        ySearchRadius = Math.abs(dy);
                        zSearchRadius = Math.abs(dz);
                    }
                }
            }

            // prefer speed instead of displacement ? --> divide by time then...
            if (displaySpeed && (tScale != 0d))
            {
                ArrayMath.divide(disp, tScale, disp);
                ArrayMath.divide(xDisp, tScale, xDisp);
                ArrayMath.divide(yDisp, tScale, yDisp);
                ArrayMath.divide(zDisp, tScale, zDisp);
            }

            double xMinDisp = ArrayMath.min(xDisp);
            double yMinDisp = ArrayMath.min(yDisp);
            double zMinDisp = ArrayMath.min(zDisp);
            double minDisp = ArrayMath.min(disp);

            double xMaxDisp = ArrayMath.max(xDisp);
            double yMaxDisp = ArrayMath.max(yDisp);
            double zMaxDisp = ArrayMath.max(zDisp);
            double maxDisp = ArrayMath.max(disp);

            double xAvgDisp = ArrayMath.mean(xDisp);
            double yAvgDisp = ArrayMath.mean(yDisp);
            double zAvgDisp = ArrayMath.mean(zDisp);
            double avgDisp = ArrayMath.mean(disp);

            // center all tracks from plotting purposes
            ArrayMath.subtract(xData, xData[0], xData);
            ArrayMath.subtract(yData, yData[0], yData);
            ArrayMath.subtract(zData, zData[0], zData);

            // plotting
            final Color color = ts.getFirstDetection().getColor();
            final TrackGroup group = ts.getOwnerTrackGroup();

            final String groupName;
            final int trackIndex;

            if (group != null)
            {
                // group name
                groupName = group.getDescription();
                // track index
                trackIndex = group.getTrackSegmentList().indexOf(ts);
            }
            else
            {
                if (sequence != null)
                    groupName = sequence.getName();
                else
                    groupName = "TrackGroup";
                // track index
                trackIndex = row - 1;
            }

            // track name
            final String trackName = groupName + " - Track " + trackIndex;

            if (xPlotPanel != null)
                xPlotPanel.addLinePlot(trackName, color, xDisp);
            if (yPlotPanel != null)
                yPlotPanel.addLinePlot(trackName, color, yDisp);
            if (zPlotPanel != null)
                zPlotPanel.addLinePlot(trackName, color, zDisp);
            if (globalPlotPanel instanceof Plot3DPanel)
                ((Plot3DPanel) globalPlotPanel).addLinePlot(trackName, color, xData, yData, zData);
            else if (globalPlotPanel instanceof Plot2DPanel)
                ((Plot2DPanel) globalPlotPanel).addLinePlot(trackName, color, xData, yData);

            // store in the spreadsheet
            XLSUtil.setCellString(globalSheet, COL_TRACK_GP, row, groupName);
            XLSUtil.setCellString(xSheet, COL_TRACK_GP, row, groupName);
            XLSUtil.setCellString(ySheet, COL_TRACK_GP, row, groupName);
            XLSUtil.setCellString(zSheet, COL_TRACK_GP, row, groupName);

            XLSUtil.setCellNumber(globalSheet, COL_TRACK_ID, row, trackIndex);
            XLSUtil.setCellNumber(xSheet, COL_TRACK_ID, row, trackIndex);
            XLSUtil.setCellNumber(ySheet, COL_TRACK_ID, row, trackIndex);
            XLSUtil.setCellNumber(zSheet, COL_TRACK_ID, row, trackIndex);

            XLSUtil.setCellNumber(globalSheet, COL_TRACK_START, row, startTime);
            XLSUtil.setCellNumber(xSheet, COL_TRACK_START, row, startTime);
            XLSUtil.setCellNumber(ySheet, COL_TRACK_START, row, startTime);
            XLSUtil.setCellNumber(zSheet, COL_TRACK_START, row, startTime);

            XLSUtil.setCellNumber(globalSheet, COL_TRACK_END, row, endTime);
            XLSUtil.setCellNumber(xSheet, COL_TRACK_END, row, endTime);
            XLSUtil.setCellNumber(ySheet, COL_TRACK_END, row, endTime);
            XLSUtil.setCellNumber(zSheet, COL_TRACK_END, row, endTime);

            XLSUtil.setCellNumber(globalSheet, COL_DURATION, row, duration);
            XLSUtil.setCellNumber(xSheet, COL_DURATION, row, duration);
            XLSUtil.setCellNumber(ySheet, COL_DURATION, row, duration);
            XLSUtil.setCellNumber(zSheet, COL_DURATION, row, duration);

            XLSUtil.setCellNumber(globalSheet, COL_TOT_DISP, row, totDisp);
            XLSUtil.setCellNumber(xSheet, COL_TOT_DISP, row, xTotDisp);
            XLSUtil.setCellNumber(ySheet, COL_TOT_DISP, row, yTotDisp);
            XLSUtil.setCellNumber(zSheet, COL_TOT_DISP, row, zTotDisp);

            XLSUtil.setCellNumber(globalSheet, COL_NET_DISP, row, netDisp);
            XLSUtil.setCellNumber(xSheet, COL_NET_DISP, row, xNetDisp);
            XLSUtil.setCellNumber(ySheet, COL_NET_DISP, row, yNetDisp);
            XLSUtil.setCellNumber(zSheet, COL_NET_DISP, row, zNetDisp);

            XLSUtil.setCellNumber(globalSheet, COL_LINEARITY, row, tortuosity);
            XLSUtil.setCellNumber(xSheet, COL_LINEARITY, row, xTortuosity);
            XLSUtil.setCellNumber(ySheet, COL_LINEARITY, row, yTortuosity);
            XLSUtil.setCellNumber(zSheet, COL_LINEARITY, row, zTortuosity);

            XLSUtil.setCellNumber(globalSheet, COL_RADIUS, row, searchRadius);
            XLSUtil.setCellNumber(xSheet, COL_RADIUS, row, xSearchRadius);
            XLSUtil.setCellNumber(ySheet, COL_RADIUS, row, ySearchRadius);
            XLSUtil.setCellNumber(zSheet, COL_RADIUS, row, zSearchRadius);

            XLSUtil.setCellNumber(globalSheet, COL_MIN_DISP, row, minDisp);
            XLSUtil.setCellNumber(xSheet, COL_MIN_DISP, row, xMinDisp);
            XLSUtil.setCellNumber(ySheet, COL_MIN_DISP, row, yMinDisp);
            XLSUtil.setCellNumber(zSheet, COL_MIN_DISP, row, zMinDisp);

            XLSUtil.setCellNumber(globalSheet, COL_MAX_DISP, row, maxDisp);
            XLSUtil.setCellNumber(xSheet, COL_MAX_DISP, row, xMaxDisp);
            XLSUtil.setCellNumber(ySheet, COL_MAX_DISP, row, yMaxDisp);
            XLSUtil.setCellNumber(zSheet, COL_MAX_DISP, row, zMaxDisp);

            XLSUtil.setCellNumber(globalSheet, COL_AVG_DISP, row, avgDisp);
            XLSUtil.setCellNumber(xSheet, COL_AVG_DISP, row, xAvgDisp);
            XLSUtil.setCellNumber(ySheet, COL_AVG_DISP, row, yAvgDisp);
            XLSUtil.setCellNumber(zSheet, COL_AVG_DISP, row, zAvgDisp);

            row++;
        }

        XLSUtil.saveAndClose(book);
    }

    /**
     * Export motion data in XLS format from the given list of {@link TrackSegment}.
     * 
     * @param tracks
     *        tracks we want to export data fort
     * @param sequence
     *        sequence used to retrieve real units from (using pixel size information from metadata).<br>
     *        Keep this value to <code>null</code> if you want use raw pixel / frame units instead.
     * @param displaySpeed
     *        display speed information instead of displacement
     * @param outputPath
     *        XLS output path (should contains <i>.xls</i> extension)
     */
    public static void exportTrackMotionToXLS(List<TrackSegment> tracks, Sequence sequence, boolean displaySpeed,
            String outputPath) throws WriteException, IOException
    {
        exportTrackMotionToXLS(tracks, sequence, displaySpeed, outputPath, null, null, null, null);
    }

    @Override
    public void declareInput(VarList inputMap)
    {
        inputMap.add("trackgroup", tracks);
        inputMap.add("sequence", sequence);
        inputMap.add("displaySpeed", displaySpeed);
        inputMap.add("output", outputFile);
    }

    @Override
    public void declareOutput(VarList outputMap)
    {
        // nothing here
    }

    @Override
    public void run()
    {
        // execution from protocol
        final TrackGroup trackGroup = tracks.getValue();
        final Object obj = outputFile.getValue();

        if ((trackGroup != null) && (obj != null))
        {
            // build the list of enabled tracks
            final List<TrackSegment> tracksToExport = new ArrayList<>();

            for (TrackSegment track : trackGroup.getTrackSegmentList())
                if (track.isAllDetectionEnabled())
                    tracksToExport.add(track);

            final File f;

            if (obj instanceof String)
                f = new File((String) obj);
            else
                f = (File) obj;

            try
            {
                // export track motion info
                exportTrackMotionToXLS(tracksToExport, sequence.getValue(), displaySpeed.getValue().booleanValue(),
                        f.getAbsolutePath());
            }
            catch (Exception e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public String getMainPluginClassName()
    {
        return SpeedProfiler.class.getName();
    }
}
