/**
 * 
 */
package plugins.stef.roi.bloc.op;

import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginBundled;
import icy.plugin.interface_.PluginLibrary;
import icy.roi.ROI;
import icy.roi.ROIDescriptor;
import icy.sequence.Sequence;
import icy.system.IcyExceptionHandler;
import icy.util.StringUtil;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import plugins.adufour.blocks.tools.roi.ROIBlock;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.gui.model.ValueSelectionModel;
import plugins.adufour.vars.lang.VarDouble;
import plugins.adufour.vars.lang.VarEnum;
import plugins.adufour.vars.lang.VarROIArray;
import plugins.adufour.vars.lang.VarSequence;
import plugins.adufour.vars.lang.VarString;
import plugins.adufour.vars.util.VarException;
import plugins.kernel.roi.descriptor.measure.ROIInteriorDescriptor;
import plugins.stef.roi.bloc.RoiBlocks;

/**
 * Block to filter a set of ROIs using a specific descriptor.
 * 
 * @author Stephane
 */
public class FilterROI extends Plugin implements ROIBlock, PluginLibrary, PluginBundled
{
    public static enum CompareOperator
    {
        EQUAL
        {
            @Override
            public String toString()
            {
                return "Equal to";
            }
        },
        NOT_EQUAL
        {
            @Override
            public String toString()
            {
                return "Not equal to";
            }
        },
        LOWER
        {
            @Override
            public String toString()
            {
                return "Lower than";
            }
        },
        LOWER_OR_EQUAL
        {
            @Override
            public String toString()
            {
                return "Lower or equal than";
            }
        },
        GREATER
        {
            @Override
            public String toString()
            {
                return "Greater than";
            }
        },
        GREATER_OR_EQUAL
        {
            @Override
            public String toString()
            {
                return "Greater or equal than";
            }
        }
    }

    final protected VarROIArray roiSet = new VarROIArray("ROI(s)", null);
    final protected VarSequence sequence = new VarSequence("Sequence", null);
    final protected VarString descriptors = new VarString("Filter on", "");
    final protected VarEnum<CompareOperator> operator = new VarEnum<CompareOperator>("Accept if",
            CompareOperator.EQUAL);
    final protected VarDouble value = new VarDouble("Value", 0);
    final protected VarROIArray output = new VarROIArray("Filtered ROI(s)");

    @Override
    public void run()
    {
        final List<ROI> result = new ArrayList<ROI>();
        final Sequence seq = sequence.getValue();
        final double v = value.getValue().doubleValue();
        final CompareOperator op = operator.getValue();
        final String descriptorId = descriptors.getValue();
        final ROIDescriptor roiDescriptor = ROIDescriptor.getDescriptor(ROIDescriptor.getDescriptors().keySet(),
                descriptorId);

        if (roiDescriptor == null)
            throw new VarException(descriptors, "Cannot found '" + descriptorId + "' ROI descriptor !");

        for (ROI roi : roiSet.getValue())
        {
            if (roi != null)
            {
                try
                {
                    final Object res = roiDescriptor.compute(roi, seq);
                    boolean accepted;

                    // we only support Number result for filtering
                    if (res instanceof Number)
                    {
                        final double computedValue = ((Number) res).doubleValue();

                        switch (op)
                        {
                            default:
                            case EQUAL:
                                accepted = computedValue == v;
                                break;
                            case NOT_EQUAL:
                                accepted = computedValue != v;
                                break;
                            case LOWER:
                                accepted = computedValue < v;
                                break;
                            case LOWER_OR_EQUAL:
                                accepted = computedValue <= v;
                                break;
                            case GREATER:
                                accepted = computedValue > v;
                                break;
                            case GREATER_OR_EQUAL:
                                accepted = computedValue >= v;
                                break;
                        }
                    }
                    else
                        accepted = true;

                    if (accepted)
                        result.add(roi);
                }
                catch (Exception e)
                {
                    IcyExceptionHandler.showErrorMessage(e, false, true);
                    // if we can't compute the descriptor we accept the ROI by default...
                    result.add(roi);
                }
            }
        }

        output.setValue(result.toArray(new ROI[result.size()]));
    }

    @Override
    public void declareInput(VarList inputMap)
    {
        final List<ROIDescriptor> roiDescriptors = new ArrayList<ROIDescriptor>(
                ROIDescriptor.getDescriptors().keySet());
        // build list of descriptor id
        final List<String> descriptorsId = new ArrayList<String>();
        String sizeDescriptorId = null;

        for (ROIDescriptor descriptor : roiDescriptors)
        {
            final String id = descriptor.getId();

            // keep trace of size descriptor
            if (StringUtil.equals(ROIInteriorDescriptor.ID, id))
                sizeDescriptorId = id;

            descriptorsId.add(id);
        }

        // alpha sort
        Collections.sort(descriptorsId);

        // initialize descriptors field
        descriptors.setDefaultEditorModel(new ValueSelectionModel<String>(
                descriptorsId.toArray(new String[descriptorsId.size()]), sizeDescriptorId, false));

        inputMap.add("roi", roiSet);
        inputMap.add("sequence", sequence);
        inputMap.add("descriptors", descriptors);
        inputMap.add("operator", operator);
        inputMap.add("value", value);
    }

    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("out", output);
    }

    @Override
    public String getMainPluginClassName()
    {
        return RoiBlocks.class.getName();
    }
}