import sys # for standard streams manipulation, byteorder
import execnet # where all the hard work is done


# register remote stdout and stderr channels,
# so that prints and errors are printed in the Icy console
def setup_stdstreams(gateway):
    # send stdout and stderr channels to Icy,
    # so that prints and errors are printed in the Icy console    
    channel = gateway.remote_exec("""
    import sys
            
    outchan = channel.gateway.newchannel()
    sys.stdout = outchan.makefile("w")
    channel.send(outchan)
    
    errchan = channel.gateway.newchannel()
    sys.stderr = errchan.makefile("w")
    channel.send(errchan)
    """)
    
    # receive remote stdout and stderr as channels
    outchan = channel.receive()
    errchan = channel.receive()

    # associate the channels with printing callbacks
    # note: callbacks execute in receiver thread
    outchan.setcallback(lambda data: sys.stdout.write(str(data)))
    errchan.setcallback(lambda data: sys.stderr.write(str(data)))

    # wait for the remote script to be finished
    channel.waitclose()


# returns a Python object containing all the data needed to reconstruct a 2D image
# on the remote as a Numpy array
def pack_image(image):
    imageData = image.getDataXY(0)
    typecode = imageData.typecode

    # handle the fact that unsigned bytes do not exist natively in java
    if typecode == 'b' and image.getDataType_() == DataType.UBYTE:
        typecode = 'B'

    return [image.getSizeY(),
            image.getSizeX(),
            typecode,
            sys.byteorder,
            imageData.tostring()]


# returns an IcyBufferedImage built from a 2D Numpy array packaged on the remote side
def unpack_image(packed_image):
    from array import array
    from icy.image import IcyBufferedImage
    
    [rows, cols, typecode, byteorder, data] = packed_image
    
    a = array(typecode, data)

    # handle byteorder conversion
    if sys.byteorder <> byteorder:
        a.byteswap()

    # handle signedness
    if typecode in ['B', 'H', 'I', 'L']:
        signed = False
    else:
        signed = True
    
    return IcyBufferedImage(cols, rows, a, signed)


class IcyExecnetGateway():
    def __init__(self, python_path=None, gateway_spec=None, debug=False):
        self.debug = debug
        
        # use gateway_spec if it is provided
        if gateway_spec == None:
            # if gateway_spec is not provided, use popen
            # if python_path is provided, try to launch that interpreter
            # or launch the python that is the system path if python_path is not provided
            if python_path <> None:
                gateway_spec = "popen//python=" + python_path
            else:
                gateway_spec = "popen//python=python"
        
        self.gateway = execnet.makegateway(gateway_spec)
        
        self.channel = None
        
        self.setup_stdstreams()
        
        # send the numpyexecnet module so that it can be imported
        # on the remote

    # define __enter__ and __exit__ methods so that the class can be
    # used with the 'with' statement
    def __enter__(self):
        return self

    # Note: this waits for the previous remote execution to be finished
    # make sure the remote script properly terminates at some point
    def __exit__(self, type, value, traceback):
        if type==None and value==None and traceback==None:
            # This is the normal end of the 'with'-block, without any exception in-between.
            # Wait for the remote to properly end.
            self.cleanup()
        else:
            # An exception occurred on the local side in the 'with'-block.
            # Waiting for the remote to end will likely deadlock,
            # since the remote itself is properly waiting for data from the local side.
            # So do a forced exit of the remote.
            self.cleanup(force=True)

    # Note: this waits for the previous remote execution to be finished
    # make sure the remote script properly terminates at some point
    def cleanup(self, force=False):
        if self.channel <> None and not force:
            self.channel.waitclose() # make sure the remote process has finished
        self.gateway.exit() # close the remote interpreter
        execnet.default_group.terminate(timeout=1.) # robust termination
        
        if self.debug:
            print "cleaned up"

    # register remote stdout and stderr channels,
    # so that prints and errors are printed in the Icy console
    def setup_stdstreams(self):
        setup_stdstreams(self.gateway)

    # Note: this waits for the previous remote execution to be finished
    # make sure the remote script properly terminates at some point 
    def remote_exec(self, args):
        # wait for the previous channel to be closed
        # Note: this prevents simultaneous remote_exec() run 
        if self.channel <> None:
            self.channel.waitclose()
        self.channel = self.gateway.remote_exec(args)

    # send some data over the channel
    def send(self, args):
        self.channel.send(args)

    # receive some data over the channel
    def receive(self):
        return self.channel.receive()
