package plugins.adufour.hcs.data;

import java.io.IOException;
import java.util.Iterator;

import org.xml.sax.SAXException;

import icy.sequence.Sequence;
import ome.xml.meta.OMEXMLMetadata;
import plugins.adufour.hcs.io.WellPlateReader;

public class WellPlate
{
    private final WellPlateReader wellPlateReader;
    
    private final OMEXMLMetadata metadata;
    
    private final Sequence sequence;
    
    /**
     * Linearized 2D array containing all wells
     */
    private final Well[] wells;
    
    /**
     * @param reader
     *            the reader that creates this well plate (will be used to load individual fields)
     * @param metadata
     *            the well plate metadata containing the plate layout
     * @param shape
     *            the shape of the wells in this plate
     */
    public WellPlate(WellPlateReader reader, OMEXMLMetadata metadata, Well.Shape shape)
    {
        this.wellPlateReader = reader;
        this.metadata = metadata;
        this.sequence = new Sequence(metadata);
        
        // Initialize the wells (even if they contain no image)
        int nbRows = getNbRows();
        int nbCols = getNbCols();
        wells = new Well[nbRows * nbCols];
        for (int row = 0; row < nbRows; row++)
            for (int col = 0; col < nbCols; col++)
                wells[row * nbCols + col] = new Well(shape, row, col);
    }
    
    /**
     * @return The number of rows in this well plate
     */
    public int getNbRows()
    {
        return metadata.getPlateRows(0).getValue();
    }
    
    /**
     * @return The number of columns in this well plate
     */
    public int getNbCols()
    {
        return metadata.getPlateColumns(0).getValue();
    }
    
    public Well getWellAt(int row, int col)
    {
        return wells[row * getNbCols() + col];
    }
    
    public OMEXMLMetadata getMetaData()
    {
        return metadata;
    }
    
    @Override
    public String toString()
    {
        String plateID = metadata.getPlateID(0);
        String plateType = metadata.getPlateName(0);
        
        return plateID + " (" + plateType + ")";
    }
    
    public Sequence loadField(Field field) throws IOException, SAXException
    {
        wellPlateReader.loadField(field, sequence);
        if (field != null) sequence.setName(field.getWell().getAlphanumericID());
        return sequence;
    }
    
    /**
     * @return A well iterator that automatically skips empty wells
     */
    public Iterator<Well> wellIterator()
    {
        return wellIterator(null);
    }
    
    /**
     * @return A well iterator that automatically skips empty wells
     * @param filters
     *            one or more templates (separated by spaces) used to select only a subset of wells,
     *            e.g.:
     *            <ul>
     *            <li><code>"A"</code> will select all wells in row A</li>
     *            <li><code>"A 03 B12"</code> will select all wells in row A, column 03, and well
     *            B12</li>
     *            <li>etc.</li>
     *            </ul>
     */
    public Iterator<Well> wellIterator(final String filters)
    {
        return new Iterator<Well>()
        {
            int currentWellIndex = -1;
            int nextWellIndex = 0;
            Well nextWell = null;
            
            @Override
            public Well next()
            {
                currentWellIndex = nextWellIndex;
                return nextWell;
            }
            
            @Override
            public boolean hasNext()
            {
                for (int newWell = currentWellIndex + 1; newWell < wells.length; newWell++)
                {
                    Well candidateWell = wells[newWell];
                    
                    if (candidateWell.isEmpty()) continue;
                    
                    // the well is automatically valid if there are no filters
                    boolean isValidWell = (filters == null);
                    
                    if (filters != null)
                    {
                        for (String filter : filters.split("\\s+"))
                            if (candidateWell.getAlphanumericID().contains(filter))
                            {
                                isValidWell = true;
                                break;
                            }
                    }
                    
                    if (isValidWell)
                    {
                        nextWell = wells[newWell];
                        nextWellIndex = newWell;
                        return true;
                    }
                }
                return false;
            }
        };
    }
    
    /**
     * @return A field iterator that automatically skips empty wells
     */
    public Iterator<Field> fieldIterator()
    {
        return fieldIterator(null);
    }
    
    /**
     * @return A field iterator that automatically skips empty wells
     * @param templates
     *            one or more templates (separated by spaces) used to select only a subset of wells,
     *            e.g.:
     *            <ul>
     *            <li><code>"A"</code> will select all wells in row A</li>
     *            <li><code>"A 03 B12"</code> will select all wells in row A, column 03, and well
     *            B12</li>
     *            <li>etc.</li>
     *            </ul>
     */
    public Iterator<Field> fieldIterator(final String templates)
    {
        return new Iterator<Field>()
        {
            Iterator<Well> wellIterator = wellIterator(templates);
            Iterator<Field> currentFieldIterator = null;
            
            @Override
            public Field next()
            {
                return currentFieldIterator.next();
            }
            
            @Override
            public boolean hasNext()
            {
                // Are there more fields remaining?
                if (currentFieldIterator != null && currentFieldIterator.hasNext()) return true;
                
                // If not, a new well perhaps?
                if (wellIterator.hasNext())
                {
                    currentFieldIterator = wellIterator.next().fieldIterator();
                    return currentFieldIterator.hasNext();
                }
                
                return false;
            }
        };
    }
}