package plugins.ylemontag.ssim;

import icy.gui.frame.GenericFrame;
import icy.preferences.XMLPreferences;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JScrollPane;
import javax.swing.JTextPane;

import plugins.adufour.ezplug.EzButton;
import plugins.adufour.ezplug.EzComponent;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.vars.lang.Var;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Components used to define the SSIM parameters
 */
public class SSIMParameterComponent
{
	private EzVarDouble _K1;
	private EzVarDouble _K2;
	private EzVarDouble _L;
	private EzVarDouble _sigmaX;
	private EzVarDouble _sigmaY;
	private LinkedList<Var<?>> _vars;
	private LinkedList<EzComponent> _components;
	
	/**
	 * Base constructor
	 */
	public SSIMParameterComponent()
	{
		_vars = new LinkedList<Var<?>>();
		_components = new LinkedList<EzComponent>();
		
		// Components
		_L = new EzVarDouble("Dynamic range (L)");
		_K1 = new EzVarDouble("K1");
		_K2 = new EzVarDouble("K2");
		_sigmaX = new EzVarDouble("Sigma X");
		_sigmaY = new EzVarDouble("Sigma Y");
		_vars.add(_L     .getVariable());
		_vars.add(_K1    .getVariable());
		_vars.add(_K2    .getVariable());
		_vars.add(_sigmaX.getVariable());
		_vars.add(_sigmaY.getVariable());
		EzButton detailsButton = new EzButton("Information about SSIM", new ActionListener()
		{			
			@Override
			public void actionPerformed(ActionEvent e) {
				onDetailsClicked();
			}
		});
		_components.add(_L     );
		_components.add(_K1    );
		_components.add(_K2    );
		_components.add(_sigmaX);
		_components.add(_sigmaY);
		_components.add(detailsButton);
		
		// Default values
		_L .setValue(SSIMCalculator.DEFAULT_L );
		_K1.setValue(SSIMCalculator.DEFAULT_K1);
		_K2.setValue(SSIMCalculator.DEFAULT_K2);
		_sigmaX.setValue(SSIMCalculator.DEFAULT_GAUSSIAN_SIGMA);
		_sigmaY.setValue(SSIMCalculator.DEFAULT_GAUSSIAN_SIGMA);
		
		// Tooltips
		_L.setToolTipText("Dynamic range of the pixel values of the sequences "
			+ "(for example: might be 255 if the sequences are 8 bit encoded)");
		_K1.setToolTipText("First regularization constant (the smaller the better)");
		_K2.setToolTipText("Second regularization constant (the smaller the better)");
		_sigmaX.setToolTipText("Standard deviation along the X axis of the Gaussian window used to compute the SSIM");
		_sigmaY.setToolTipText("Standard deviation along the Y axis of the Gaussian window used to compute the SSIM");
	}
	
	/**
	 * List of variables
	 */
	public List<Var<?>> getVars()
	{
		return _vars;
	}
	
	/**
	 * List of components
	 */
	public List<EzComponent> getComponents()
	{
		return _components;
	}
	
	/**
	 * Load all the parameters from a XML preference object
	 */
	public void load(XMLPreferences prefs)
	{
		_L     .setValue(prefs.getDouble("L"     , _L     .getValue()));
		_K1    .setValue(prefs.getDouble("K1"    , _K1    .getValue()));
		_K2    .setValue(prefs.getDouble("K2"    , _K2    .getValue()));
		_sigmaX.setValue(prefs.getDouble("SigmaX", _sigmaX.getValue()));
		_sigmaY.setValue(prefs.getDouble("SigmaY", _sigmaY.getValue()));
	}
	
	/**
	 * Save the current parameters in a XML preference object
	 */
	public void save(XMLPreferences prefs)
	{
		prefs.putDouble("L"     , _L     .getValue());
		prefs.putDouble("K1"    , _K1    .getValue());
		prefs.putDouble("K2"    , _K2    .getValue());
		prefs.putDouble("SigmaX", _sigmaX.getValue());
		prefs.putDouble("SigmaY", _sigmaY.getValue());
	}
	
	/**
	 * Create a new SSIMCalculator with its parameters set to the value specified
	 * by the GUI components
	 * @throws IllegalSSIMParameterException if one of the parameter has an inconsistent value (example: L must be > 0)
	 */
	public SSIMCalculator createCalculator()
	{
		SSIMCalculator retVal = new SSIMCalculator();
		retVal.setL (_L .getValue(true));
		retVal.setK1(_K1.getValue(true));
		retVal.setK2(_K2.getValue(true));
		retVal.setGaussianWindow(_sigmaX.getValue(true), _sigmaY.getValue(true));
		return retVal;
	}
	
	/**
	 * Action performed when the user clicks on the details button
	 */
	private void onDetailsClicked()
	{
		String    title   = "SSIM informations";
		JTextPane message = new JTextPane();
		message.setEditable(false);
		message.setContentType("text/html");
		message.setText(
			"<p>" +
				"The SSIM is an index measuring the structural similarity between two images. " +
				"It is valued between -1 and 1. When two images are nearly identical, " +
				"their SSIM is close to 1." +
			"</p>" +
			"<p>" +
				"Formula computing the SSIM between two sequences seq1 and seq2 at a given pixel " +
				"or voxel <tt>P</tt>:" +
				"<pre>" +
				"              2*mu1(P)*mu2(P) + C1         2*cov(P) + C2     \n" +
				"  SSIM(P) = ------------------------ x ----------------------\n" +
				"            mu1(P)^2 + mu2(P)^2 + C1   s1(P)^2 + s2(P)^2 + C2\n" +
				"</pre>" +
				"With:" +
				"<ul>" +
					"<li><tt>mu1(P)</tt> and <tt>mu2(P)</tt>: mean value of seq1 and seq2 computed " +
					"over a small XY window located around <tt>P</tt></li>" +
					"<li><tt>s1(P)</tt> and <tt>s2(P)</tt>: standard deviation of seq1 and seq2 computed " +
					"over the same window</li>" +
					"<li><tt>cov(P)</tt>: covariance between seq1 and seq2 computed over the same window</li>" +
					"<li><tt>C1 = (K1*L)^2</tt>: regularization constant (should be as small as possible)</li>" +
					"<li><tt>C2 = (K2*L)^2</tt>: regularization constant (should be as small as possible)</li>" +
					"<li><tt>K1</tt>, <tt>K2</tt>: regularization parameters (must be >0)</li>" +
					"<li><tt>L</tt>: dynamic range of the pixel values (example: <tt>L=255</tt> " + 
					"if the sequence is 8 bit encoded)</li>" +
				"</ul>" +
			"</p>" +
			"<p>" +
				"The default window is a Gaussian window with standard deviation 1.5 along both " +
				"the X and the Y axis." +
			"</p>" +
			"<p>" +
				"Reference:" +
				"<quote>" +
					"Z. Wang, A. C. Bovik, H. R. Sheikh, E. P. Simoncelli (2004),<br/>" +
					"<em>Image quality assessment: from error visibility to structural similarity</em>,<br/>" +
					"IEEE Transactions on Image Processing, 13(4), 600-612." +
				"</quote>" +
			"</p>" +
			"<p>" +
				"This current implementation sticks as much as possible to the Matlab SSIM " +
				"implementation provided by these authors at:<br/>" +
				"<a href=\"https://ece.uwaterloo.ca/~z70wang/research/ssim/\">" +
					"https://ece.uwaterloo.ca/~z70wang/research/ssim/" +
				"</a>" +
			"</p>"
		);
		Dimension dim = message.getPreferredSize();
		dim.setSize(600, dim.getHeight()+100);
		message.setPreferredSize(dim);
		JScrollPane scroll = new JScrollPane(message);
		dim = scroll.getPreferredSize();
		dim.setSize(600, 500);
		scroll.setPreferredSize(dim);
		GenericFrame infoFrame = new GenericFrame(title, scroll);
		infoFrame.addToMainDesktopPane();
		infoFrame.setVisible(true);
		infoFrame.requestFocus();
	}
}
