package plugins.nchenouard.timeandzbinningviewer;

/**
 * @author Nicolas Chenouard
 * @date 4/8/2019
 * @email nicolas.chenouard.dev@gmail.com
 * */

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import java.awt.Component;

import javax.swing.Box;

import icy.canvas.Canvas2D;
import icy.canvas.IcyCanvas;
import icy.gui.frame.IcyFrame;
import icy.gui.viewer.Viewer;
import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginCanvas;
import icy.type.DataType;

public class TimeAndZBinningViewer extends Plugin implements PluginCanvas {

	public TimeAndZBinningViewer()
	{
		super();
	}
	
	@Override
	protected void finalize() throws Throwable
	{
		super.finalize();
	}

	@Override
	public String getCanvasClassName() {
		return "Image time and z binning";
	}

	@Override
	public IcyCanvas createCanvas(Viewer viewer) {
		Canvas2DSpinnerListener currentCanvas = new Canvas2DSpinnerListener(viewer);
		return currentCanvas;
	}

	class Canvas2DSpinnerListener extends Canvas2D implements ChangeListener
	{

		int frameAverageFactor = 1;
		JSpinner frameAvgSpinner;
		int sliceAverageFactor = 1;
		JSpinner sliceAvgSpinner;
		IcyFrame mainFrame = null;
		
		public Canvas2DSpinnerListener(Viewer viewer) {
			super(viewer);
						
			if (mainFrame == null)
			{
				mainFrame = new IcyFrame("Time and z binning", true, false, true, true);
				JPanel contentPane = new JPanel();			
				Box box = Box.createVerticalBox();	       
				
					JLabel label1 = new JLabel("Frame bin size");
				label1.setHorizontalAlignment((int) Component.LEFT_ALIGNMENT);
				box.add(label1);
				
				frameAvgSpinner = new JSpinner(
						new SpinnerNumberModel(frameAverageFactor, 1, Integer.MAX_VALUE, 1));
				frameAvgSpinner.setAlignmentX((int) Component.LEFT_ALIGNMENT);
				frameAvgSpinner.addChangeListener(this);
				box.add(frameAvgSpinner);

			
				JLabel label2 = new JLabel("Slice bin size");
				label2.setHorizontalAlignment((int) Component.LEFT_ALIGNMENT);
				box.add(label2);	       
				
				sliceAvgSpinner = new JSpinner(
						new SpinnerNumberModel(sliceAverageFactor, 1, Integer.MAX_VALUE, 1));
				sliceAvgSpinner.setAlignmentX((int) Component.LEFT_ALIGNMENT);
				sliceAvgSpinner.addChangeListener(this);
				box.add(sliceAvgSpinner);	       
				
				contentPane.add(box);				    
				mainFrame.setContentPane(contentPane);
				mainFrame.pack();
				addIcyFrame(mainFrame);
			}
			mainFrame.setVisible(true);
		}

		/**
		 * 
		 */
		private static final long serialVersionUID = -9095507794788387502L;

		@Override
		public IcyBufferedImage getImage(int t, int z, int c)
		{			
			IcyBufferedImage currentImage = super.getImage(t, z, c);
			if ((frameAverageFactor <= 1 && sliceAverageFactor <= 1) || (super.getSequence().getSizeZ() <= 1 && super.getSequence().getSizeT() <= 1))
			{
				return currentImage;				
			}
			else
			{
				// average through time only
				if (sliceAverageFactor == 1 || super.getSequence().getSizeZ() <= 1)
				{
					currentImage = IcyBufferedImageUtil.convertToType(currentImage, DataType.DOUBLE, false);
					double[] currentImageArray = currentImage.getDataXYAsDouble(0);

					int last_t = Math.min(super.getSequence().getSizeT() - 1, t + frameAverageFactor - 1); 				
					int numAverage = last_t - t + 1;
					final int zz = z;
					final int cc = c;			
					try
					{
						for (int j = 1; j < numAverage; j++)
						{

							double[] image = IcyBufferedImageUtil.convertToType(super.getImage(t + j, zz, cc), DataType.DOUBLE, false).getDataXYAsDouble(0);
							for (int x = 0; x < currentImageArray.length; x++)
							{
								currentImageArray[x] += image[x];                		
							}
						}                               
						for (int x = 0; x < currentImageArray.length; x++)
						{
							currentImageArray[x] /= numAverage;                		
						}            
						currentImage.updateChannelsBounds();
						return currentImage;
					}
					catch (Exception e)
					{
						return currentImage;
					}
				}
				// average through slices only
				if (frameAverageFactor == 1 || super.getSequence().getSizeT() <= 1)
				{
					currentImage = IcyBufferedImageUtil.convertToType(currentImage, DataType.DOUBLE, false);
					double[] currentImageArray = currentImage.getDataXYAsDouble(0);

					int last_z = Math.min(super.getSequence().getSizeZ() - 1, z + sliceAverageFactor - 1); 				
					int numAverage = last_z - z + 1;
					
					final int tt = t;
					final int cc = c;			
					try
					{
						for (int j = 1; j < numAverage; j++)
						{

							double[] image = IcyBufferedImageUtil.convertToType(super.getImage(tt, z + j, cc), DataType.DOUBLE, false).getDataXYAsDouble(0);
							for (int x = 0; x < currentImageArray.length; x++)
							{
								currentImageArray[x] += image[x];                		
							}
						}                               
						for (int x = 0; x < currentImageArray.length; x++)
						{
							currentImageArray[x] /= numAverage;                		
						}            
						currentImage.updateChannelsBounds();
						return currentImage;
					}
					catch (Exception e)
					{
						return currentImage;
					}					
				}
				// average through both slices and frames
				currentImage = IcyBufferedImageUtil.convertToType(currentImage, DataType.DOUBLE, false);
				double[] currentImageArray = currentImage.getDataXYAsDouble(0);

				int last_z = Math.min(super.getSequence().getSizeZ() - 1, z + sliceAverageFactor - 1); 				
				int numAverage_z = last_z - z + 1;			
				int last_t = Math.min(super.getSequence().getSizeT() - 1, t + frameAverageFactor - 1); 				
				int numAverage_t = last_t - t + 1;
				final int cc = c;			
				try
				{
					for (int j = 0; j < numAverage_t; j++)
					{
						for (int k = 0; k < numAverage_z; k++)
						{
							if (j >0 || k > 0)
							{
								double[] image = IcyBufferedImageUtil.convertToType(super.getImage(t + j, z + k, cc), DataType.DOUBLE, false).getDataXYAsDouble(0);
								for (int x = 0; x < currentImageArray.length; x++)
								{
									currentImageArray[x] += image[x];                		
								}
							}
						}
					}                               
					for (int x = 0; x < currentImageArray.length; x++)
					{
						currentImageArray[x] /= (numAverage_z*numAverage_t);                		
					}            
					currentImage.updateChannelsBounds();
					return currentImage;
				}
				catch (Exception e)
				{
					return currentImage;
				}			
			}
		}

		@Override
		public void stateChanged(ChangeEvent arg0) {
			frameAverageFactor = (Integer)frameAvgSpinner.getModel().getValue();
			sliceAverageFactor = (Integer)sliceAvgSpinner.getModel().getValue();			
			super.refresh();			
		}
		
		@Override
	    public void shutDown()
	    {
			frameAvgSpinner.removeChangeListener(this);
			sliceAvgSpinner.removeChangeListener(this);
			
			mainFrame.close();
			mainFrame = null;
						
	    }
	}
}

