package plugins.adufour.vars.lang;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import icy.main.Icy;
import icy.sequence.Sequence;
import icy.util.XMLUtil;
import plugins.adufour.vars.gui.VarEditor;
import plugins.adufour.vars.gui.VarEditorFactory;
import plugins.adufour.vars.util.VarListener;

public class VarSequence extends Var<Sequence>
{
    public static final String NO_SEQUENCE = "No Sequence";
    public static final String ACTIVE_SEQUENCE = "Active Sequence";

    private boolean activeSequenceSelected;

    /**
     * @param name
     * @param defaultValue
     */
    public VarSequence(String name, Sequence defaultValue)
    {
        this(name, defaultValue, null);
    }

    /**
     * @param name
     * @param defaultValue
     * @param defaultListener
     *        A listener to add to this variable immediately after creation
     */
    public VarSequence(String name, Sequence defaultValue, VarListener<Sequence> defaultListener)
    {
        super(name, Sequence.class, defaultValue, defaultListener);

        activeSequenceSelected = false;
    }

    /**
     * Saves the current variable to the specified node
     * 
     * @throws UnsupportedOperationException
     *         if the functionality is not supported by the current variable type
     */
    @Override
    public boolean saveToXML(Node node) throws UnsupportedOperationException
    {
        XMLUtil.setAttributeValue((Element) node, Var.XML_KEY_VALUE, getValueAsString());

        return true;
    }

    @Override
    public boolean loadFromXML(Node node)
    {
        setValueAsString(XMLUtil.getAttributeValue((Element) node, XML_KEY_VALUE, null));

        return true;
    }

    @Override
    public VarEditor<Sequence> createVarEditor()
    {
        return VarEditorFactory.getDefaultFactory().createSequenceChooser(this);
    }

    @Override
    public VarEditor<Sequence> createVarViewer()
    {
        return VarEditorFactory.getDefaultFactory().createSequenceViewer(this);
    }

    @Override
    public String getValueAsString()
    {
        Sequence s = getValue();

        if (s == null)
            return NO_SEQUENCE;
        if ((s == Icy.getMainInterface().getActiveSequence()) && activeSequenceSelected)
            return ACTIVE_SEQUENCE;

        if (s.getFilename() != null)
            return s.getFilename();

        return s.getName();
    }

    public boolean isActiveSequenceSelected()
    {
        return activeSequenceSelected;
    }

    public void setActiveSequenceSelected(boolean value)
    {
        activeSequenceSelected = value;
    }

    @Override
    public void setValueAsString(String newValue)
    {
        if ((newValue == null) || newValue.equals(NO_SEQUENCE))
        {
            setActiveSequenceSelected(false);
            setValue(null);
        }
        else if (newValue.equals(ACTIVE_SEQUENCE))
        {
            setActiveSequenceSelected(true);
            setValue(Icy.getMainInterface().getActiveSequence());
        }
        else
        {
            // not supported
            setActiveSequenceSelected(false);
            setValue(null);
        }
    }

    @Override
    public void setValue(Sequence newValue) throws IllegalAccessError
    {
        // not the active sequence ?
        if ((newValue == null) || (newValue != Icy.getMainInterface().getActiveSequence()))
            setActiveSequenceSelected(false);

        super.setValue(newValue);
    }

}
