package vtk;

import java.io.File;

/**
 * Enum used to load native library more easily. If you don't want to set the
 * specific environment variable you can provide the path of the directory that
 * contains the VTK library through a Java Property. Like in the following
 * command line:
 *
 * > java -cp vtk.jar -Dvtk.lib.dir=.../vtk-libs vtk.sample.SimpleVTK
 *
 * The directory .../vtk-libs must contain the so/dll/dylib + the jnilib files
 *
 * @author sebastien jourdain - sebastien.jourdain@kitware.com
 */
public enum vtkNativeLibrary {

    VTKACCELERATORSDAX("vtkAcceleratorsDaxJava", false),
    VTKACCELERATORSPISTON("vtkAcceleratorsPistonJava", false),
    VTKCHARTSCORE("vtkChartsCoreJava", true),
    VTKCOMMONCOLOR("vtkCommonColorJava", true),
    VTKCOMMONCOMPUTATIONALGEOMETRY("vtkCommonComputationalGeometryJava", true),
    VTKCOMMONCORE("vtkCommonCoreJava", true),
    VTKCOMMONDATAMODEL("vtkCommonDataModelJava", true),
    VTKCOMMONEXECUTIONMODEL("vtkCommonExecutionModelJava", true),
    VTKCOMMONMATH("vtkCommonMathJava", true),
    VTKCOMMONMISC("vtkCommonMiscJava", true),
    VTKCOMMONSYSTEM("vtkCommonSystemJava", true),
    VTKCOMMONTRANSFORMS("vtkCommonTransformsJava", true),
    VTKDOMAINSCHEMISTRY("vtkDomainsChemistryJava", true),
    VTKDOMAINSCHEMISTRYOPENGL2("vtkDomainsChemistryOpenGL2Java", true),
    VTKFILTERSAMR("vtkFiltersAMRJava", true),
    VTKFILTERSCORE("vtkFiltersCoreJava", true),
    VTKFILTERSEXTRACTION("vtkFiltersExtractionJava", true),
    VTKFILTERSFLOWPATHS("vtkFiltersFlowPathsJava", true),
    VTKFILTERSGENERAL("vtkFiltersGeneralJava", true),
    VTKFILTERSGENERIC("vtkFiltersGenericJava", true),
    VTKFILTERSGEOMETRY("vtkFiltersGeometryJava", true),
    VTKFILTERSHYBRID("vtkFiltersHybridJava", true),
    VTKFILTERSHYPERTREE("vtkFiltersHyperTreeJava", true),
    VTKFILTERSIMAGING("vtkFiltersImagingJava", true),
    VTKFILTERSMODELING("vtkFiltersModelingJava", true),
    VTKFILTERSPARALLEL("vtkFiltersParallelJava", true),
    VTKFILTERSPARALLELFLOWPATHS("vtkFiltersParallelFlowPathsJava", false),
    VTKFILTERSPARALLELGEOMETRY("vtkFiltersParallelGeometryJava", false),
    VTKFILTERSPARALLELIMAGING("vtkFiltersParallelImagingJava", true),
    VTKFILTERSPARALLELMPI("vtkFiltersParallelMPIJava", false),
    VTKFILTERSPARALLELSTATISTICS("vtkFiltersParallelStatisticsJava", false),
    VTKFILTERSPROGRAMMABLE("vtkFiltersProgrammableJava", true),
    VTKFILTERSREEBGRAPH("vtkFiltersReebGraphJava", false),
    VTKFILTERSSELECTION("vtkFiltersSelectionJava", true),
    VTKFILTERSSMP("vtkFiltersSMPJava", true),
    VTKFILTERSSOURCES("vtkFiltersSourcesJava", true),
    VTKFILTERSSTATISTICS("vtkFiltersStatisticsJava", true),
    VTKFILTERSSTATISTICSGNUR("vtkFiltersStatisticsGnuRJava", false),
    VTKFILTERSTEXTURE("vtkFiltersTextureJava", true),
    VTKFILTERSVERDICT("vtkFiltersVerdictJava", true),
    VTKGEOVISCORE("vtkGeovisCoreJava", true),
    VTKIMAGINGCOLOR("vtkImagingColorJava", true),
    VTKIMAGINGCORE("vtkImagingCoreJava", true),
    VTKIMAGINGFOURIER("vtkImagingFourierJava", true),
    VTKIMAGINGGENERAL("vtkImagingGeneralJava", true),
    VTKIMAGINGHYBRID("vtkImagingHybridJava", true),
    VTKIMAGINGMATH("vtkImagingMathJava", true),
    VTKIMAGINGMORPHOLOGICAL("vtkImagingMorphologicalJava", true),
    VTKIMAGINGSOURCES("vtkImagingSourcesJava", true),
    VTKIMAGINGSTATISTICS("vtkImagingStatisticsJava", true),
    VTKIMAGINGSTENCIL("vtkImagingStencilJava", true),
    VTKINFOVISBOOSTGRAPHALGORITHMS("vtkInfovisBoostGraphAlgorithmsJava", false),
    VTKINFOVISCORE("vtkInfovisCoreJava", true),
    VTKINFOVISLAYOUT("vtkInfovisLayoutJava", true),
    VTKINFOVISPARALLEL("vtkInfovisParallelJava", false),
    VTKINTERACTIONIMAGE("vtkInteractionImageJava", true),
    VTKINTERACTIONSTYLE("vtkInteractionStyleJava", true),
    VTKINTERACTIONWIDGETS("vtkInteractionWidgetsJava", true),
    VTKIOADIOS("vtkIOADIOSJava", false),
    VTKIOAMR("vtkIOAMRJava", true),
    VTKIOCORE("vtkIOCoreJava", true),
    VTKIOENSIGHT("vtkIOEnSightJava", true),
    VTKIOEXODUS("vtkIOExodusJava", true),
    VTKIOEXPORT("vtkIOExportJava", true),
    VTKIOFFMPEG("vtkIOFFMPEGJava", false),
    VTKIOGDAL("vtkIOGDALJava", false),
    VTKIOGEOJSON("vtkIOGeoJSONJava", false),
    VTKIOGEOMETRY("vtkIOGeometryJava", true),
    VTKIOIMAGE("vtkIOImageJava", true),
    VTKIOIMPORT("vtkIOImportJava", true),
    VTKIOINFOVIS("vtkIOInfovisJava", true),
    VTKIOLEGACY("vtkIOLegacyJava", true),
    VTKIOLSDYNA("vtkIOLSDynaJava", true),
    VTKIOMINC("vtkIOMINCJava", true),
    VTKIOMOVIE("vtkIOMovieJava", true),
    VTKIOMPIIMAGE("vtkIOMPIImageJava", false),
    VTKIOMPIPARALLEL("vtkIOMPIParallelJava", false),
    VTKIOMYSQL("vtkIOMySQLJava", false),
    VTKIONETCDF("vtkIONetCDFJava", true),
    VTKIOODBC("vtkIOODBCJava", false),
    VTKIOPARALLEL("vtkIOParallelJava", true),
    VTKIOPARALLELEXODUS("vtkIOParallelExodusJava", false),
    VTKIOPARALLELLSDYNA("vtkIOParallelLSDynaJava", false),
    VTKIOPARALLELNETCDF("vtkIOParallelNetCDFJava", false),
    VTKIOPARALLELXML("vtkIOParallelXMLJava", true),
    VTKIOPLY("vtkIOPLYJava", true),
    VTKIOPOSTGRESQL("vtkIOPostgreSQLJava", false),
    VTKIOSQL("vtkIOSQLJava", true),
    VTKIOVIDEO("vtkIOVideoJava", true),
    VTKIOVPIC("vtkIOVPICJava", false),
    VTKIOXDMF2("vtkIOXdmf2Java", false),
    VTKIOXDMF3("vtkIOXdmf3Java", false),
    VTKIOXML("vtkIOXMLJava", true),
    VTKIOXMLPARSER("vtkIOXMLParserJava", true),
    VTKPARALLELCORE("vtkParallelCoreJava", true),
    VTKPARALLELMPI("vtkParallelMPIJava", false),
    VTKRENDERINGANNOTATION("vtkRenderingAnnotationJava", true),
    VTKRENDERINGCONTEXT2D("vtkRenderingContext2DJava", true),
    VTKRENDERINGCONTEXTOPENGL("vtkRenderingContextOpenGLJava", false),
    VTKRENDERINGCONTEXTOPENGL2("vtkRenderingContextOpenGL2Java", true),
    VTKRENDERINGCORE("vtkRenderingCoreJava", true),
    VTKRENDERINGEXTERNAL("vtkRenderingExternalJava", false),
    VTKRENDERINGFREETYPE("vtkRenderingFreeTypeJava", true),
    VTKRENDERINGGL2PS("vtkRenderingGL2PSJava", false),
    VTKRENDERINGIMAGE("vtkRenderingImageJava", true),
    VTKRENDERINGLABEL("vtkRenderingLabelJava", true),
    VTKRENDERINGLIC("vtkRenderingLICJava", false),
    VTKRENDERINGLICOPENGL2("vtkRenderingLICOpenGL2Java", false),
    VTKRENDERINGLOD("vtkRenderingLODJava", true),
    VTKRENDERINGMATPLOTLIB("vtkRenderingMatplotlibJava", false),
    VTKRENDERINGOPENGL("vtkRenderingOpenGLJava", false),
    VTKRENDERINGOPENGL2("vtkRenderingOpenGL2Java", true),
    VTKRENDERINGPARALLEL("vtkRenderingParallelJava", false),
    VTKRENDERINGPARALLELLIC("vtkRenderingParallelLICJava", false),
    VTKRENDERINGQT("vtkRenderingQtJava", false),
    VTKRENDERINGVOLUME("vtkRenderingVolumeJava", true),
    VTKRENDERINGVOLUMEAMR("vtkRenderingVolumeAMRJava", false),
    VTKRENDERINGVOLUMEOPENGL("vtkRenderingVolumeOpenGLJava", false),
    VTKRENDERINGVOLUMEOPENGL2("vtkRenderingVolumeOpenGL2Java", true),
    VTKTESTINGRENDERING("vtkTestingRenderingJava", false),
    VTKPYTHONINTERPRETER("vtkPythonInterpreterJava", false),
    VTKVIEWSCONTEXT2D("vtkViewsContext2DJava", true),
    VTKVIEWSCORE("vtkViewsCoreJava", true),
    VTKVIEWSGEOVIS("vtkViewsGeovisJava", false),
    VTKVIEWSINFOVIS("vtkViewsInfovisJava", true),
    VTKWEBCORE("vtkWebCoreJava", false),
    VTKWEBGLEXPORTER("vtkWebGLExporterJava", false);


  /**
   * Try to load all library
   *
   * @return true if all library have been successfully loaded
   */
  public static boolean LoadAllNativeLibraries() {
    boolean isEveryThingLoaded = true;
    for (vtkNativeLibrary lib : values()) {
      try {
        if(lib.IsBuilt()) {
          lib.LoadLibrary();
        }
      } catch (UnsatisfiedLinkError e) {
        isEveryThingLoaded = false;
        e.printStackTrace();
      }
    }

    return isEveryThingLoaded;
  }

  /**
   * Load the set of given library and trows runtime exception if any given
   * library failed in the loading process
   *
   * @param nativeLibraries
   */
  public static void LoadNativeLibraries(vtkNativeLibrary... nativeLibraries) {
    for (vtkNativeLibrary lib : nativeLibraries) {
      lib.LoadLibrary();
    }
  }

  /**
   * Disable the pop-in vtkErrorWindow by writing the error to a log file.
   * If the provided logFile is null the default "vtkError.txt" file will be
   * used.
   *
   * @param logFile
   */
  public static void DisableOutputWindow(File logFile) {
    if(logFile == null) {
      logFile = new File("vtkError.txt");
    }
    vtkFileOutputWindow outputError = new vtkFileOutputWindow();
    outputError.SetFileName(logFile.getAbsolutePath());
    outputError.SetInstance(outputError);
  }

  private vtkNativeLibrary(String nativeLibraryName, boolean built) {
    this.nativeLibraryName = nativeLibraryName;
    this.loaded = false;
    this.built = built;
  }

  /**
   * Load the library and throws runtime exception if the library failed in
   * the loading process
   */
  public void LoadLibrary() throws UnsatisfiedLinkError {
    if (!loaded) {
      if (System.getProperty("vtk.lib.dir") != null) {
        File dir = new File(System.getProperty("vtk.lib.dir"));
        patchJavaLibraryPath(dir.getAbsolutePath());
        File libPath = new File(dir, System.mapLibraryName(nativeLibraryName));
        if (libPath.exists()) {
          try {
            Runtime.getRuntime().load(libPath.getAbsolutePath());
            loaded = true;
            return;
          } catch (UnsatisfiedLinkError e) {
            e.printStackTrace();
          }
        }
      }
      System.loadLibrary(nativeLibraryName);
    }
    loaded = true;
  }

  /**
   * @return true if the library has already been successfully loaded
   */
  public boolean IsLoaded() {
    return loaded;
  }

  /**
   * @return true if the module was enabled and therefore build
   */
  public boolean IsBuilt() {
    return built;
  }

  /**
   * @return the library name
   */
  public String GetLibraryName() {
    return nativeLibraryName;
  }

  private static void patchJavaLibraryPath(String vtkLibDir) {
    if (vtkLibDir != null) {
      String path_separator = System.getProperty("path.separator");
      String s = System.getProperty("java.library.path");
      if (!s.contains(vtkLibDir)) {
        s = s + path_separator + vtkLibDir;
        System.setProperty("java.library.path", s);
      }
    }
  }

  private String nativeLibraryName;
  private boolean loaded;
  private boolean built;
}
